package com.els.base.query.business.web.controller;

import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.query.QueryParamWapper;
import com.els.base.query.business.command.CreateCommand;
import com.els.base.query.business.command.SwitchCommand;
import com.els.base.query.business.command.UpdateCommand;
import com.els.base.query.business.entity.QueryBusiness;
import com.els.base.query.business.entity.QueryBusinessExample;
import com.els.base.query.business.service.QueryBusinessService;
import com.els.base.query.common.QueryConditionInvoker;
import com.els.base.query.condition.entity.QueryCondition;
import com.els.base.query.condition.entity.QueryConditionExample;
import com.els.base.query.condition.service.QueryConditionService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import javax.annotation.Resource;
import java.util.List;

@Api(value = "自定义查询条件业务")
@Controller
@RequestMapping("queryBusiness")
public class QueryBusinessController {
    @Resource
    protected QueryBusinessService queryBusinessService;
    @Resource
    protected QueryConditionService queryConditionService;
    @Resource
    protected QueryConditionInvoker queryConditionInvoker;

    @ApiOperation(httpMethod = "POST", value = "创建自定义查询条件业务")
    @RequestMapping("service/create")
    @ResponseBody
    public ResponseResult<String> create(@RequestBody QueryBusiness queryBusiness) {
        CreateCommand cmd = new CreateCommand(queryBusiness);
        queryConditionInvoker.invoke(cmd);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "编辑自定义查询条件业务")
    @RequestMapping("service/edit")
    @ResponseBody
    public ResponseResult<String> edit(@RequestBody QueryBusiness queryBusiness) {
        Assert.isNotBlank(queryBusiness.getId(), "id 为空，保存失败");
        UpdateCommand cmd = new UpdateCommand(queryBusiness);
        queryConditionInvoker.invoke(cmd);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "停用，启用自定义查询条件业务")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "conditionFlag", required = true, value = "开关", paramType = "query", dataType = "String", defaultValue = "0"),
    })
    @RequestMapping("service/switch")
    @ResponseBody
    public ResponseResult<String> stop(
            @RequestBody(required = true) List<String> ids,
            @RequestParam(required = true, defaultValue = "0") Integer conditionFlag) {
        Assert.isNotEmpty(ids, "id 为空，保存失败");
        SwitchCommand cmd = new SwitchCommand(ids, conditionFlag);
        queryConditionInvoker.invoke(cmd);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "根据ID查询详情")
    @RequestMapping("service/findByBusinessId")
    @ResponseBody
    public ResponseResult<QueryBusiness> findByBusinessId(@RequestParam String id) {
        Assert.isNotBlank(id, "id不能为空！");
        QueryBusiness queryBusiness = this.queryBusinessService.queryObjById(id);
        Assert.isNotNull(queryBusiness, "查询条件业务不存在，请检查！");
        List<QueryCondition> queryConditionList = queryConditionService.findByBusinessId(id);
        if (CollectionUtils.isNotEmpty(queryConditionList)) {
            queryBusiness.setQueryConditionList(queryConditionList);
        }
        return ResponseResult.success(queryBusiness);
    }

    @ApiOperation(httpMethod = "POST", value = "根据业务查询详情")
    @RequestMapping("service/findByBusinessCode")
    @ResponseBody
    public ResponseResult<QueryBusiness> findByBusinessCode(@RequestParam String businessCode) {
        Assert.isNotBlank(businessCode, "业务编码不能为空！");
        QueryBusiness queryBusiness = this.queryBusinessService.findByBusinessCode(businessCode);
        QueryConditionExample queryConditionExample = new QueryConditionExample();
        queryConditionExample.createCriteria()
                .andParentIdIsNull()
                .andQueryBusinessCodeEqualTo(businessCode)
                .andQueryBusinessIdEqualTo(queryBusiness.getId());
        List<QueryCondition> queryConditionList = queryConditionService.queryAllObjByExample(queryConditionExample);
        if (CollectionUtils.isNotEmpty(queryConditionList)) {
            queryBusiness.setQueryConditionList(queryConditionList);
            for (QueryCondition queryCondition : queryConditionList) {
                QueryConditionExample example = new QueryConditionExample();
                example.createCriteria().andParentIdEqualTo(queryCondition.getId());
                List<QueryCondition> list = queryConditionService.queryAllObjByExample(example);
                queryCondition.setQueryConditionList(list);
            }
        }
        return ResponseResult.success(queryBusiness);
    }


    @ApiOperation(httpMethod = "POST", value = "删除自定义查询条件业务")
    @RequestMapping("service/deleteByIds")
    @ResponseBody
    public ResponseResult<String> deleteByIds(@RequestBody(required = true) List<String> ids) {
        Assert.isNotEmpty(ids, "删除失败,id不能为空");
        QueryBusinessExample example = new QueryBusinessExample();
        example.createCriteria().andIdIn(ids);
        this.queryBusinessService.deleteByExample(example);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "查询自定义查询条件业务")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "pageNo", required = false, value = "所在页", paramType = "query", dataType = "String", defaultValue = "0"),
            @ApiImplicitParam(name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10"),
            @ApiImplicitParam(name = "wapper", required = false, value = "查询条件,属性名请参考 QueryBusiness", paramType = "body", dataType = "QueryParamWapper")
    })
    @RequestMapping("service/findByPage")
    @ResponseBody
    public ResponseResult<PageView<QueryBusiness>> findByPage(
            @RequestParam(defaultValue = "0") int pageNo,
            @RequestParam(defaultValue = "10") int pageSize,
            @RequestBody(required = false) QueryParamWapper wapper) {
        QueryBusinessExample example = new QueryBusinessExample();
        example.setPageView(new PageView<QueryBusiness>(pageNo, pageSize));

        QueryBusinessExample.Criteria criteria = example.createCriteria();
        //填写具体的查询条件，例如
        criteria.andIsEnableEqualTo(Constant.YES_INT);
        //criteria.andConditionFlagEqualTo(Constant.YES_INT); 未启用的也可以查询出来
        if (wapper != null) {
            CriteriaUtils.addExample(example, wapper);
        }

        PageView<QueryBusiness> pageData = this.queryBusinessService.queryObjByPage(example);
        return ResponseResult.success(pageData);
    }
}