package com.els.base.query.condition.web.controller;

import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.query.QueryParam;
import com.els.base.core.utils.query.QueryParamWapper;
import com.els.base.query.common.QueryConditionInvoker;
import com.els.base.query.condition.command.BatchCreateCommand;
import com.els.base.query.condition.command.CreateCommand;
import com.els.base.query.condition.entity.QueryCondition;
import com.els.base.query.condition.entity.QueryConditionExample;
import com.els.base.query.condition.service.QueryConditionService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import javax.annotation.Resource;
import java.util.List;

@Api(value = "自定义查询条件字段")
@Controller
@RequestMapping("queryCondition")
public class QueryConditionController {
    @Resource
    protected QueryConditionService queryConditionService;
    @Resource
    protected QueryConditionInvoker queryConditionInvoker;


    @ApiOperation(httpMethod = "POST", value = "创建自定义查询条件字段")
    @RequestMapping("service/batch/create")
    @ResponseBody
    public ResponseResult<String> batchCreate(
            @RequestParam(required = true) String queryBusinessCode,
            @RequestBody(required = true) List<QueryCondition> queryConditionList) {
        BatchCreateCommand cmd = new BatchCreateCommand(queryConditionList, queryBusinessCode);
        queryConditionInvoker.invoke(cmd);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "创建自定义查询条件字段")
    @RequestMapping("service/create")
    @ResponseBody
    public ResponseResult<String> create(@RequestBody QueryCondition queryCondition) {
        //this.queryConditionService.addObj(queryCondition);

        CreateCommand cmd = new CreateCommand(queryCondition);
        queryConditionInvoker.invoke(cmd);

        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "编辑自定义查询条件字段")
    @RequestMapping("service/edit")
    @ResponseBody
    public ResponseResult<String> edit(@RequestBody QueryCondition queryCondition) {
        Assert.isNotBlank(queryCondition.getId(), "id 为空，保存失败");
        this.queryConditionService.modifyObj(queryCondition);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "删除自定义查询条件字段")
    @RequestMapping("service/deleteByIds")
    @ResponseBody
    public ResponseResult<String> deleteByIds(@RequestBody(required = true) List<String> ids) {
        Assert.isNotEmpty(ids, "删除失败,id不能为空");
        QueryConditionExample example = new QueryConditionExample();
        example.createCriteria().andIdIn(ids);
        this.queryConditionService.deleteByExample(example);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "查询自定义查询条件字段")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "pageNo", required = false, value = "所在页", paramType = "query", dataType = "String", defaultValue = "0"),
            @ApiImplicitParam(name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10"),
            @ApiImplicitParam(name = "wapper", required = false, value = "查询条件,属性名请参考 QueryCondition", paramType = "body", dataType = "QueryParamWapper")
    })
    @RequestMapping("service/findByPage")
    @ResponseBody
    public ResponseResult<PageView<QueryCondition>> findByPage(
            @RequestParam(defaultValue = "0") int pageNo,
            @RequestParam(defaultValue = "10") int pageSize,
            @RequestBody(required = false) QueryParamWapper wapper) {
        QueryConditionExample example = new QueryConditionExample();
        example.setPageView(new PageView<QueryCondition>(pageNo, pageSize));

        QueryConditionExample.Criteria criteria = example.createCriteria();
        //填写具体的查询条件，例如
        //criteria.andIdEqualTo("1");
        if (wapper != null) {
            CriteriaUtils.addExample(example, wapper);
        }

        PageView<QueryCondition> pageData = this.queryConditionService.queryObjByPage(example);
        return ResponseResult.success(pageData);
    }

    @ApiOperation(httpMethod = "POST", value = "根据业务编码查询全部条件")
    @RequestMapping("service/findAll")
    @ResponseBody
    public ResponseResult<List<QueryCondition>> findAll(@RequestBody(required = false) QueryParamWapper wapper) {
        QueryConditionExample example = new QueryConditionExample();
        QueryConditionExample.Criteria criteria = example.createCriteria();
        if (wapper != null) {
            CriteriaUtils.addExample(example, wapper);
        }
        String queryBusinessCode = wapper.getQueryParams().stream()
                .filter(params -> "queryBusinessCode".equals(params.getProperty()))
                .map(QueryParam::getValue)
                .findAny().orElseGet(() -> null);
        Assert.isNotBlank(queryBusinessCode, "业务编码不能为空，请检查！");
        List<QueryCondition> list = this.queryConditionService.findByBusinessCode(queryBusinessCode);
        return ResponseResult.success(list);
    }
}