package com.els.base.schedule.web.controller;

import java.text.ParseException;

import javax.annotation.Resource;

import org.apache.commons.lang.StringUtils;
import org.quartz.SchedulerException;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Constant;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.query.QueryParamWapper;
import com.els.base.schedule.entity.ScheduleJob;
import com.els.base.schedule.entity.ScheduleJobExample;
import com.els.base.schedule.plugin.QuartzManager;
import com.els.base.schedule.service.ScheduleJobService;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

@Api(value="定时任务")
@Controller
@RequestMapping("scheduleJob")
public class ScheduleJobController {
    @Resource
    protected ScheduleJobService scheduleJobService;
    
    @Resource
    protected QuartzManager quartzManager;

    @ApiOperation(httpMethod="POST", value="创建定时任务")
    @RequestMapping("service/create")
    @ResponseBody
    public ResponseResult<String> create(@RequestBody ScheduleJob scheduleJob) {
    	
        this.scheduleJobService.addObj(scheduleJob);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod="POST", value="编辑定时任务")
    @RequestMapping("service/edit")
    @ResponseBody
    public ResponseResult<String> edit(@RequestBody ScheduleJob scheduleJob) throws SchedulerException {
        if (StringUtils.isBlank(scheduleJob.getId())) {
            throw new CommonException("id 为空，保存失败","id_is_blank");
        }
        
        ScheduleJob tmp = this.scheduleJobService.queryObjById(scheduleJob.getId());
        if (ScheduleJob.JOB_STATE_NORMAL.equals(tmp.getJobState())) {
        	throw new CommonException("job 已启动，无法修改","base_fail");
		}
        
        if (!this.quartzManager.isCronCorrect(scheduleJob.getCronExpression())) {
			throw new CommonException("cron 表达式错误","expression_error");
		}
        
        if(StringUtils.isBlank(scheduleJob.getCronExpression()) || StringUtils.isBlank(scheduleJob.getJobCode())){
        	throw new CommonException("cron表达式 与 jobcode不能为空","base_canot_be_null","cron表达式 与 jobcode");
        }
        
        if (!scheduleJob.getCronExpression().equals(tmp.getCronExpression()) || 
        		!scheduleJob.getJobCode().equals(tmp.getJobCode()) ||
        		!scheduleJob.getIsConcurrent().equals(tmp.getIsConcurrent())) {
			this.quartzManager.deleteJob(tmp);
		}
        
        this.scheduleJobService.modifyObj(scheduleJob);
        
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod="POST", value="删除定时任务")
    @RequestMapping("service/deleteById")
    @ResponseBody
    public ResponseResult<String> deleteById(String id) {
        this.scheduleJobService.deleteObjById(id);
        
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod="GET", value="查询定时任务")
    @RequestMapping("service/findByPage")
    @ResponseBody
    public ResponseResult<PageView<ScheduleJob>> findByPage(
    		@ApiParam(value="所在页",defaultValue="1" ) @RequestParam(defaultValue="1") int pageNo, 
    		@ApiParam(value="每页数量", defaultValue="10") @RequestParam(defaultValue="10") int pageSize, 
    		@ApiParam(value="查询条件") @RequestBody(required=false) QueryParamWapper queryParams) {

    	ScheduleJobExample example = new ScheduleJobExample();
        example.setPageView(new PageView<ScheduleJob>(pageNo, pageSize));
        
        ScheduleJobExample.Criteria criteria = example.createCriteria();
        if (queryParams != null) {
        	CriteriaUtils.addCriterion(criteria, queryParams);
        }
        return ResponseResult.success(this.scheduleJobService.queryObjByPage(example));
    }
    
    @ApiOperation(httpMethod="GET", value="现在执行任务")
    @RequestMapping("service/runJobNow")
	@ResponseBody
	public ResponseResult<String> runJobNow(String id) throws SchedulerException{
		ScheduleJob job = this.scheduleJobService.queryObjById(id);
		
		try {
			this.quartzManager.runJobNow(job);
		} catch (ClassNotFoundException e) {
			
			throw new CommonException("找不到该bean","base_fail");
		} catch (NoSuchMethodException e) {
			
			throw new CommonException("找不到该方法","not_found_fun");
		} 
		
		return ResponseResult.success();
	}
    
    @ApiOperation(httpMethod="GET", value="停止任务")
    @RequestMapping("service/pauseJob")
	@ResponseBody
	public ResponseResult<String> pauseJob(String id){
		
        ScheduleJob job = this.scheduleJobService.queryObjById(id);
		try {
			this.quartzManager.pauseJob(job);
			
			ScheduleJob tmp = new ScheduleJob();
			tmp.setId(id);
			tmp.setIsEnable(Constant.NO_INT);
			this.scheduleJobService.modifyObj(tmp);
			
		} catch (SchedulerException e) {
			throw new CommonException("job 停止任务失败","base_fail");
		}
		
		return ResponseResult.success();
	}
	
    @ApiOperation(httpMethod="GET", value="启动任务")
	@RequestMapping("service/resumeJob")
	@ResponseBody
	public ResponseResult<String> resumeJob(String id){
        
        ScheduleJob job = this.scheduleJobService.queryObjById(id);
		try {
			this.quartzManager.resumeJob(job);
			
			ScheduleJob tmp = new ScheduleJob();
			tmp.setId(id);
			tmp.setIsEnable(Constant.YES_INT);
			this.scheduleJobService.modifyObj(tmp);
			
		} catch (ClassNotFoundException | NoSuchMethodException | SchedulerException | ParseException e) {
			throw new CommonException("job 启动任务失败","base_fail");
		}
		
		return ResponseResult.success();
	}
}