package com.els.base.schedule.config;

import java.io.IOException;
import java.util.Properties;

import javax.annotation.Resource;
import javax.sql.DataSource;

import org.quartz.Scheduler;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.AutoConfigureOrder;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.scheduling.quartz.SchedulerFactoryBean;
import org.springframework.scheduling.quartz.SpringBeanJobFactory;

import com.els.base.schedule.plugin.QuartzManager;
import com.els.base.schedule.plugin.impl.QuartzManagerImpl;
import com.els.base.utils.SpringContextHolder;

@Configuration
//@ConditionalOnProperty(name = "job.is.enable", havingValue = "true", matchIfMissing = true)
@AutoConfigureOrder(Ordered.HIGHEST_PRECEDENCE)
public class ScheduleConfiguration {
	
	private static final String DEFAULT_EMPTY_CONFIG = "DEFAULT";
	
	@Resource
	protected ThreadPoolTaskExecutor defaultThreadPool;

    @Value("${org.quartz.scheduler.instanceName:DefaultScheduler}")
    protected String instanceName;

    @Value("${org.quartz.scheduler.instanceId:AUTO}")
    protected String instanceId;

    @Value("${org.quartz.jobStore.class:org.quartz.impl.jdbcjobstore.JobStoreTX}")
    protected String jobStoreClass;

    @Value("${org.quartz.jobStore.isClustered:true}")
    protected String isClustered;

    @Value("${org.quartz.jobStore.clusterCheckinInterval:15000}")
    protected String clusterCheckinInterval;

    @Value("${org.quartz.jobStore.maxMisfiresToHandleAtATime:1}")
    protected String maxMisfiresToHandleAtATime;

    @Value("${org.quartz.jobStore.misfireThreshold:12000}")
    protected String misfireThreshold;

    @Value("${org.quartz.jobStore.tablePrefix:QRTZ_}")
    protected String tablePrefix;

    @Value("${org.quartz.jobStore.selectWithLockSQL:SELECT * FROM {0}LOCKS WHERE SCHED_NAME = {1} AND LOCK_NAME = ? FOR UPDATE}")
    protected String selectWithLockSQL;
    
    @Value("${org.quartz.scheduler.instanceIdGenerator.class:com.els.base.schedule.utils.CommonInstanceIdGenerator}")
    protected String instanceIdGeneratorClass;
    
    @Value("${org.quartz.jobStore.driverDelegateClass:DEFAULT}")
    protected String driverDelegateClass;

    @Bean("stdSchedule")
    public SchedulerFactoryBean schedulerFactoryBean(DataSource dataSource, SpringContextHolder springContextHolder) throws IOException {

    	Properties prop = new Properties();
    	prop.put("org.quartz.scheduler.instanceId", instanceId);
        prop.put("org.quartz.scheduler.instanceName", instanceName);
        prop.put("org.quartz.scheduler.instanceIdGenerator.class", this.instanceIdGeneratorClass);
        //JobStore配置
        prop.put("org.quartz.jobStore.class", jobStoreClass);
        //集群配置
        prop.put("org.quartz.jobStore.isClustered", isClustered);
        prop.put("org.quartz.jobStore.clusterCheckinInterval", clusterCheckinInterval);
        prop.put("org.quartz.jobStore.maxMisfiresToHandleAtATime", maxMisfiresToHandleAtATime);

        prop.put("org.quartz.jobStore.misfireThreshold", misfireThreshold);
        prop.put("org.quartz.jobStore.tablePrefix", tablePrefix);
        prop.put("org.quartz.jobStore.selectWithLockSQL", selectWithLockSQL);
        
    	if (!DEFAULT_EMPTY_CONFIG.equals(driverDelegateClass)) {
			prop.put("org.quartz.jobStore.driverDelegateClass", driverDelegateClass);
		}
    	
        SchedulerFactoryBean factory = new SchedulerFactoryBean();
        factory.setQuartzProperties(prop);
        //可选,QuartzScheduler启动时更新己存在的Job,这样就不用每次修改targetObject后删除qrtz_job_details表对应记录
        factory.setOverwriteExistingJobs(true);
        factory.setAutoStartup(false); //设置自行启动
        factory.setDataSource(dataSource);
        factory.setTaskExecutor(defaultThreadPool);
        factory.setJobFactory(new SpringBeanJobFactory());
        return factory;
    }

	@Autowired
    @Bean("quartzManager")
    public QuartzManager getQuartzManager(Scheduler stdScheduler) {
        return new QuartzManagerImpl(stdScheduler);
    }

}
