
package com.els.base.schedule.plugin;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Date;
import java.util.UUID;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.commons.lang.builder.ToStringStyle;
import org.quartz.DisallowConcurrentExecution;
import org.quartz.Job;
import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.quartz.SchedulerException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import org.springframework.beans.BeanUtils;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Component;
import org.springframework.util.ReflectionUtils;

import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Constant;
import com.els.base.schedule.entity.ScheduleJob;
import com.els.base.schedule.entity.ScheduleJobLog;
import com.els.base.schedule.plugin.impl.QuartzManagerImpl;
import com.els.base.schedule.service.ScheduleJobLogService;
import com.els.base.schedule.utils.CommonInstanceIdGenerator;
import com.els.base.utils.SpringContextHolder;
import com.els.base.utils.uuid.UUIDGenerator;


/**
 * :@DisallowConcurrentExecution : 此标记用在实现Job的类上面,意思是不允许并发执行.
 * :注意org.quartz.threadPool.threadCount线程池中线程的数量至少要多个,否则@DisallowConcurrentExecution不生效
 * :假如Job的设置时间间隔为3秒,但Job执行时间是5秒,设置@DisallowConcurrentExecution以后程序会等任务执行完毕以后再去执行,否则会在3秒时再启用新的线程执行
 */

@DisallowConcurrentExecution
@Component
public class DynamicJob implements Job {
    private static final Logger logger = LoggerFactory.getLogger(DynamicJob.class);

    private static ScheduleJobLogService scheduleJobLogService;
    
    private static ThreadPoolTaskExecutor threadPoolTaskExecutor;

    /**
     * 核心方法,Quartz Job真正的执行逻辑.
     * @param executorContext executorContext JobExecutionContext中封装有Quartz运行所需要的所有信息
     * @throws JobExecutionException execute()方法只允许抛出JobExecutionException异常
     */

    @Override
    public void execute(JobExecutionContext executorContext) throws JobExecutionException {

    	String mdcId = UUID.randomUUID().toString().replaceAll("-", "");
    	MDC.put(Constant.REQUEST_UUID, mdcId);
    	
        //JobDetail中的JobDataMap是共用的,从getMergedJobDataMap获取的JobDataMap是全新的对象
        JobDataMap map = executorContext.getMergedJobDataMap();

        ScheduleJob scheduleJob = (ScheduleJob) map.get(QuartzManagerImpl.JOB_PARAM_KEY);

        long startTime = System.currentTimeMillis();
        logger.debug("EVENT=开始执行定时任务|job={}|class={}|KEY={}", scheduleJob.getJobCode(), scheduleJob.getJobClass(), mdcId);

        try {
            this.runJobNow(scheduleJob);
        } catch (ClassNotFoundException | NoSuchMethodException | IllegalAccessException | IllegalArgumentException
                | InvocationTargetException | SchedulerException e) {
            logger.error("定时任务执行异常", e);
            this.insertScheduleJobLog(scheduleJob, ScheduleJob.JOB_STATE_EXCEPTION, null);
            throw new CommonException("定时任务执行异常");

        }

        long times=  System.currentTimeMillis() - startTime;
        logger.debug("EVENT=结束执行命令|COMMAND={}|class={}|KEY={}|COST={} ms", scheduleJob.getJobCode(), scheduleJob.getJobClass(), mdcId, times);

        this.insertScheduleJobLog(scheduleJob, ScheduleJob.JOB_STATE_NORMAL, times);
        MDC.remove(Constant.REQUEST_UUID);
    }

    public void runJobNow(ScheduleJob scheduleJob) throws SchedulerException, ClassNotFoundException, NoSuchMethodException, IllegalAccessException, IllegalArgumentException, InvocationTargetException {

        logger.info("Running Job : {} ", ToStringBuilder.reflectionToString(scheduleJob, ToStringStyle.DEFAULT_STYLE));

        ClassLoader classLoader = this.getClass().getClassLoader();
        Class<?> clazz = classLoader.loadClass(scheduleJob.getJobClass());
        Object excutionBean = SpringContextHolder.getOneBean(clazz);

        if (excutionBean == null) {
            throw new NullPointerException(String.format("spring context 中找不到 class:%s", clazz.getName()));
        }

        String jobMethod = scheduleJob.getJobMethod();
        String jobMethodParams = scheduleJob.getJobMethodParams();

        if(StringUtils.isBlank(jobMethodParams)){
            Method method = clazz.getDeclaredMethod(jobMethod);
            ReflectionUtils.makeAccessible(method);
            method.invoke(excutionBean);
        }else{
            Method method = clazz.getDeclaredMethod(jobMethod,String.class);
            ReflectionUtils.makeAccessible(method);
            method.invoke(excutionBean, jobMethodParams);
        }
    }
    
    private void insertScheduleJobLog(ScheduleJob scheduleJob, Integer jobState, Long times){
    	ScheduleJobLog scheduleJobLog =new ScheduleJobLog();
        BeanUtils.copyProperties(scheduleJob,scheduleJobLog);
        scheduleJobLog.setId(UUIDGenerator.generateUUID());
        scheduleJobLog.setCreateTime(new Date());
        scheduleJobLog.setJobId(scheduleJob.getId());
        scheduleJobLog.setJobState(jobState);
        scheduleJobLog.setTimes(times);
        
        scheduleJobLog.setInstanceName(CommonInstanceIdGenerator.getDefaultInstanceId());
        getThreadPoolTaskExecutor().execute(() -> getScheduleJobLogService().addObj(scheduleJobLog));
    }
    
    private static ScheduleJobLogService getScheduleJobLogService(){
    	if (scheduleJobLogService == null) {
    		scheduleJobLogService = SpringContextHolder.getOneBean(ScheduleJobLogService.class);
		}
    	
    	return scheduleJobLogService;
    }
    
    private static ThreadPoolTaskExecutor getThreadPoolTaskExecutor(){
    	if (threadPoolTaskExecutor == null) {
			threadPoolTaskExecutor = SpringContextHolder.getOneBean(ThreadPoolTaskExecutor.class);
		}
    	
    	return threadPoolTaskExecutor;
    }

}

