package com.els.base.schedule.plugin.impl;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.quartz.CronExpression;
import org.quartz.CronScheduleBuilder;
import org.quartz.JobBuilder;
import org.quartz.JobDataMap;
import org.quartz.JobDetail;
import org.quartz.JobExecutionContext;
import org.quartz.JobKey;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.Trigger;
import org.quartz.TriggerBuilder;
import org.quartz.TriggerKey;
import org.quartz.impl.matchers.GroupMatcher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Constant;
import com.els.base.schedule.entity.ScheduleJob;
import com.els.base.schedule.entity.ScheduleJobExample;
import com.els.base.schedule.plugin.DynamicJob;
import com.els.base.schedule.plugin.QuartzManager;
import com.els.base.schedule.service.ScheduleJobService;

public class QuartzManagerImpl implements QuartzManager {

	private static final Logger logger = LoggerFactory.getLogger(QuartzManagerImpl.class);

	public QuartzManagerImpl(Scheduler stdScheduler){
		scheduler = stdScheduler;
	}

	@Autowired
	protected ScheduleJobService scheduleJobService;

	@Value("${job.is.enable:true}")
	protected String isJobEnalbe;

	private static Scheduler scheduler = null;

	/**
	 * 任务调度参数key
	 */
	public static String JOB_PARAM_KEY = "JOB_PARAM_KEY";

	/**
	 * 获取JobDataMap.(Job参数对象)
	 */
	@Override
	public JobDataMap getJobDataMap(ScheduleJob job) {
		JobDataMap map = new JobDataMap();
		map.put(JOB_PARAM_KEY,job);
		return map;
	}

	/**
	 * 获取JobDetail,JobDetail是任务的定义,而Job是任务的执行逻辑,JobDetail里会引用一个Job Class来定义
	 */
	@Override
	public JobDetail getJobDetail(ScheduleJob job) {
		return JobBuilder.newJob(DynamicJob.class)
				.withIdentity(getJobKey(job))
				.withDescription(job.getJobName())
				.setJobData(getJobDataMap(job))
				.storeDurably()
				.build();
	}

	/**
	 * 获取Trigger (Job的触发器,执行规则)
	 */
	@Override
	public Trigger getTrigger(ScheduleJob job) {
		if(StringUtils.isBlank(job.getJobGroup())){
			return TriggerBuilder.newTrigger()
					.withIdentity(job.getJobCode(), Scheduler.DEFAULT_GROUP)
					.withSchedule(CronScheduleBuilder.cronSchedule(job.getCronExpression()))
					.build();
		}else {
			return TriggerBuilder.newTrigger()
					.withIdentity(job.getJobCode(), job.getJobGroup())
					.withSchedule(CronScheduleBuilder.cronSchedule(job.getCronExpression()))
					.build();
		}

	}

	//获取JobKey,包含Name和Group
	@Override
	public JobKey getJobKey(ScheduleJob job) {
		if(StringUtils.isBlank(job.getJobGroup())){
			return JobKey.jobKey(job.getJobCode(), Scheduler.DEFAULT_GROUP);
		}else{
			return JobKey.jobKey(job.getJobCode(), job.getJobGroup());
		}

	}

	//获取TriggerKey,包含Name和Group
	@Override
	public TriggerKey getTriggerKey(ScheduleJob job) {
		if(StringUtils.isBlank(job.getJobGroup())){
			return  TriggerKey.triggerKey(job.getJobCode(), Scheduler.DEFAULT_GROUP);
		}else{
			return  TriggerKey.triggerKey(job.getJobCode(), job.getJobGroup());
		}

	}

	//从数据库中加载获取到所有Job
	public List<ScheduleJob> loadJobs() {
		ScheduleJobExample scheduleJobExample = new ScheduleJobExample();
		List<ScheduleJob> scheduleJobs = scheduleJobService.queryAllObjByExample(scheduleJobExample);
		return scheduleJobs;
	}

	@Override
	public JobDetail addJob(ScheduleJob job) throws SchedulerException {
		if (!this.checkSwitch()) {
			throw new CommonException("本应用，没有开启定时任务功能，请在 conf.properties文件中配置开启","base_fail");
		}
		
		JobDetail jobDetail = getJobDetail(job);
		scheduler.addJob(jobDetail, true);
		return jobDetail;
	}

	@Override
	public void runJobNow(ScheduleJob job) throws SchedulerException{
//		if (!this.checkSwitch()) {
//			throw new CommonException("本应用，没有开启定时任务功能，请在 conf.properties文件中配置开启","base_fail");
//		}
		JobKey jobKey = getJobKey(job);
		
		if (scheduler.checkExists(jobKey)) {
			scheduler.triggerJob(jobKey);
		}else{
			this.addJob(job);
			scheduler.triggerJob(jobKey);
		}
		
	}

	@Override
	public void pauseJob(ScheduleJob job) throws SchedulerException {
		if (!this.checkSwitch()) {
			throw new CommonException("本应用，没有开启定时任务功能，请在 conf.properties文件中配置开启","base_fail");
		}
		JobKey jobKey = getJobKey(job);
		if (scheduler.checkExists(jobKey)) {
			scheduler.pauseJob(jobKey);
		}

		this.scheduleJobService.modifyStatus(job.getId(), ScheduleJob.JOB_STATE_PAUSE, Constant.NO_INT);
	}

	@Override
	public void resumeJob(ScheduleJob job) throws SchedulerException{
		if (!this.checkSwitch()) {
			throw new CommonException("本应用，没有开启定时任务功能，请在 conf.properties文件中配置开启","base_fail");
		}
		
		// 1、检查job是否存在
		JobDetail jobDetail = null;
		
		if (this.checkExists(job)) {
			// 如果存在就取出来，检查是否有变更，有变更就要更新
			jobDetail = scheduler.getJobDetail(this.getJobKey(job));
			if (this.checkChange(jobDetail, job)) {
				this.deleteJob(job);
				jobDetail = this.addJob(job);
			}
			
		}else {
			// 如果job不存在就添加新的job
			jobDetail = this.addJob(job);
		}
		
		Trigger trigger = TriggerBuilder.newTrigger()
			.withIdentity(job.getJobCode(), Scheduler.DEFAULT_GROUP)
			.withSchedule(CronScheduleBuilder.cronSchedule(job.getCronExpression()))
			.forJob(jobDetail)
			.build();
		
		// 2、检查触发器是否存在
		if(scheduler.checkExists(trigger.getKey())){
			// 如果存在就更新
			scheduler.rescheduleJob(trigger.getKey(), trigger);
		}else{
			// 不存在就添加触发器
			scheduler.scheduleJob(trigger);
		}
		
		// 5、更新数据
		this.scheduleJobService.modifyStatus(job.getId(), ScheduleJob.JOB_STATE_NORMAL, Constant.YES_INT);
	}
	
	public boolean checkExists(ScheduleJob job) throws SchedulerException{
		return scheduler.checkExists(this.getJobKey(job));
	}
	
	@Override
	public void deleteJob(ScheduleJob job) throws SchedulerException {
		if (!this.checkSwitch()) {
			throw new CommonException("本应用，没有开启定时任务功能，请在 conf.properties文件中配置开启","base_fail");
		}
		JobKey jobKey = getJobKey(job);
		if(scheduler.checkExists(jobKey)){
			scheduler.deleteJob(jobKey);
		}
	}

	@Override
	public void updateJobCron(ScheduleJob job) throws SchedulerException {
		if (!this.checkSwitch()) {
			throw new CommonException("本应用，没有开启定时任务功能，请在 conf.properties文件中配置开启","base_fail");
		}
		scheduler.rescheduleJob(this.getTriggerKey(job), this.getTrigger(job));
	}

	@Override
	public List<ScheduleJob> getAllJob() throws SchedulerException {
		GroupMatcher<JobKey> matcher = GroupMatcher.anyJobGroup();
		Set<JobKey> jobKeys = scheduler.getJobKeys(matcher);
		
		List<ScheduleJob> allJobs = new ArrayList<>();
		for(JobKey jobKey : jobKeys){
			JobDetail jobDetail = scheduler.getJobDetail(jobKey);
			ScheduleJob job = (ScheduleJob) jobDetail.getJobDataMap().get(JOB_PARAM_KEY);
			allJobs.add(job);
		}
		return allJobs;
	}

	@Override
	public List<ScheduleJob> getRunningJob() throws SchedulerException {
		List<JobExecutionContext> executingJobs = scheduler.getCurrentlyExecutingJobs();
		
		List<ScheduleJob> runningJobs = new ArrayList<>();
		for (JobExecutionContext executingJob : executingJobs) {
			JobDetail jobDetail = executingJob.getJobDetail();
			ScheduleJob job = (ScheduleJob) jobDetail.getJobDataMap().get(JOB_PARAM_KEY);
			runningJobs.add(job);
		}
		return runningJobs;
	}

	@Override
	public boolean isCronCorrect(String cronStr) {
		return CronExpression.isValidExpression(cronStr);
	}

	@PostConstruct
	protected void init() throws SchedulerException{
		if (!checkSwitch()) {
			return;
		}
		
		List<ScheduleJob> scheduleJobs = loadJobs();
		
		if(CollectionUtils.isEmpty(scheduleJobs)){
			return;
		}
		
		for (ScheduleJob scheduleJob : scheduleJobs) {
			restartJob(scheduleJob);
		}
		
		scheduler.start();
		logger.info("INIT SCHEDULE JOBS SUCCESS");
	}
	
	@PreDestroy
	protected void destory() throws SchedulerException{
		scheduler.shutdown(false);
	}

	@Override
	public void restartJob(ScheduleJob job) {
		try {
			if (!Constant.YES_INT.equals(job.getIsEnable())) {
				// 如果是禁用，就删除
				this.deleteJob(job);
				return;
			}
			
			this.resumeJob(job);
			
			logger.info("Job register successed. jobCode : {} , jobName : {} , cron : {}", job.getJobCode(), job.getJobName(), job.getCronExpression());
		} catch (Exception e) {
			String message = MessageFormat.format("Job register failed. jobCode : {0} , jobName{1} , cron:{2}", job.getJobCode(), job.getJobName(), job.getCronExpression());
			logger.error(message, e);
			
			if (!job.getJobState().equals(ScheduleJob.JOB_STATE_EXCEPTION)) {
				//更新状态
				this.scheduleJobService.modifyStatus(job.getId(), ScheduleJob.JOB_STATE_EXCEPTION, null);
			}
		}
	}

	private boolean checkChange(JobDetail oldJobDetail, ScheduleJob job) throws SchedulerException {
		ScheduleJob oldJob = null;
		try {
			oldJob = (ScheduleJob) oldJobDetail.getJobDataMap().get(JOB_PARAM_KEY);
		} catch (Exception e) {
			logger.warn("get jobDetail error", e);
		}
		
		if (oldJob == null) {
			return true;
		}
		
		return oldJob.getJobClass().equals(job.getJobClass()) 
				&& oldJob.getJobMethod().equals(job.getJobMethod())
				&& StringUtils.equals(oldJob.getJobMethodParams(), job.getJobMethodParams());
	}

	/**
	 * 检查开关是否开启,开启返回true,没有开启返回false
	 * @return
	 */
	public boolean checkSwitch(){
		return "true".equals(this.isJobEnalbe);
	}

}
