package com.els.base.schedule.service.impl;

import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.quartz.SchedulerException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.els.base.core.entity.PageView;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.schedule.dao.ScheduleJobMapper;
import com.els.base.schedule.entity.ScheduleJob;
import com.els.base.schedule.entity.ScheduleJobExample;
import com.els.base.schedule.plugin.QuartzManager;
import com.els.base.schedule.service.ScheduleJobService;
import com.els.base.utils.SpringContextHolder;

@Service("defaultScheduleJobService")
public class ScheduleJobServiceImpl implements ScheduleJobService {
	
	private static final Logger logger = LoggerFactory.getLogger(ScheduleJobServiceImpl.class);
	
    @Resource
    protected ScheduleJobMapper scheduleJobMapper;
    
    private QuartzManager quartzManager;
    
	private QuartzManager getQuartzManager(){
		if (quartzManager == null) {
			quartzManager = SpringContextHolder.getOneBean(QuartzManager.class);
		}

		if (quartzManager == null) {
			throw new CommonException("系统未开启定时任务功能");
		}
		return quartzManager;
	}

    @CacheEvict(value={"scheduleJob"}, allEntries=true)
    @Override
    public void addObj(ScheduleJob scheduleJob) {
    	Assert.isNotNull(scheduleJob,"定时任务信息不能为空！");
		Assert.isNotBlank(scheduleJob.getJobClass(),"定时任务类不能为空，请检查！");
		Assert.isNotBlank(scheduleJob.getCronExpression(),"定时任务Cron表达式不能为空，请检查！");	
    	
		boolean beanExist = this.isBeanExist(scheduleJob.getJobClass());
		if(!beanExist){
			throw new CommonException("执行任务类不存在，请检查！");
		}
		
		if (!getQuartzManager().isCronCorrect(scheduleJob.getCronExpression())) {
			throw new CommonException("cron 表达式错误","expression_error");
		}
		
		try {
			getQuartzManager().addJob(scheduleJob);
			
		} catch (SchedulerException e) {
			logger.error("添加定时任务失败", e);
			throw new CommonException("添加定时任务失败");
		}
		
		if (this.checkExists(scheduleJob)) {
			throw new CommonException("定时器的编码【"+scheduleJob.getJobCode()+"】已经存在,请检查！");
		}
		this.scheduleJobMapper.insertSelective(scheduleJob);
		
    }

    @CacheEvict(value={"scheduleJob"}, allEntries=true)
    @Override
    public void deleteObjById(String id) {
    	Assert.isNotBlank(id,"ID不能为空！");
		ScheduleJob scheduleJob = this.queryObjById(id);
		Assert.isNotNull(scheduleJob, "定时任务不存在");
		
		if (ScheduleJob.JOB_STATE_NORMAL.equals(scheduleJob.getJobState())) {
			throw new CommonException("job 已启动，请停止后操作","base_fail");
		}
		
		try {
			getQuartzManager().deleteJob(scheduleJob);
			
		} catch (SchedulerException e) {
			logger.error("删除定时任务失败", e);
			throw new CommonException("删除定时任务失败");
		}
		this.scheduleJobMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value={"scheduleJob"}, allEntries=true)
    @Override
    public void modifyObj(ScheduleJob scheduleJob) {
        
        Assert.isNotBlank(scheduleJob.getId(), "id 为空，保存失败","id_is_blank");
		
		ScheduleJob tmp = this.queryObjById(scheduleJob.getId());
		Assert.isNotNull(tmp,"定时任务不存在，请检查！");
		
		if (ScheduleJob.JOB_STATE_NORMAL.equals(tmp.getJobState())) {
			throw new CommonException("job 已启动，请停止后修改","base_fail");
		}
		
		Assert.isNotBlank(scheduleJob.getCronExpression(), "cron表达式不能为空");
		Assert.isNotBlank(scheduleJob.getJobCode(), "jobcode不能为空");

		if (!getQuartzManager().isCronCorrect(scheduleJob.getCronExpression())) {
			throw new CommonException("cron 表达式错误","expression_error");
		}

		scheduleJob.setUpdateTime(new Date());
        this.scheduleJobMapper.updateByPrimaryKeySelective(scheduleJob);
    }

    @Cacheable(value="scheduleJob", keyGenerator="redisKeyGenerator")
    @Override
    public ScheduleJob queryObjById(String id) {
        return this.scheduleJobMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value="scheduleJob", keyGenerator="redisKeyGenerator")
    @Override
    public List<ScheduleJob> queryAllObjByExample(ScheduleJobExample example) {
        return this.scheduleJobMapper.selectByExample(example);
    }

    @Cacheable(value="scheduleJob", keyGenerator="redisKeyGenerator")
    @Override
    public PageView<ScheduleJob> queryObjByPage(ScheduleJobExample example) {
        PageView<ScheduleJob> pageView = example.getPageView();
        pageView.setQueryResult(this.scheduleJobMapper.selectByExampleByPage(example));
        return pageView;
    }
    
    @CacheEvict(value={"scheduleJob"}, allEntries=true)
   	@Override
   	public void deleteByExample(ScheduleJobExample example) {
       	Assert.isNotNull(example, "参数不能为空");
       	Assert.isNotEmpty(example.getOredCriteria(), "批量删除不能全表删除");
       	
       	this.scheduleJobMapper.deleteByExample(example);
   	}

    @CacheEvict(value={"scheduleJob"}, allEntries=true)
    @Transactional
   	@Override
   	public void addAll(List<ScheduleJob> list) {
   		if (CollectionUtils.isEmpty(list)) {
   			return;
   		}
   		
   		list.forEach(record ->{
   			this.addObj(record);
   		});
   	}

	@Override
	public boolean checkExists(ScheduleJob job) {
		ScheduleJobExample example = new ScheduleJobExample();
		ScheduleJobExample.Criteria cri = example.createCriteria().andJobCodeEqualTo(job.getJobCode());
		if (StringUtils.isNotBlank(job.getJobGroup())) {
			cri.andJobGroupEqualTo(job.getJobGroup());
		}
		
		return this.scheduleJobMapper.countByExample(example) > 0;
	}
	
	private boolean isBeanExist(String jobClass) {
		if(StringUtils.isBlank(jobClass)){
			return false;
		}
		ClassLoader classLoader = this.getClass().getClassLoader();
		Class<?> clazz = null;
		try {
			clazz = classLoader.loadClass(jobClass);
			Object bean = SpringContextHolder.getOneBean(clazz);
			if(bean==null){
				return false;
			}
		} catch (ClassNotFoundException e) {
			e.printStackTrace();
			throw  new CommonException("包名和类名是否正确，请检查！");
			//return false;
		}
		return true;
	}

	@CacheEvict(value={"scheduleJob"}, allEntries=true)
	@Override
	public void modifyStatus(String id, Integer jobStatus, Integer isEnable) {
		if(jobStatus == null && isEnable == null){
			throw new CommonException("状态为空无法变更状态");
		}
		
		ScheduleJob scheduleJob = new ScheduleJob();
		scheduleJob.setId(id);
		scheduleJob.setJobState(jobStatus);
		scheduleJob.setIsEnable(isEnable);
		
		this.scheduleJobMapper.updateByPrimaryKeySelective(scheduleJob);
	}

}