package com.els.base.session.config;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.redis.connection.RedisClusterConfiguration;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import org.springframework.data.redis.core.RedisOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.listener.RedisMessageListenerContainer;
import org.springframework.session.ExpiringSession;
import org.springframework.session.SessionRepository;
import org.springframework.session.data.redis.RedisOperationsSessionRepository;
import org.springframework.session.data.redis.config.annotation.web.http.RedisHttpSessionConfiguration;
import org.springframework.session.web.http.SessionEventHttpSessionListenerAdapter;
import org.springframework.session.web.http.SessionRepositoryFilter;

import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;

import redis.clients.jedis.JedisPoolConfig;

@ConditionalOnProperty(name="http.session.redis.enable", havingValue="true", matchIfMissing=true)
@Configuration
public class SessionRedisConfiguration extends RedisHttpSessionConfiguration {
	
	private static Logger logger = LoggerFactory.getLogger(SessionRedisConfiguration.class);
	
	private static final String DEFALUT_NO_PASSWORD = "_default_empty";
	
	@Value("${http.session.redis.pool.maxTotal:50}")
	protected String maxTotal;
	
	@Value("${http.session.redis.pool.maxIdle:10}")
	protected String maxIdle;
	
	@Value("${http.session.redis.pool.maxWaitMillis:300000}")
	protected String maxWaitMillis;
	
	@Value("${http.session.redis.pool.testOnBorrow:true}")
	protected String testOnBorrow;
	
	@Value("${http.session.redis.hostname}")
	protected String hostName;
	
	@Value("${http.session.redis.port}")
	protected String port;
	
	@Value("${http.session.redis.password:"+DEFALUT_NO_PASSWORD+"}")
	protected String password;
	
	@Value("${http.session.redis.database:0}")
	protected String database;
	
	@Value("${http.session.redis.cluster.is.enable:false}")
	protected String isCluster;
	
	@Value("${http.session.redis.cluster.nodes:false}")
	protected String nodes;
	
	@Value("${http.session.redis.timeout:1800}")
	@Override
	public void setMaxInactiveIntervalInSeconds(int maxInactiveIntervalInSeconds) {
		super.setMaxInactiveIntervalInSeconds(maxInactiveIntervalInSeconds);
	}
	
	@Bean("httpSessionRedisConfig")
	public JedisPoolConfig getJedisPoolConfig(){
		JedisPoolConfig jedisPoolConfig = new JedisPoolConfig();
		jedisPoolConfig.setMaxTotal(Integer.valueOf(maxTotal));
		jedisPoolConfig.setMaxIdle(Integer.valueOf(maxIdle));
		jedisPoolConfig.setMaxWaitMillis(Long.valueOf(maxWaitMillis));
		jedisPoolConfig.setTestOnBorrow(Boolean.valueOf(testOnBorrow));
		return jedisPoolConfig;
	}
	
	@Bean("sessionRedisConnectionFactory")
	public JedisConnectionFactory getJedisConnectionFactory(){
		
		if ("true".equals(this.isCluster.trim().toLowerCase())) {
			
			RedisClusterConfiguration clusterConfig = this.getClusterConfig();
			
			JedisConnectionFactory connectionFactory = new JedisConnectionFactory(clusterConfig);
			connectionFactory.setPoolConfig(getJedisPoolConfig());
			if (StringUtils.isNotBlank(password) && !DEFALUT_NO_PASSWORD.equals(password)) {
				connectionFactory.setPassword(password);
			}
			return connectionFactory;
			
		}else {
			Assert.isNotBlank(hostName, "配置 http.session.redis.hostname 不能为空");
			Assert.isNotBlank(port, "配置 http.session.redis.port 不能为空");
			Assert.isNotBlank(password, "配置 http.session.redis.password 不能为空");
			if (!StringUtils.isNumeric(database)) {
				throw new CommonException("配置 http.session.redis.database， 是一个整数");
			}
			
			logger.info("session redis config server[{}:{}] pass[{}]", hostName, port, this.centerPad(password));
			
			JedisConnectionFactory connectionFactory = new JedisConnectionFactory();
			connectionFactory.setPoolConfig(getJedisPoolConfig());
			connectionFactory.setHostName(hostName);
			connectionFactory.setPort(Integer.valueOf(port));
			
			if (StringUtils.isNotBlank(password) && !DEFALUT_NO_PASSWORD.equals(password)) {
				connectionFactory.setPassword(password);
			}
			
			connectionFactory.setDatabase(Integer.valueOf(database));
			return connectionFactory;
		}
		
	}

	private RedisClusterConfiguration getClusterConfig() {
		Assert.isNotBlank(nodes, "配置 http.session.redis.cluster.nodes 不能为空");
		String[] nodeArray = nodes.split(";");
		
		Set<String> nodeSet = new HashSet<>();
		for (String node : nodeArray) {
			String[] ipAndPort = node.split(":");
			if (ipAndPort.length != 2) {
				throw new CommonException(String.format("http.session.redis.cluster.nodes 集群中，ip与端口的配置错误: {%s}", node));
			}
			
			if (!StringUtils.isNumeric(ipAndPort[1])) {
				throw new CommonException(String.format("http.session.redis.cluster.nodes 集群中，ip与端口的配置错误: {%s}", node));
			}
			
			if (!ipAndPort[0].matches("[a-zA-Z0-9][-a-zA-Z0-9]{0,62}(.[a-zA-Z0-9][-a-zA-Z0-9]{0,62})+.?")) {
				throw new CommonException(String.format("集群中，ip与端口的配置错误: {%s}", node));
			}
			
			nodeSet.add(node);
		}
		
		RedisClusterConfiguration clusterConfiguration = new RedisClusterConfiguration(nodeSet);
		return clusterConfiguration;
	}

	@Bean("sessionRedisTemplate")
	public RedisTemplate<?, ?> getRedisTemplate() {
		RedisTemplate<String, Object> template = new RedisTemplate<>();
		template.setConnectionFactory(this.getJedisConnectionFactory());
		return template;
	}
	
	@Bean("sessionRedisListener")
	public SessionEventHttpSessionListenerAdapter getListener(){
		SessionEventHttpSessionListenerAdapter listener = new SessionEventHttpSessionListenerAdapter(new ArrayList<>());
		return listener;
	}
	
	@Bean("sessionRepository")
	public RedisOperationsSessionRepository getSessionRepository(){
		RedisOperationsSessionRepository repository = new RedisOperationsSessionRepository(getJedisConnectionFactory());
		return repository;
	}
	
	@Override
	public RedisTemplate<Object, Object> sessionRedisTemplate(
			@Qualifier(value="sessionRedisConnectionFactory")RedisConnectionFactory connectionFactory) {
		return super.sessionRedisTemplate(this.getJedisConnectionFactory());
	}

	@Override
	public RedisMessageListenerContainer redisMessageListenerContainer(
			@Qualifier(value="sessionRedisConnectionFactory") RedisConnectionFactory connectionFactory,
			@Qualifier(value="sessionRepository")  RedisOperationsSessionRepository messageListener) {
		return super.redisMessageListenerContainer(connectionFactory, messageListener);
	}

	@Override
	public InitializingBean enableRedisKeyspaceNotificationsInitializer(
			@Qualifier(value="sessionRedisConnectionFactory") RedisConnectionFactory connectionFactory) {
		return super.enableRedisKeyspaceNotificationsInitializer(connectionFactory);
	}
	
	@Override
	public <S extends ExpiringSession> SessionRepositoryFilter<? extends ExpiringSession> springSessionRepositoryFilter(
			@Qualifier(value="sessionRepository") SessionRepository<S> sessionRepository) {
		return super.springSessionRepositoryFilter(sessionRepository);
	}

	@Override
	public RedisOperationsSessionRepository sessionRepository(
			@Qualifier(value="sessionRedisTemplate") RedisOperations<Object, Object> sessionRedisTemplate,
			@Qualifier(value="sessionRedisListener") ApplicationEventPublisher applicationEventPublisher) {
		return super.sessionRepository(sessionRedisTemplate, applicationEventPublisher);
	}
	
	private String centerPad(String password){
		if (password.length() <= 2) {
			return password;
		}
		
		return password.substring(0, 1) + StringUtils.leftPad(password.substring(password.length() - 1), password.length() - 1 , "*");
	}
	
}
