package com.els.base.userprofile.service.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.UUID;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.els.base.core.entity.PageView;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.userprofile.dao.DefaultProfileMapper;
import com.els.base.userprofile.dao.ProfileOptionMapper;
import com.els.base.userprofile.entity.DefaultProfile;
import com.els.base.userprofile.entity.DefaultProfileExample;
import com.els.base.userprofile.entity.ProfileOption;
import com.els.base.userprofile.entity.ProfileOptionExample;
import com.els.base.userprofile.service.DefaultProfileService;
import com.els.base.userprofile.web.controller.vo.ProfileVO;
import com.els.base.utils.json.JsonUtils;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;

@Service("defaultDefaultProfileService")
public class DefaultProfileServiceImpl implements DefaultProfileService {
	
	private static Logger logger = LoggerFactory.getLogger(DefaultProfileServiceImpl.class);
	
	@Resource
	protected DefaultProfileMapper defaultProfileMapper;

	@Resource
	protected ProfileOptionMapper profileOptionMapper;

	@CacheEvict(value = { "defaultProfile" }, allEntries = true)
	@Override
	public void addObj(DefaultProfile t) {
		this.defaultProfileMapper.insertSelective(t);
	}

	@CacheEvict(value = { "defaultProfile" }, allEntries = true)
	@Override
	public void deleteObjById(String id) {
		this.defaultProfileMapper.deleteByPrimaryKey(id);
	}

	@CacheEvict(value = { "defaultProfile" }, allEntries = true)
	@Override
	public void modifyObj(DefaultProfile t) {
		if (StringUtils.isBlank(t.getId())) {
			throw new NullPointerException("id 为空，无法更新");
		}
		this.defaultProfileMapper.updateByPrimaryKeySelective(t);
	}

	@Cacheable(value = "defaultProfile", keyGenerator = "redisKeyGenerator")
	@Override
	public DefaultProfile queryObjById(String id) {
		return this.defaultProfileMapper.selectByPrimaryKey(id);
	}

	@Cacheable(value = "defaultProfile", key = "'DefaultProfileService_' + #root.methodName + '_'+ T(com.els.base.utils.encryption.Md5Utils).md5Object(#example)")
	@Override
	public List<DefaultProfile> queryAllObjByExample(DefaultProfileExample example) {
		return this.defaultProfileMapper.selectByExample(example);
	}

	@Cacheable(value = "defaultProfile", key = "'DefaultProfileService_' + #root.methodName + '_'+ T(com.els.base.utils.encryption.Md5Utils).md5Object(#example)")
	@Override
	public PageView<DefaultProfile> queryObjByPage(DefaultProfileExample example) {
		PageView<DefaultProfile> pageView = example.getPageView();
		pageView.setQueryResult(this.defaultProfileMapper.selectByExampleByPage(example));
		return pageView;
	}

	/**
	 * 新增一个用户自定义设置项
	 * @author  zhoulongpeng
	 * @param   profileVO
	 */
	@CacheEvict(value = { "defaultProfile" }, allEntries = true)
	@Override
	public void insertObj(ProfileVO profileVO) {

		DefaultProfile defaultProfile = this.commonModify(profileVO);

		this.defaultProfileMapper.insertSelective(defaultProfile);

		String profileId = defaultProfile.getId();
		Map<String, String> options = profileVO.getOptions();

		for (Map.Entry<String, String> entry : options.entrySet()) {
			ProfileOption profileOption = new ProfileOption();
			profileOption.setId(UUID.randomUUID().toString().replaceAll("-", ""));
			profileOption.setProfileId(profileId);
			profileOption.setName(entry.getKey());
			profileOption.setVal(entry.getValue());
			profileOptionMapper.insert(profileOption);
		}
	}
	
	private DefaultProfile commonModify(ProfileVO profileVO) {
		if (profileVO == null) {
			throw new CommonException("参数不正确");
		}
		
		if (org.springframework.util.StringUtils.isEmpty(profileVO.getCode())) {
			throw new CommonException("设置项CODE值不能为空");
		}

		if (org.springframework.util.StringUtils.isEmpty(profileVO.getName())) {
			throw new CommonException("设置项名称不能为空");
		}

		if (profileVO.getDefaultValue() == null) {
			throw new CommonException("默认值不能为空");
		}

		if (profileVO.getOptions() == null) {
			throw new CommonException("设置项不能为空");
		}

		DefaultProfile defaultProfile = new DefaultProfile();

		defaultProfile.setName(profileVO.getName());
		defaultProfile.setProfileType(profileVO.getProfileType());

		String defaultValue;
		try {
			defaultValue = JsonUtils.writeValueAsString(profileVO.getDefaultValue());
		} catch (JsonProcessingException e) {
			e.printStackTrace();
			logger.error("{}", e);
			throw new CommonException("默认值转换为JSON出错");
		}

		defaultProfile.setDefaultValue(defaultValue);
		
		return defaultProfile;
	}
	
	@Override
	public void updateObj(ProfileVO profileVO) {
		if (org.springframework.util.StringUtils.isEmpty(profileVO.getId())) {
			throw new CommonException("设置项ID值不能为空");
		}
		
		DefaultProfile defaultProfile = this.commonModify(profileVO);
		
		DefaultProfileExample example = new DefaultProfileExample();
		DefaultProfileExample.Criteria criteria = example.createCriteria();
		criteria.andIdEqualTo(profileVO.getId());
		
		this.defaultProfileMapper.updateByExample(defaultProfile, example);

		String profileId = defaultProfile.getId();
		
		ProfileOptionExample optionExample = new ProfileOptionExample();
		ProfileOptionExample.Criteria optionCriteria = optionExample.createCriteria();
		optionCriteria.andProfileIdEqualTo(profileId);
		profileOptionMapper.deleteByExample(optionExample);
		
		Map<String, String> options = profileVO.getOptions();

		for (Map.Entry<String, String> entry : options.entrySet()) {
			ProfileOption profileOption = new ProfileOption();
			profileOption.setId(UUID.randomUUID().toString().replaceAll("-", ""));
			profileOption.setProfileId(profileId);
			profileOption.setName(entry.getKey());
			profileOption.setVal(entry.getValue());
			profileOptionMapper.insert(profileOption);
		}
	}

	/**
	 * 查询用户自定义设置项列表
	 * @author  zhoulongpeng
	 * @return  List<ProfileVO>
	 */
	@Override
	public List<ProfileVO> listAll(String code) {
		List<ProfileVO> list = new ArrayList<>();
		List<DefaultProfile> profiles;
		if (code == null) {
			profiles = defaultProfileMapper.selectByExample(null);
		} else {
			DefaultProfileExample example = new DefaultProfileExample();
			DefaultProfileExample.Criteria criteria = example.createCriteria();
			criteria.andCodeEqualTo(code);
			profiles = defaultProfileMapper.selectByExample(example);
		}
		if (profiles == null || profiles.isEmpty()) {
			return list;
		} 
		for(DefaultProfile profile : profiles) {
			ProfileVO profileVO = new ProfileVO();
			
			String profileId = profile.getId();
			
			profileVO.setId(profileId);
			profileVO.setName(profile.getName());
			profileVO.setProfileType(profile.getProfileType());
			profileVO.setCode(profile.getCode());
			profileVO.setSort(profile.getSort());
			
			String defaultValue = profile.getDefaultValue();
			
			ObjectMapper mapper = new ObjectMapper();  
			TypeReference<Map<String, Set<String>>> typeRef = new TypeReference<Map<String, Set<String>>>() {};
			Map<String, Set<String>> defaultValueMap = new HashMap<>();
			try {
				defaultValueMap = mapper.readValue(defaultValue, typeRef);
			} catch (JsonParseException e) {
				e.printStackTrace();
				logger.error("{}", e);
			} catch (JsonMappingException e) {
				e.printStackTrace();
				logger.error("{}", e);
			} catch (IOException e) {
				e.printStackTrace();
				logger.error("{}", e);
			}
			profileVO.setDefaultValue(defaultValueMap);
			
			ProfileOptionExample example = new ProfileOptionExample();
			ProfileOptionExample.Criteria criteria = example.createCriteria();
			criteria.andProfileIdEqualTo(profileId);
			List<ProfileOption> profileOptions = profileOptionMapper.selectByExample(example);
			if (profileOptions != null) {
				Map<String, String> options = new TreeMap<>();
				for(ProfileOption profileOption : profileOptions) {
					options.put(profileOption.getName(), profileOption.getVal());
				}
				profileVO.setOptions(options);
			}
			list.add(profileVO);
		}
		return list;
	}

	@CacheEvict(value = { "defaultProfile" }, allEntries = true)
	@Override
	public void deleteByExample(DefaultProfileExample example) {
		Assert.isNotNull(example, "参数不能为空");
       	Assert.isNotEmpty(example.getOredCriteria(), "批量删除不能全表删除");
       	
       	this.defaultProfileMapper.deleteByExample(example);
	}

	@CacheEvict(value = { "defaultProfile" }, allEntries = true)
	@Transactional
	@Override
	public void addAll(List<DefaultProfile> list) {
		if (CollectionUtils.isEmpty(list)) {
   			return;
   		}
   		
   		list.forEach(record ->{
   			this.defaultProfileMapper.insertSelective(record);		
   		});
	}

}