package com.els.base.utils.txtImport;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.lang.reflect.FieldUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.multipart.MultipartFile;

import com.els.base.utils.reflect.ReflectUtils;

/**
 * TXT文本导入工具类
 * @author linyuqin
 */
public class TxtImportUtils {

    private static Logger logger = LoggerFactory.getLogger(TxtImportUtils.class);

    /**
     * 导入TXT文件
     * @param file TXT文件
     * @param strTokenizer 分隔符
     * @param clazz 实体的class
     * @param keys 字段名称
     */
    @SuppressWarnings({ "rawtypes", "unchecked" })
	public static <T> TxtImportResponse<T> importTxt(MultipartFile file, String strTokenizer, Class<T> clazz, String[] keys) {
        logger.info("字符分隔符：{}", strTokenizer);
        TxtImportResponse response;
        // 判断文件类型是否为TXT
        if (checkFile(file)) {
            response = CreateResponse.createResponse(CreateResponse.NO_TXT);
            response.setMessage("文件格式不是TXT，导入失败！");
            logger.error("TXT文件导入失败：{}", response.getMessage());
            return response;
        }
        try (InputStream inputStream = file.getInputStream(); BufferedReader bf = new BufferedReader(new InputStreamReader(inputStream, "gbk"))) {
            String line;
            List<T> list = new ArrayList<>(0);
            // 标记去掉TXT文件的第一行
            boolean flag = true;
            while ((line = bf.readLine()) != null) {
                logger.info("每行的数据：{}", line);
                if (line.equals("")) {
                    continue;
                }
                if (flag) {
                    flag = false;
                    logger.info("去掉第一行数据");
                    continue;
                }
                String[] data = dealString(line, strTokenizer).split(strTokenizer);
                stringReplace(data);
                logger.info("处理后的数据：{}", Arrays.toString(data));
                // 判断字段的数量是否与值的数量相同
                if (data.length != keys.length) {
                    // 数量不同
                    response = CreateResponse.createResponse(CreateResponse.ERROR);
                    response.setMessage("导入失败,字段数量必须与文件每行的值数量相同！");
                    logger.error("TXT文件导入失败：{}", response.getMessage());
                    return response;
                }
                // 获取class对象的字段
                Field[] fields = clazz.getDeclaredFields();
                T object = clazz.newInstance();
                for (int i = 0; i < keys.length; i++) {
                    second: for (int j = 0; j < fields.length; j++) {
                        Field f = fields[j];
                        f.setAccessible(true);
                        // 判断实体字段是否与keys字段名称相同
                        if (keys[i].equals(f.getName())) {
                            // 获取字段的类型
//                            Class<?> type = metaObject.getGetterType(f.getName());
//                            Class<?> type = clazz.getDeclaredField(f.getName()).getNametType();
                            Class<?> type = FieldUtils.getField(object.getClass(), f.getName(), true).getType();
                            // 如果是Sting类型，直接set
                            if (type.equals(String.class)) {
                                ReflectUtils.setValue(object, f.getName(), data[i]);
                                break second;
                            }
                            // 其他类型，获取转换器
                            StringToObjectConverter converter = ConverterFactory.getDefaultToObjConverter(type);
                            if (converter == null) {
                                throw new RuntimeException("找不到合适转换器 class[" + type + "]");
                            }
                            ReflectUtils.setValue(object, f.getName(), converter.convert(data[i]));
                            break second;
                        }
                    }
                }
                list.add(object);
            }
            response = CreateResponse.createResponse(CreateResponse.SUCCESS);
            response.setList(list);
        } catch (Exception e) {
            e.printStackTrace();
            response = CreateResponse.createResponse(CreateResponse.ERROR);
            response.setMessage("TXT导入出错！");
            logger.error("TXT文件导入异常：{}", e.getMessage());
            return response;
        }
        return response;
    }

    /**
     * 数组中的空值替换为null
     * @param data
     */
    private static void stringReplace(String[] data) {
        for (int i = 0; i < data.length; i++) {
            if ("".equals(data[i])) {
                data[i] = null;
            }
        }
    }

    /**
     * 如果字符串最后一个字符是分隔符，需要在最后添加一个字符
     * @param data 文件每行数据
     * @param strTokenizer 分隔符
     */
    private static String dealString(String data, String strTokenizer) {
        // 字符串最后一个字符
        char lastChar = data.charAt(data.length() - 1);
        // 判断是否与分隔符相同
       return strTokenizer.charAt(strTokenizer.length() - 1) == lastChar ? data + null : data;
    }

    /**
     * 校验文件是否是TXT
     * @param file 文件
     */
    private static boolean checkFile(MultipartFile file) {
        if (file == null) {
            throw new RuntimeException("文件为空");
        }
        String type = file.getOriginalFilename().substring(file.getOriginalFilename().lastIndexOf("."));
        logger.info("文件名：{}", file.getOriginalFilename());
        logger.info("文件后缀：{}", type);
        return !".txt".equalsIgnoreCase(type);
    }

}
