package com.els.base.utils.math;

import java.math.BigDecimal;
import java.math.RoundingMode;

import org.springframework.util.Assert;

/**
 * @see RoundingMode
 * @see BigDecimal
 */
public class BigDecimalUtils {

	/**
	 * 默认精度为小数点后6
	 */
	public static final int DEFALUT_SCALE = 6;

	/**
	 * 提供精确加法计算的add方法
	 * 
	 * @param value1 被加数
	 * @param value2 加数
	 * @return 两个参数的和
	 */
	public static double add(double value1, double value2) {
		BigDecimal b1 = new BigDecimal(value1);
		BigDecimal b2 = new BigDecimal(value2);
		BigDecimal result = b1.add(b2).setScale(DEFALUT_SCALE, RoundingMode.HALF_UP);
		return result.doubleValue();
	}
	
	public static double add(BigDecimal b1, BigDecimal b2) {
		Assert.isNull(b1, "参数不能为空");
		Assert.isNull(b2, "参数不能为空");
		
		BigDecimal result = b1.add(b2).setScale(DEFALUT_SCALE, RoundingMode.HALF_UP);
		return result.doubleValue();
	}
	
	/**
	 * 提供精确减法运算的sub方法
	 * 
	 * @param value1 被减数
	 * @param value2 减数
	 * @return 两个参数的差
	 */
	public static double subtract(double value1, double value2) {
		BigDecimal b1 = new BigDecimal(value1);
		BigDecimal b2 = new BigDecimal(value2);
		BigDecimal result = b1.subtract(b2).setScale(DEFALUT_SCALE, RoundingMode.HALF_UP);
		return result.doubleValue();
	}

	/**
	 * 提供精确乘法运算的mul方法
	 * 
	 * @param value1 被乘数
	 * @param value2 乘数
	 * @return 两个参数的积
	 */
	public static double multiply(double value1, double value2) {
		BigDecimal b1 = new BigDecimal(value1);
		BigDecimal b2 = new BigDecimal(value2);
		BigDecimal result = b1.multiply(b2).setScale(DEFALUT_SCALE, RoundingMode.HALF_UP);
		return result.doubleValue();
	}

	/**
	 * 提供精确的除法运算方法div
	 * 
	 * @param value1 被除数
	 * @param value2 除数
	 * @param scale  小数点精确范围
	 * @return 两个参数的商
	 * @throws IllegalAccessException
	 */
	public static double divide(double value1, double value2, int scale) throws IllegalAccessException {
		// 如果精确范围小于0，抛出异常信息
		if (scale < 0) {
			throw new IllegalAccessException("精确度不能小于0");
		}
		if (value2 == 0) {
			throw new IllegalAccessException("除数不能等于0");
		}

		BigDecimal b1 = new BigDecimal(value1);
		BigDecimal b2 = new BigDecimal(value2);
		BigDecimal result = b1.divide(b2, scale, RoundingMode.HALF_UP);
		return result.doubleValue();
	}

	/**
	 * 求商数以及余数
	 * 
	 * @param value1 被除数
	 * @param value2 除数
	 * @return
	 * @throws IllegalAccessException
	 */
	public static BigDecimal[] divideAndRemainder(double value1, double value2) throws IllegalAccessException {
		// 如果精确范围小于0，抛出异常信息
		if (value2 == 0) {
			throw new IllegalAccessException("除数不能等于0");
		}
		BigDecimal b1 = new BigDecimal(value1);
		BigDecimal b2 = new BigDecimal(value2);
		return b1.divideAndRemainder(b2);
	}

	/*
	 * public static void main(String[] args) throws IllegalAccessException {
	 * 
	 * //存在的情况 System.out.print("0.06+0.01="); System.out.println(0.06+0.01);
	 * 
	 * System.out.print("1.0-0.42="); System.out.println(1.0-0.42);
	 * 
	 * System.out.print("4.015*100="); System.out.println(4.015*100);
	 * 
	 * System.out.print("303.1/1000="); System.out.println(303.1/1000);
	 * 
	 * //解决问题 System.out.println("解决问题："+BigDecimalUtils.add(0.06,0.01));
	 * System.out.println("解决问题："+BigDecimalUtils.add(0.1,0.01));
	 * 
	 * System.out.println("解决问题："+BigDecimalUtils.subtract(1.0,0.42));
	 * System.out.println("解决问题："+BigDecimalUtils.multiply(4.015,100));
	 * System.out.println("解决问题："+BigDecimalUtils.divide(303.1,1000.0,
	 * DEFALUT_SCALE));
	 * 
	 * System.out.println("解决问题："+BigDecimalUtils.divideAndRemainder(10,10)[0]);
	 * 
	 * }
	 */

}