package com.els.base.utils.pdf;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;

import com.els.base.utils.SpringContextHolder;
import com.els.base.utils.file.BaseFileUtils;
import com.els.base.utils.uuid.UUIDGenerator;

/**
 * html转pdf工具
 * @author hzy
 *
 */
public abstract class HtmlToPdfUtils {
	private static Logger logger = LoggerFactory.getLogger(HtmlToPdfUtils.class);
	
	public static final String PAGE_SIZE_A4 = "A4";
	public static final String PAGE_SIZE_A5 = "A5";
	public static final String PAGE_SIZE_LETTER = "LETTER";
	public static final String PAGE_SIZE_LETTER_A5 = "LETTER_A5";
	
	public static String WXHTMLTOPDF_BIN_PATH = "/usr/local/bin/wkhtmltopdf";
	
	static{
		try {
			String binPath = SpringContextHolder.getProperty("wkhtmltopdf.bin.path");
			if (StringUtils.isNotBlank(binPath)  ) {
				
				File binFile = new File(binPath);
				if (binFile.exists()) {
					WXHTMLTOPDF_BIN_PATH = binPath;
				}else{
					logger.warn("找不到配置文件中wkhtml的可执行文件，配置的路径：" + binPath);
				}
			}
		} catch (Exception e) {
			logger.error("设置wkhtml的可执行文件路径失败", e);
		}
	}
	
	/**
	 * 
	 * @param srcHtml 模版生成html
	 * @param destPdf	pdf文件
	 * @throws IOException
	 * @throws InterruptedException
	 */
	public static void generatePdf(String srcHtml, File destPdf) throws IOException, InterruptedException{
		generatePdf(srcHtml, destPdf, PAGE_SIZE_A4);
	}
	
	/**
	 * 
	 * @param srcHtml	模版生成html
	 * @param destPdf	pdf文件
	 * @param pageSize	页面大小(A4或A5)
	 * @throws IOException
	 * @throws InterruptedException
	 */
	public static void generatePdf(String srcHtml, File destPdf, String pageSize) throws IOException, InterruptedException{
		generatePdf(srcHtml, destPdf, pageSize, true);
	}
	
	/**
	 * 
	 * @param srcHtml	模版生成html
	 * @param destPdf	pdf文件
	 * @param pageSize	页面大小(A4或A5)
	 * @param isVertical	是否竖版(true为竖版,false为横版)
	 * @throws IOException
	 * @throws InterruptedException
	 */
	public static void generatePdf(String srcHtml, File destPdf, String pageSize, boolean isVertical) throws IOException, InterruptedException{
		generatePdf(srcHtml, destPdf, pageSize, isVertical, null);
	}
	
	/**
	 * 
	 * @param srcHtml	模版生成html
	 * @param destPdf	pdf文件
	 * @param pageSize	页面大小(A4或A5)
	 * @param isVertical	是否竖版(true为竖版,false为横版)
	 * @param title		标题
	 * @throws IOException
	 * @throws InterruptedException
	 */
	public static void generatePdf(String srcHtml, File destPdf, String pageSize, boolean isVertical, String title) throws IOException, InterruptedException{
		
		HtmlToPdfParam params = new HtmlToPdfParam()
				.setVertical(isVertical)
				.setPageSize(pageSize)
				.setTitle(title);
		
		generatePdf(srcHtml, destPdf, params);
	}
	
	public static void generatePdf(String srcHtml, File destPdf, float width, float height, String title) throws IOException, InterruptedException{
//		wkhtmltopdf -q --page-height 297 --page-width 210 --disable-smart-shrinking  --dpi 250   printA4.html printA4.pdf
		
		HtmlToPdfParam params = new HtmlToPdfParam()
				.setPageHeight(String.valueOf(height))
				.setPageWidth(String.valueOf(width))
				.setTitle(title);
		
		generatePdf(srcHtml, destPdf, params );
	}
	
	public static void generatePdf(String html, File file, HtmlToPdfParam params) throws IOException, InterruptedException {
		
		String fileName= UUIDGenerator.generateUUID() + ".content.html";
		File htmlFile = BaseFileUtils.createEmptyTmpFile(fileName, html);
		
		generatePdf(htmlFile, file, params);
		
		FileUtils.forceDelete(htmlFile);
		if (params.getHeaderFile() != null) {
			FileUtils.forceDelete(params.getHeaderFile());
		}
		if (params.getFooterFile() != null) {
			FileUtils.forceDelete(params.getFooterFile());
		}
	}
	
	public static void generatePdf(File htmlFile, File file, HtmlToPdfParam params) throws IOException, InterruptedException {
		if (file == null) {
			throw new NullPointerException("pdf file cannot be null");
		}
		
		if (!file.exists()) {
			FileUtils.touch(file);
		}
		
		StringBuffer cmd = new StringBuffer();
		cmd.append(WXHTMLTOPDF_BIN_PATH)
			.append(" ");
		
		cmd.append(params.generateCmdParams());
		
		cmd.append(htmlFile.getAbsolutePath())
			.append(" ")
			.append(file.getAbsolutePath());
		
		logger.info("run cmd:" + cmd);
		
		Process proc = Runtime.getRuntime().exec(cmd.toString());
		runCmdLog(proc.getInputStream(), proc.getErrorStream());
        
		proc.waitFor();
	}
	
	private static void runCmdLog(InputStream outputStream, InputStream errorStream){ 
		HtmlToPdfInterceptor output = new HtmlToPdfInterceptor(outputStream);
		HtmlToPdfInterceptor error = new HtmlToPdfInterceptor(errorStream);
		
		ThreadPoolTaskExecutor executor = SpringContextHolder.getOneBean(ThreadPoolTaskExecutor.class);
		if (executor != null) {
			executor.execute(output);
			executor.execute(error);
		}else{
			Thread thread1 = new Thread(output);
			thread1.start();
			
			Thread thread2 = new Thread(error);
			thread2.start();
		}
		
	}
	
	public static void main(String[] args) {
		String tempDir = HtmlToPdfUtils.class.getClassLoader().getResource("") + "tempDir"; 
//		FileUtils.forceMkdir(new File(tempDir));
		
		String htmlFilePath = tempDir + "/" + System.currentTimeMillis() + ".html";
		File htmlFile = new File(htmlFilePath);
		System.out.println(htmlFile.getAbsolutePath());
	}
	
}
