package com.els.base.utils.pdf;

import java.io.File;
import java.io.IOException;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;

import com.els.base.utils.file.BaseFileUtils;
import com.els.base.utils.uuid.UUIDGenerator;

public class HtmlToPdfParam {
	
	public static final String A4 = "A4";
	public static final String A5 = "A5";
	public static final String LETTER = "Letter";
	public static final String Letter_A5 = "LetterA5";
	
	/**
	 * 设置页面下边距 (default 00mm) 
	 */
	private String marginBottom = "0mm";
	/**
	 * 将左边页边距 (default 0mm)
	 */
	private String marginLeft = "0mm"; 
	/**
	 * 设置页面右边距 (default 0mm)
	 */
	private String marginRight = "0mm";
	/**
	 * 设置页面上边距 (default 0mm)
	 */
	private String marginTop = "0mm";
	/**
	 * 禁止使用WebKit的智能战略收缩，使像素/ DPI比没有不变。默认true
	 */
	private boolean disableSmartShrinking = true;
	
	/**
	 * 使用的打印介质类型，而不是屏幕。默认true
	 */
	private boolean printMediaType = true;
	
	/**
	 * 页面高度,如果设置了pageSize,这个设置会失效
	 */
	private String pageHeight; 
	
	/**
	 * 页面宽度,如果设置了pageSize,这个设置会失效
	 */
	private String pageWidth;
	
	/**
	 * 显式更改DPI,默认 250
	 */
	private Integer dpi = 250;
	
	/**
	 * 设置纸张大小: A4, Letter等等
	 */
	private String pageSize;
	
	/**
	 * 生成的PDF文件的标题
	 */
	private String title;
	
	/**
	 * 设置页眉和内容的距离,默认0
	 */
	private String headerSpacing;
	
	/**
	 * 设置页脚和内容的距离
	 */
	private String footerSpacing;
	
	/**
	 * 是否垂直显示，默认true
	 */
	private boolean isVertical = true;
	
	/**
	 * 页眉的html 内容（非文件）
	 */
	private String headerHtml;
	
	/**
	 * 页脚的html 内容(非文件)
	 */
	private String footerHtml;
	
	/**
	 * 页眉的文件
	 */
	private File headerFile;
	
	/**
	 * 页脚的文件
	 */
	private File footerFile;

	public String getMarginBottom() {
		return marginBottom;
	}

	public HtmlToPdfParam setMarginBottom(String marginBottom) {
		this.marginBottom = marginBottom;
		return this;
	}

	public String getMarginLeft() {
		return marginLeft;
	}

	public HtmlToPdfParam setMarginLeft(String marginLeft) {
		this.marginLeft = marginLeft;
		return this;
	}

	public String getMarginRight() {
		return marginRight;
	}

	public HtmlToPdfParam setMarginRight(String marginRight) {
		this.marginRight = marginRight;
		return this;
	}

	public String getMarginTop() {
		return marginTop;
	}

	public HtmlToPdfParam setMarginTop(String marginTop) {
		this.marginTop = marginTop;
		return this;
	}

	public boolean isDisableSmartShrinking() {
		return disableSmartShrinking;
	}

	public HtmlToPdfParam setDisableSmartShrinking(boolean disableSmartShrinking) {
		this.disableSmartShrinking = disableSmartShrinking;
		return this;
	}

	public boolean isPrintMediaType() {
		return printMediaType;
	}

	public HtmlToPdfParam setPrintMediaType(boolean printMediaType) {
		this.printMediaType = printMediaType;
		return this;
	}

	public String getPageHeight() {
		return pageHeight;
	}

	public HtmlToPdfParam setPageHeight(String pageHeight) {
		this.pageHeight = pageHeight;
		return this;
	}

	public String getPageWidth() {
		return pageWidth;
	}

	public HtmlToPdfParam setPageWidth(String pageWidth) {
		this.pageWidth = pageWidth;
		return this;
	}

	public Integer getDpi() {
		return dpi;
	}

	public HtmlToPdfParam setDpi(Integer dpi) {
		this.dpi = dpi;
		return this;
	}

	public String getPageSize() {
		return pageSize;
	}

	public HtmlToPdfParam setPageSize(String pageSize) {
		this.pageSize = pageSize;
		return this;
	}

	public String getTitle() {
		return title;
	}

	public HtmlToPdfParam setTitle(String title) {
		this.title = title;
		return this;
	}

	public String getHeaderSpacing() {
		return headerSpacing;
	}

	public HtmlToPdfParam setHeaderSpacing(String headerSpacing) {
		this.headerSpacing = headerSpacing;
		return this;
	}

	public String getFooterSpacing() {
		return footerSpacing;
	}

	public HtmlToPdfParam setFooterSpacing(String footerSpacing) {
		this.footerSpacing = footerSpacing;
		return this;
	}

	public boolean isVertical() {
		return isVertical;
	}

	public HtmlToPdfParam setVertical(boolean isVertical) {
		this.isVertical = isVertical;
		return this;
	}

	public String getHeaderHtml() {
		return headerHtml;
	}

	public HtmlToPdfParam setHeaderHtml(String headerHtml) {
		this.headerHtml = headerHtml;
		return this;
	}

	public String getFooterHtml() {
		return footerHtml;
	}

	public HtmlToPdfParam setFooterHtml(String footerHtml) {
		this.footerHtml = footerHtml;
		return this;
	}
	
	public File getHeaderFile() {
		return headerFile;
	}

	public File getFooterFile() {
		return footerFile;
	}

	public String generateCmdParams() throws IOException{
		Map<String, String> paramsMap = new LinkedHashMap<>();
		paramsMap.put("-T", this.marginTop);
		paramsMap.put("-B", this.marginBottom);
		paramsMap.put("-L", marginLeft);
		paramsMap.put("-R", this.marginRight);
		paramsMap.put("--dpi", this.dpi.toString());
		
		if (StringUtils.isNotBlank(this.title)) {
			paramsMap.put("--title", title);
		}
		
		if (disableSmartShrinking) {
			paramsMap.put("--disable-smart-shrinking", "");
		}
		
		if (this.printMediaType) {
			paramsMap.put("--print-media-type", "");
		}
		
		if (!this.isVertical) {
			paramsMap.put("-O", "Landscape");
		}
		
		this.setPageSize(paramsMap);
		
		this.setHeaderOrFooter(paramsMap);
		
		return generateCmdParams(paramsMap);
	}

	private void setHeaderOrFooter(Map<String, String> paramsMap) throws IOException {
		if (StringUtils.isNotBlank(this.headerHtml)) {
			if (StringUtils.isNotBlank(headerSpacing)) {
				paramsMap.put("--header-spacing", this.headerSpacing);
			}
			
			String fileName= UUIDGenerator.generateUUID() + ".header.html";
			this.headerFile = BaseFileUtils.createEmptyTmpFile(fileName, this.headerHtml);
			paramsMap.put("--header-html", this.headerFile.getAbsolutePath());
		}
		
		if (StringUtils.isNotBlank(this.footerHtml)) {
			if (StringUtils.isNotBlank(footerSpacing)) {
				paramsMap.put("--footer-spacing", this.footerSpacing);
			}
			
			String fileName= UUIDGenerator.generateUUID() + ".footer.html";
			this.footerFile = BaseFileUtils.createEmptyTmpFile(fileName, this.footerHtml);
			paramsMap.put("--footer-html", footerFile.getAbsolutePath());
		}
	}

	private void setPageSize(Map<String, String> paramsMap) {
		if (StringUtils.isNotBlank(this.pageSize)
				&& !this.pageSize.trim().equals(Letter_A5)) {
			
			paramsMap.put("--page-size", this.pageSize);
			return;
		}
		
		if (Letter_A5.equals(pageSize)) {
			this.pageHeight = "140mm";
			this.pageWidth = "216mm";
		}
		
		if (StringUtils.isNotBlank(this.pageHeight)) {
			paramsMap.put("--page-height", this.pageHeight);
		}
		
		if (StringUtils.isNotBlank(this.pageWidth)) {
			paramsMap.put("--page-width", this.pageWidth);
		}
	}

	private String generateCmdParams(Map<String, String> params) {
		if (MapUtils.isEmpty(params)) {
			return "";
		}
		StringBuffer cmd = new StringBuffer();
		
		Iterator<String> keyIterator = params.keySet().iterator();
		while (keyIterator.hasNext()) {
			String key = keyIterator.next();
			String value = params.get(key);
			
			cmd.append(key).append(" ").append(value).append(" ");
		}
		
		return cmd.toString();
	}
	
}
