package com.els.base.utils;

import java.util.Map;

import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

/**
 * 在服务器启动的时候，Spring容器就已经实例化好了一个ApplicationContext对象。能得到一个ApplicationContext对象呢？
 * 方法很多，最常用的办法就是用ClassPathXmlApplicationContext， FileSystemClassPathXmlApplicationContext， FileSystemXmlApplicationContext 等对象去加载Spring配置文件，
 * 这样做也是可以， 但是在加载Spring配置文件的时候，就会生成一个新的ApplicationContext对象而不是Spring容器帮我们生成的那一个，这样就产生了冗余。
 * 所以我们在这里不采用这种加载文件的方式，我们使用ApplicationContextAware让Spring容器传递自己生成的ApplicationContext给我们，
 * 然后我们把这个ApplicationContext设置成一个类的静态变量， 这样我们就随时都可以在老的代码里得到Application的对象了。
 * @author Administrator
 */
public class SpringContextHolder implements ApplicationContextAware{

	private static ApplicationContext applicationContext;
	
	@Override
	public void setApplicationContext(ApplicationContext arg0) throws BeansException {
		applicationContext = arg0;
	}
	
	/**
     * 取得存储在静态变量中的ApplicationContext.
     */
    public static ApplicationContext getApplicationContext() {
        checkApplicationContext();
        return applicationContext;
    }
    
    /**
     * 从静态变量ApplicationContext中取得Bean, 自动转型为所赋值对象的类型.
     */
    @SuppressWarnings("unchecked")
    public static <T> T getBean(String name) {
        return (T) getApplicationContext().getBean(name);
    }

    /**
     * 从静态变量ApplicationContext中取得Bean, 自动转型为所赋值对象的类型.
     */
    public static <T> T getOneBean(Class<T> clazz) {
        Map<String, T> beanMaps = getApplicationContext().getBeansOfType(clazz);
        if (beanMaps!=null && !beanMaps.isEmpty()) {
            return beanMaps.values().iterator().next();
        } else{
            return null;
        }
    }
    
    public static <T> Map<String, T> getBeans(Class<T> clazz) {
        Map<String, T> beanMaps = getApplicationContext().getBeansOfType(clazz);
        return beanMaps;
    }

    private static void checkApplicationContext() {
        if (applicationContext == null) {
            throw new IllegalStateException("spring 的配置文件中，未配置SpringContextHolder");
        }
    }
    
    /**
     * 获取系统的配置
     * @param key
     * @return
     */
    public static String getProperty(String key){
    	return getApplicationContext().getEnvironment().getProperty(key);
    }
    
    /**
     * 获取系统的配置
     * @param key
     * @return
     */
    public static String getProperty(String key, String defaultValue){
    	return getApplicationContext().getEnvironment().getProperty(key, defaultValue);
    }

}
