package com.els.base.wechat.oauth.web.controller;

import java.io.IOException;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.NameValuePair;
import org.apache.http.client.utils.URLEncodedUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.servlet.ModelAndView;

import com.els.base.core.entity.ResponseResult;
import com.els.base.core.exception.CommonException;
import com.els.base.utils.SpringContextHolder;
import com.els.base.wechat.account.entity.AccountConfig;
import com.els.base.wechat.account.entity.AccountConfigExample;
import com.els.base.wechat.account.service.AccountConfigService;
import com.els.base.wechat.account.utils.WxAccountConfigUtils;
import com.els.base.wechat.common.WxConstant;
import com.els.base.wechat.common.WxMpServiceUtils;
import com.els.base.wechat.member.service.WxMemberService;
import com.els.base.wechat.oauth.service.WechatOauthService;
import com.els.base.wechat.oauth.utils.WechatOauthUtils;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import me.chanjar.weixin.common.exception.WxErrorException;
import me.chanjar.weixin.mp.api.WxMpService;
import me.chanjar.weixin.mp.bean.result.WxMpOAuth2AccessToken;
import me.chanjar.weixin.mp.bean.result.WxMpUser;
import springfox.documentation.annotations.ApiIgnore;

@Api("微信-公众号网页授权接口")
@Controller
public class WechatOauthController {
	
	private static Logger logger = LoggerFactory.getLogger(WechatOauthController.class); 
	
    /**json字符串的正则表达式**/
    private static final Pattern urlPatther = Pattern.compile("((http|https):\\/\\/.+)?\\?(.+)");
	private static ObjectMapper objectMapper = new ObjectMapper();
	private static String defaultPage;
	
	@Autowired
	protected AccountConfigService accountConfigService;
	
	@Autowired
	private WxMemberService wxMemberService;
	
	/**
	 * 接受微信传来的 授权信息
	 * @param code openid
	 * @param params 将要跳转的地址
	 * @param modelMap
	 * @param request
	 * @return
	 * @throws WxErrorException
	 */
	@ApiIgnore
	@RequestMapping(WxConstant.OAUTH_2_GET_OPENID + "/{params}")
	public ModelAndView receiveOpenid(@PathVariable("params") String params, String code, HttpServletRequest request, HttpServletResponse response) throws WxErrorException{
		logger.debug("receive wechat post code :" + code +", with params :" + params);

		Map<String, String> paramsMap = this.getParamsMapFromState(params);
		AccountConfig accountConfig = this.accountConfigService.queryByOriginId(paramsMap.get(WxConstant.PARAMS_MAP_KEY_TARGET_APPID));
		
		WxMpService wxMpService = WxMpServiceUtils.getWxMpServiceByAccount(accountConfig);
		logger.debug("wxMpService:" + wxMpService + ",account:" +accountConfig);
		
		WxMpOAuth2AccessToken wxMpOAuth2AccessToken = wxMpService.oauth2getAccessToken(code);
		String openId = wxMpOAuth2AccessToken.getOpenId();
		WxMpUser wxMpUser = this.getWxMpUserByOpenId(openId, wxMpService);
		this.wxMemberService.createOrEditWxMember(wxMpUser, null, accountConfig.getOriginId());
		
		request.getSession().setAttribute(WxConstant.PARAMS_MAP_KEY_OPENID, openId);
		handleWithOpenidAndParamsMap(openId, paramsMap, request, response);
		
		String viewName = this.getRedirectUrl(paramsMap);
		paramsMap.put(WxConstant.PARAMS_MAP_KEY_OPENID, openId);
		paramsMap.remove(WxConstant.PARAMS_MAP_KEY_TARGET_URL);
		
		return new ModelAndView(viewName, paramsMap);
	}
	
	/**
	 * 根据授权获取会员信息
	 * @param code
	 * @param params
	 * @return
	 * @throws WxErrorException
	 */
	@ApiIgnore
	@RequestMapping(WxConstant.OAUTH_2_GET_WX_USER_INFO  + "/{params}")
	public ModelAndView receiveCodeAndGetWxUserInfo(@PathVariable("params") String params, String code, HttpServletRequest request, HttpServletResponse response) throws WxErrorException{
		logger.debug("receive wechat post code :" + code +", with params :" + params);
		long t1 = System.currentTimeMillis();
		
		Map<String, String> paramsMap = this.getParamsMapFromState(params);
		AccountConfig accountConfig = this.accountConfigService.queryByOriginId(paramsMap.get(WxConstant.PARAMS_MAP_KEY_TARGET_APPID));
		
		WxMpService wxMpService = WxMpServiceUtils.getWxMpServiceByAccount(accountConfig);
		WxMpOAuth2AccessToken wxMpOAuth2AccessToken = wxMpService.oauth2getAccessToken(code);
		
		logger.debug("get oAuth2 AccessToken, cost [{}ms], AccessToken:", (System.currentTimeMillis() - t1), wxMpOAuth2AccessToken);
		
		long t2 = System.currentTimeMillis();
		WxMpUser wxMpUser = wxMpService.oauth2getUserInfo(wxMpOAuth2AccessToken, null);
		String openId = wxMpOAuth2AccessToken.getOpenId();
		request.getSession().setAttribute(WxConstant.PARAMS_MAP_KEY_OPENID, openId);
		
		logger.debug("get userinfo cost[{}ms], userInfo", (System.currentTimeMillis()-t2), wxMpUser);
		this.wxMemberService.createOrEditWxMember(wxMpUser, null, accountConfig.getOriginId());
		handleWithWxUserAndParamsMap(wxMpUser, paramsMap, request, response);
		
		String viewName = this.getRedirectUrl(paramsMap);
		request.getSession().setAttribute(WxConstant.PARAMS_MAP_KEY_OPENID, openId);
		request.getSession().setAttribute("wxMpUser", wxMpUser);
		paramsMap.put(WxConstant.PARAMS_MAP_KEY_OPENID, openId);
		paramsMap.remove(WxConstant.PARAMS_MAP_KEY_TARGET_URL);
		
		return new ModelAndView(viewName, paramsMap);
	}
	
	private String getRedirectUrl(Map<String, String> paramsMap) {
		String targetUrl = null;
		if (paramsMap != null) {
			targetUrl = paramsMap.get(WxConstant.PARAMS_MAP_KEY_TARGET_URL);
		}
		String serverName = StringUtils.substringBefore(StringUtils.substringBetween(targetUrl, "://", "/"),":");
		if(StringUtils.isBlank(targetUrl)){
			targetUrl = serverName + defaultPage;
		}else if (targetUrl.startsWith("/")) {
			targetUrl = serverName + targetUrl;
		}
		
		return "redirect:" + targetUrl; 
//		return "forward:" + targetUrl;
	}

	private void handleWithOpenidAndParamsMap(String openId, Map<String, String> paramsMap, HttpServletRequest request, HttpServletResponse response) {
		
		ApplicationContext applicationContext = SpringContextHolder.getApplicationContext();
		Map<String, WechatOauthService> beanMaps = applicationContext.getBeansOfType(WechatOauthService.class);
		if (MapUtils.isEmpty(beanMaps)) {
			return;
		}
		
		Set<String> beanSet = beanMaps.keySet();
		Iterator<String> iterator = beanSet.iterator();
		while (iterator.hasNext()) {
			beanMaps.get(iterator.next()).handleForOpenId(openId, paramsMap, request, response);
		}
	}
	
	private void handleWithWxUserAndParamsMap(WxMpUser wxMpUser, Map<String, String> paramsMap,
			HttpServletRequest request, HttpServletResponse response) {
		ApplicationContext applicationContext = SpringContextHolder.getApplicationContext();
		Map<String, WechatOauthService> beanMaps = applicationContext.getBeansOfType(WechatOauthService.class);
		if (MapUtils.isEmpty(beanMaps)) {
			return;
		}
		
		Set<String> beanSet = beanMaps.keySet();
		Iterator<String> iterator = beanSet.iterator();
		while (iterator.hasNext()) {
			beanMaps.get(iterator.next()).handleForWxUser(wxMpUser, paramsMap, request, response);
		}
	}
	
	private WxMpUser getWxMpUserByOpenId(String openId, WxMpService wxMpService) {
		WxMpUser wxMpUser = null;
		try {
			wxMpUser = wxMpService.getUserService().userInfo(openId);
		} catch (WxErrorException e) {
			logger.warn("向公众号查询用户信息失败", e);
		}
		
		if (wxMpUser == null) {
			wxMpUser = new WxMpUser();
			wxMpUser.setOpenId(openId);
		}
		
		return wxMpUser;
	}

	/**
	 * 根据微信服务返回的 参数 “state”，解析出参数map
	 * @param params
	 * @return
	 * @throws JsonParseException
	 * @throws JsonMappingException
	 * @throws IOException
	 */
	@SuppressWarnings("unchecked")
	private Map<String, String> getParamsMapFromState(String params) {
		Map<String, String> paramsMap= new Hashtable<>();
		
		if (StringUtils.isBlank(params)) {
			return null;
		}
		
		byte[] byteArray = Base64.decodeBase64(params);
		params = new String(byteArray);
		
		// 如果是字符串格式
		if (!params.contains("\"")) { 
			Pattern tuPattern = Pattern.compile("((\\w+):(.+?))[\\,\\}]");
			Matcher tuMatcher = tuPattern.matcher(params);
			
			while(tuMatcher.find()){
				String key = tuMatcher.group(2);
				String value = tuMatcher.group(3);
				
				if(StringUtils.isNotBlank(key)){
					paramsMap.put(key, value);
				}
			}
			return paramsMap;
		}
		
		// 如果是json格式
		try {
			paramsMap = objectMapper.readValue(params, Map.class);
			logger.debug("get map from json :" + paramsMap);
			
		} catch (IOException e) {
			logger.warn("解析微信返回的state失败", e);
		}

		return paramsMap;
	}
	
	
	@ApiOperation(httpMethod="GET", value="获取授权openid的跳转地址")
    @RequestMapping("wechatOauth/front/createOauthUrlForOpenid")
    @ResponseBody
	public ResponseResult<String> createOauthUrlForOpenid(
			@ApiParam(value="授权的页面") String url, 
			@ApiParam(value="微信公众号的原始id，可以不填") String originId, HttpServletRequest request) throws Exception {
		
		Map<String, String> paramsMap = this.createUrlParamMap(url);
		String targetUrl = this.getTargetUrl(url);
		AccountConfig accountConfig = this.getAccountByOriginIdOrUrl(originId, targetUrl);
		
		if (accountConfig == null) {
			throw new CommonException("系统没有合适的公众号配置");
		}
		
		String oauthUrl = WechatOauthUtils.createOauthUrlForOpenid(targetUrl, paramsMap, accountConfig);
		
		return ResponseResult.success(oauthUrl);
	}
	
	@ApiOperation(httpMethod="GET", value="获取会员资料的跳转地址")
    @RequestMapping("wechatOauth/front/createOauthUrlForWxUserInfo")
    @ResponseBody
	public ResponseResult<String> createOauthUrlForWxUserInfo(
			@ApiParam(value="授权的页面") @RequestParam(required=true) String url, 
			@ApiParam(value="微信公众号的原始id，可以不填") String originId, HttpServletRequest request) throws Exception {
		
		Map<String, String> paramsMap = this.createUrlParamMap(url);
		String targetUrl = this.getTargetUrl(url);
		AccountConfig accountConfig = this.getAccountByOriginIdOrUrl(originId, targetUrl);
		
		if (accountConfig == null) {
			throw new CommonException("系统没有合适的公众号配置");
		}
		
		String oauthUrl = WechatOauthUtils.createOauthUrlForWxUserInfo(targetUrl, paramsMap, accountConfig);
		
		return ResponseResult.success(oauthUrl);
	}

	private AccountConfig getAccountByOriginIdOrUrl(String originId, String url) {
		AccountConfig accountConfig = null;
		if (StringUtils.isNotBlank(originId)) {
			accountConfig = WxAccountConfigUtils.getByOrginId(originId);
		}
		
		if (accountConfig == null) {
			String serverName = StringUtils.substringBetween(url, "://", "/");
			String requestUri = url.replaceAll("https?://.+?/", "/");
			accountConfig = this.accountConfigService.queryByRequest("http", serverName, 80, requestUri);
		}
		
		if (accountConfig == null) {
			AccountConfigExample example = new AccountConfigExample();
			List<AccountConfig> list = this.accountConfigService.queryAllObjByExample(example);
			if (CollectionUtils.isNotEmpty(list) && list.size() == 1) {
				accountConfig = list.get(0);
			}
		}
		return accountConfig;
	}

	private Map<String, String> createUrlParamMap(String url) {
		Matcher matcher = urlPatther.matcher(url);
		if (!matcher.find()) {
			return null;
		}
		
		Map<String, String> urlParamsMap = new HashMap<>();
		List<NameValuePair> list = URLEncodedUtils.parse(matcher.group(3), Charset.forName("utf-8"));
		for (int i = 0; i < list.size(); i++) {
			NameValuePair params = list.get(i);
			System.out.println("name" + params.getName() +",value" + params.getValue());
			urlParamsMap.put(params.getName(), params.getValue());
		}
		
		return urlParamsMap;
	}
	
	private String getTargetUrl(String url) {
		Matcher matcher = urlPatther.matcher(url);
		if (matcher.find()) {
			return matcher.group(1);
			
		}else{
			return url;
		}
	}
	
}
