package com.els.base.certification.certificate.util;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Service;

import com.els.base.certification.agent.entity.Agent;
import com.els.base.certification.agent.entity.AgentExample;
import com.els.base.certification.agent.service.AgentService;
import com.els.base.certification.auth.entity.CompanyAuth;
import com.els.base.certification.auth.entity.CompanyAuthExample;
import com.els.base.certification.auth.service.CompanyAuthService;
import com.els.base.certification.certificate.entity.CertificateExpiry;
import com.els.base.certification.certificate.entity.CertificateExpiryExample;
import com.els.base.certification.certificate.service.CertificateExpiryService;
import com.els.base.certification.common.CertificateTypeCodeEnum;
import com.els.base.certification.patents.entity.CompanyPatents;
import com.els.base.certification.patents.entity.CompanyPatentsExample;
import com.els.base.certification.patents.service.CompanyPatentsService;
import com.els.base.certification.remind.entity.RemindInfo;
import com.els.base.certification.remind.entity.RemindInfoExample;
import com.els.base.certification.remind.entity.RemindInfoExample.Criteria;
import com.els.base.certification.remind.service.RemindInfoService;
import com.els.base.company.entity.Company;
import com.els.base.company.service.CompanyService;
import com.els.base.core.utils.Constant;


/**
 * 证书到期提醒任务
 * @author lenovo
 *
 */

@Service
public class ExpirationRemindingJob {
	
	@Resource
	private CertificateExpiryService certificateExpiryService;
	
	@Resource
	private CompanyAuthService companyAuthService;
	
	@Resource
	private CompanyService companyService;
	
	@Resource
	private RemindInfoService remindInfoService;
	
	@Resource
	private CompanyPatentsService companyPatentsService;
	
	@Resource
	private AgentService agentService;
	
	public void remindInformation() throws ParseException {
		// 每天对T_BASE_SUP_REMIND_INFO到期提醒供应商信息表进行全表信息删除操作，然后再添加要过期的证书信息
		RemindInfoExample remindInfoExample = new RemindInfoExample();
		remindInfoExample.createCriteria().andIsEnableEqualTo(Constant.YES_INT);
		this.remindInfoService.deleteByExample(remindInfoExample);
		// 查出全部可用的证书到期提醒信息
		CertificateExpiryExample example = new CertificateExpiryExample();
		example.createCriteria().andIsEnableEqualTo(Constant.YES_INT)
			.andIsEnableRemindersEqualTo(Constant.YES_INT);
		List<CertificateExpiry> certificateList = this.certificateExpiryService.queryAllObjByExample(example);
		if (CollectionUtils.isEmpty(certificateList)) {
			return;
		}
		/**
		 * 现在证书到期提醒任务监控了供应商基础资料的三个页签：
		 * 1.企业的通过认证
		 * 2.专利/专用技术/许可(patent)
		 * 3.代理数据，代理数据中有代理证(agent)、生产商质量证书(quality)
		 */
		// 根据到期提醒信息中分类名称查询证书信息
		for (CertificateExpiry certificate : certificateList){
			if (certificate.getClassificationName().equals(CertificateTypeCodeEnum.PATENT.getCode())) {
				// 专利/许可证书
				CompanyPatentsExample patentsExample = new CompanyPatentsExample();
				patentsExample.createCriteria()
								.andIsEnableEqualTo(Constant.YES_INT)
								.andEndTimeIsNotNull();
				List<CompanyPatents> list = this.companyPatentsService.queryAllObjByExample(patentsExample);
				if (CollectionUtils.isEmpty(list)) {
					continue;
				}
				for (CompanyPatents companyPatents : list) {
					this.insertToRemindInfo(companyPatents.getEndTime(), companyPatents.getSupCompanyId(), CertificateTypeCodeEnum.PATENT.getCode(), 
							CertificateTypeCodeEnum.PATENT.getDesc(), certificate.getAdvanceReminderDays());
				}
			} else if (CertificateTypeCodeEnum.AGENT.getCode().equals(certificate.getClassificationName())) {
				// 代理证书
				AgentExample agentExample = new AgentExample();
				agentExample.createCriteria().andIsEnableEqualTo(Constant.YES_INT)
											.andAgentTimeIsNotNull();
				List<Agent> list = this.agentService.queryAllObjByExample(agentExample);
				if (CollectionUtils.isEmpty(list)) {
					continue;
				}
				for (Agent agent : list) {
					this.insertToRemindInfo(agent.getAgentTime(), agent.getCompanyId(), CertificateTypeCodeEnum.AGENT.getCode(), 
							CertificateTypeCodeEnum.AGENT.getDesc(), certificate.getAdvanceReminderDays());
				}
				
			} else if (CertificateTypeCodeEnum.QUALITY.getCode().equals(certificate.getClassificationName())) {
				// 生产商质量证书
				AgentExample agentExample = new AgentExample();
				agentExample.createCriteria().andIsEnableEqualTo(Constant.YES_INT)
											.andQualityCertificateTimeIsNotNull();
				List<Agent> list = this.agentService.queryAllObjByExample(agentExample);
				if (CollectionUtils.isEmpty(list)) {
					continue;
				}
				for (Agent agent : list) {
					this.insertToRemindInfo(agent.getQualityCertificateTime(), agent.getCompanyId(), CertificateTypeCodeEnum.QUALITY.getCode(), 
							CertificateTypeCodeEnum.QUALITY.getDesc(), certificate.getAdvanceReminderDays());
				}
			} else {
				// 企业通过认证
				CompanyAuthExample authExample = new CompanyAuthExample();
				authExample.createCriteria().andIsEnableEqualTo(Constant.YES_INT)
					.andCertificateTypeCodeEqualTo(certificate.getClassificationName())
					.andEndTimeIsNotNull();
				List<CompanyAuth> authList = this.companyAuthService.queryAllObjByExample(authExample);
				if (CollectionUtils.isEmpty(authList)) {
					continue;
				}
				for (CompanyAuth auth : authList) {
					// 判断是否有证书过期
					String name = CertificateTypeCodeEnum.getDesc(auth.getCertificateTypeCode());
					this.insertToRemindInfo(auth.getEndTime(),auth.getSupCompanyId(),auth.getCertificateTypeCode(),name,certificate.getAdvanceReminderDays());
				}
			}
		}
	}

	private void insertToRemindInfo(Date endTime, String supCompanyId, String code, String name,
			Integer advanceReminderDays) throws ParseException {
		SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
		String newDate = formatter.format(new Date());
		String endDate = formatter.format(endTime);
		// 现在时间毫秒值
		long nowDateTime = formatter.parse(newDate).getTime();
		// 证书到期时间毫秒值
		long endDateTime = formatter.parse(endDate).getTime();
		// 提前提醒天数毫秒值
		long reminderTime = advanceReminderDays*24*60*60*1000;
		// 证书的到期时间减去现在时间，结果小于或等于提前提醒的天数，界面的提醒信息就会一直显示
		if (endDateTime - nowDateTime <= reminderTime) {
			// 根据供应商ID 查询供应商信息
			Company company = this.companyService.queryObjById(supCompanyId);
			if (company == null) {
				return;
			}
			RemindInfo remindInfo = new RemindInfo();
			remindInfo.setSupCompanyId(company.getId());
			remindInfo.setSupCompanyName(company.getCompanyFullName());
			remindInfo.setSupCompanySrmCode(company.getCompanyCode());
			remindInfo.setCertificateTypeCode(code);
			remindInfo.setCertificateName(name);
			remindInfo.setEndTime(endTime);
			remindInfo.setIsEnable(Constant.YES_INT);
			remindInfo.setCreateTime(new Date());
			// 单据状态未确认
			remindInfo.setStatus(Constant.NO_INT);
			this.remindInfoService.addObj(remindInfo);
		}
	}
}
