package com.els.base.certification.contacts.web.controller;

import com.els.base.certification.common.CompanyFinishFlag;
import com.els.base.company.service.CompanyService;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Constant;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.project.ProjectUtils;
import com.els.base.core.utils.query.QueryParamWapper;
import com.els.base.certification.contacts.entity.Contacts;
import com.els.base.certification.contacts.entity.ContactsExample;
import com.els.base.certification.contacts.service.ContactsService;
import com.els.base.company.entity.Company;
import com.els.base.company.utils.CompanyUtils;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import java.util.List;
import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

@Api(value="联系人信息")
@Controller
@RequestMapping("contacts")
public class ContactsController {
    @Resource
    protected ContactsService contactsService;
    @Resource
    private CompanyService companyService;

    @ApiOperation(httpMethod="POST", value="创建联系人信息")
    @RequestMapping("service/create")
    @ResponseBody
    public ResponseResult<CompanyFinishFlag> create(@RequestBody List<Contacts> contacts) {
    	if(CollectionUtils.isEmpty(contacts)){
    		throw new CommonException("操作失败，数据不能为空");
    	}
        CompanyFinishFlag companyFinishFlag =new CompanyFinishFlag();
    	Company company=CompanyUtils.currentCompany();
        Integer contact = this.contactsService.addContacts(company.getProjectId(),company.getId(),contacts);
        companyFinishFlag.setContactInfo(contact);
        return ResponseResult.success(companyFinishFlag);
    }

    @ApiOperation(httpMethod="POST", value="编辑联系人信息")
    @RequestMapping("service/edit")
    @ResponseBody
    public ResponseResult<String> edit(@RequestBody List<Contacts> contacts) {
       for(Contacts t:contacts){
    	   if (StringUtils.isBlank(t.getId())) {
    		   throw new CommonException("id 为空，修改失败");
    	   }
    	   this.contactsService.modifyObj(t);
       }
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod="POST", value="删除联系人信息")
    @RequestMapping("service/deleteById")
    @ResponseBody
    public ResponseResult<String> deleteById(@RequestBody(required=true) List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            throw new CommonException("删除失败,id不能为空");
        }
        this.contactsService.deleteObjByIds(ids);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod="POST", value="查询联系人信息")
    @ApiImplicitParams({
             @ApiImplicitParam( name = "supCompanySrmCode", required = false, value = "供应商SRM编码", paramType = "query", dataType = "String"),
             @ApiImplicitParam( name = "pageNo",required = false,value = "所在页", paramType = "query", dataType = "String", defaultValue = "0" ),
             @ApiImplicitParam( name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10" ),
             @ApiImplicitParam( name = "wapper", required = false, value = "查询条件,属性名请参考 Contacts", paramType = "body", dataType = "QueryParamWapper" )
    })
    @RequestMapping("service/findByPage")
    @ResponseBody
    public ResponseResult<PageView<Contacts>> findByPage(
            @RequestParam(required=false) String supCompanySrmCode,
            @RequestParam(defaultValue="0") int pageNo,
            @RequestParam(defaultValue="10") int pageSize,
            @RequestBody(required=false) QueryParamWapper wapper) {

        ContactsExample example = new ContactsExample();
        example.setPageView(new PageView<Contacts>(pageNo, pageSize));
        example.setOrderByClause(" CREATE_TIME DESC ");

        ContactsExample contactsExample =new ContactsExample();
        ContactsExample.Criteria criteria = contactsExample.createCriteria();

        if(StringUtils.isBlank(supCompanySrmCode)){
            //添加查询条件
            criteria.andProjectIdEqualTo(ProjectUtils.getProjectId())
                    .andSupCompanyIdEqualTo(CompanyUtils.currentCompanyId())
                    .andIsEnableEqualTo(Constant.YES_INT);
        }else{
            //否则则按照该编码进行查询
            Company company = companyService.queryCompanyByCode(supCompanySrmCode);
            if(company==null){
                throw new CommonException("公司不存在，请检查");
            }
            criteria.andProjectIdEqualTo(ProjectUtils.getProjectId())
                    .andSupCompanyIdEqualTo(company.getId())
                    .andIsEnableEqualTo(Constant.YES_INT);
        }

        if (wapper != null) {
            CriteriaUtils.addCriterion(criteria, wapper);
        }
        
        PageView<Contacts> pageData = this.contactsService.queryObjByPage(example);
        return ResponseResult.success(pageData);
    }
    
    @ApiOperation(httpMethod = "POST", value = "查询信息")
    @ApiImplicitParams({
            @ApiImplicitParam( name = "supCompanySrmCode", required = false, value = "供应商SRM编码", paramType = "query", dataType = "String"),
            @ApiImplicitParam( name = "wapper", required = false, value = "查询条件,属性名请参考 CompanyPatents", paramType = "body", dataType = "QueryParamWapper" )
    })
	@RequestMapping("service/findContacts")
	@ResponseBody
	public ResponseResult<List<Contacts>> findContacts(
	        @RequestParam(required=false) String supCompanySrmCode,
            @RequestBody(required=false) QueryParamWapper wapper) {

        ContactsExample contactsExample =new ContactsExample();
        ContactsExample.Criteria criteria = contactsExample.createCriteria();
        contactsExample.setOrderByClause(" CREATE_TIME DESC ");

        if(StringUtils.isBlank(supCompanySrmCode)){
            //添加查询条件
            criteria.andProjectIdEqualTo(ProjectUtils.getProjectId())
                    .andSupCompanyIdEqualTo(CompanyUtils.currentCompanyId())
                    .andIsEnableEqualTo(Constant.YES_INT);
        }else{
            //否则则按照该编码进行查询
            Company company = companyService.queryCompanyByCode(supCompanySrmCode);
            if(company==null){
                throw new CommonException("公司不存在，请检查");
            }
            criteria.andProjectIdEqualTo(ProjectUtils.getProjectId())
                    .andSupCompanyIdEqualTo(company.getId())
                    .andIsEnableEqualTo(Constant.YES_INT);
        }

        if (wapper != null) {
            CriteriaUtils.addCriterion(criteria, wapper);
        }

        List<Contacts> contacts = contactsService.queryAllObjByExample(contactsExample);

        return ResponseResult.success(contacts);
    }
}