package com.els.base.certification.qualification.service.impl;

import com.els.base.certification.qualification.util.DocumentStatusEnum;
import com.els.base.certification.qualification.util.QualificationImTemplet;
import com.els.base.codegenerator.service.GenerateCodeService;
import com.els.base.company.entity.Company;
import com.els.base.company.service.CompanyService;
import com.els.base.company.service.CompanyUserRefService;
import com.els.base.certification.process.entity.Process;
import com.els.base.certification.process.service.ProcessService;
import com.els.base.certification.process.util.AccessProcessEnum;
import com.els.base.certification.process.util.AuthenticationEnum;
import com.els.base.certification.process.util.ReviewResultEnum;
import com.els.base.certification.qualification.dao.QualificationMapper;
import com.els.base.certification.qualification.entity.Qualification;
import com.els.base.certification.qualification.entity.QualificationExample;
import com.els.base.certification.qualification.entity.QualificationItem;
import com.els.base.certification.qualification.entity.QualificationItemExample;
import com.els.base.certification.qualification.service.QualificationItemService;
import com.els.base.certification.qualification.service.QualificationService;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.user.User;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.msg.Message;
import com.els.base.msg.MessageLevelEnum;
import com.els.base.msg.MessageSendUtils;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import javax.annotation.Resource;
import org.apache.commons.lang.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

@Service("defaultQualificationService")
public class QualificationServiceImpl implements QualificationService {
	
	private static final String IM_CODE = "ADMITTANCE_QUALIFICATION";
	
    @Resource
    protected QualificationMapper qualificationMapper;
    
    @Resource
    protected CompanyService companyService;
    
    @Resource
	protected GenerateCodeService generateCodeService;
    
    @Resource
    protected QualificationItemService qualificationItemService;
    
    @Resource
    protected CompanyUserRefService companyUserRefService;
    
    @Resource
    protected ProcessService processService;
    
    @Resource
	private ThreadPoolTaskExecutor defaultThreadPool;

    @CacheEvict(value={"qualification"}, allEntries=true)
    @Override
    public void addObj(Qualification t) {
        this.qualificationMapper.insertSelective(t);
    }
    
    @Transactional
    @CacheEvict(value={"qualification"}, allEntries=true)
    @Override
	public void insert(User user, Qualification qualification,Company purCompany) {
    	// 判断此供应商是否已存在单据
    	QualificationExample example = new QualificationExample();
    	example.createCriteria()
    		.andDocumentStatusNotEqualTo(DocumentStatusEnum.TERMINATE_AUDIT.getCode())
    		.andIsEnableEqualTo(Constant.YES_INT)
    		.andSupCompanySrmCodeEqualTo(qualification.getSupCompanySrmCode());
    	if (this.qualificationMapper.countByExample(example) > 0) {
    		throw new CommonException("供应商编码为：" + qualification.getSupCompanySrmCode() +"已存在，不能再创建单据");
    	}
    	// 设置供应商信息
    	this.setSupCompanyInfo(qualification);
    	// 设置采购商信息
    	this.setPurCompanyInfo(user,purCompany,qualification);
    	// 设置其他信息
    	this.setOtherInfo(qualification);
    	// 保存到资质审核表中
    	this.qualificationMapper.insertSelective(qualification);
		// 对资质审核文件表操作
		qualification.getQualificationItemList().forEach(qualificationItem -> {
			Assert.isNotBlank(qualificationItem.getPurFile(), "采购方上传附件不能为空");
			if (qualificationItem.getPurFile().length() > 3900) {
				throw new CommonException("上传的附件个数超长，请减少附件个数后再保存");
			}
			// 资质审核表ID
			qualificationItem.setQualificationReviewId(qualification.getId());
			// 采购方上传时间
			qualificationItem.setPurUploadTime(new Date());
			// 创建时间
			qualificationItem.setCreateTime(new Date());
			// 附件属于采购商上传的(0为采购商上传)
			qualificationItem.setCompanyType(Constant.NO_INT);
			// 执行保存操作
			qualificationItemService.addObj(qualificationItem);
		});
		// 修改准入流程中资质审核状态为审核中
		Process process = this.processService.queryObjByCompanyId(qualification.getSupCompanyId());
		process.setSceneInvestigate("");
		process.setQualificationAudit(ReviewResultEnum.AUDIT.getCode());
		this.processService.modifyObj(process);
	}
    
    /**
     * 设置其他信息
     * @param qualification
     */
    private void setOtherInfo(Qualification qualification) {
    	// 单据是否可用(默认可用)
    	qualification.setIsEnable(Constant.YES_INT);
    	// 新建单据，单据状态为未发送状态
    	qualification.setDocumentStatus(DocumentStatusEnum.UNSENT.getCode());
    	// 制单日期
    	qualification.setCreateTime(new Date());
    	// 单据号
    	qualification.setDocumentNo(generateCodeService.getNextCode("QUALIFICATION_AUDIT_DOCUMENT_NO"));
	}

	/**
     * 设置采购商信息
     * @param user
     * @param purCompany
     * @param qualification
     */
    private void setPurCompanyInfo(User user, Company purCompany, Qualification qualification) {
    	// 采购商用户ID
    	qualification.setPurCompanyUserId(user.getId());
    	// 制单人
    	qualification.setSinglePerson(user.getNickName());
    	// 采购商名称
    	qualification.setPurCompanyName(purCompany.getCompanyName());
    	// 采购商SRM编码
    	qualification.setPurCompanySrmCode(purCompany.getCompanyCode());
	}

	/**
     * 设置供应商信息
     * @param qualification
     */
    private void setSupCompanyInfo(Qualification qualification) {
    	// 根据供应商SRM编码查询供应商信息
    	Company company = companyService.queryCompanyByCode(qualification.getSupCompanySrmCode());
    	
    	qualification.setSupCompanyId(company.getId());
    	qualification.setSupCompanyName(company.getCompanyFullName());
    	qualification.setSupContacts(company.getContacts());
    	qualification.setAddress(company.getAddress());
    	// 根据供应商ID查询供应商用户ID
    	User supUser = this.companyUserRefService.queryMainUserOfCompany(company.getId());
    	qualification.setSupCompanyUserId(supUser.getId());
    	// 供应商走了资质审核流程，将供应商的审核状态改为1，不让供应商再修改其基础资料
    	Company t = new Company();
    	t.setId(company.getId());
    	t.setApproveStatus(Constant.YES_INT);
    	this.companyService.modifyObj(t);
	}

	@CacheEvict(value={"qualification"}, allEntries=true)
    @Override
    public void deleteObjById(String id) {
        this.qualificationMapper.deleteByPrimaryKey(id);
    }
    
    @Transactional
    @CacheEvict(value={"qualification"}, allEntries=true)
    @Override
	public void deleteObjByIds(List<String> ids) {
    	// 单据状态不是未发送的不能删除
		QualificationExample example = new QualificationExample();
		example.createCriteria().andIdIn(ids)
			.andIsEnableEqualTo(Constant.YES_INT)
			.andDocumentStatusNotEqualTo(DocumentStatusEnum.UNSENT.getCode());
		if (qualificationMapper.countByExample(example) > 0){
			throw new CommonException("所选单据不全是未发送单据，不能执行删除操作");
		}
		
		// 删除单据
		Qualification qualification = new Qualification();
		qualification.setIsEnable(Constant.NO_INT);
		example.clear();
		example.createCriteria().andIdIn(ids);
			
		this.qualificationMapper.updateByExampleSelective(qualification, example);
		// 准入流程的资质审核状态变为空
		this.queryAllObjByExample(example).forEach(qualifications ->{
			Process process = this.processService.queryObjByCompanyId(qualifications.getSupCompanyId());
			process.setQualificationAudit(null);
			this.processService.updateById(process);
		});
	}

    @CacheEvict(value={"qualification"}, allEntries=true)
    @Override
    public void modifyObj(Qualification t) {
        if (StringUtils.isBlank(t.getId())) {
            throw new NullPointerException("id 为空，无法更新");
        }
        this.qualificationMapper.updateByPrimaryKeySelective(t);
    }
    
    @Transactional
    @CacheEvict(value={"qualification"}, allEntries=true)
    @Override
	public void purModifyObj(Qualification qualification) {
		if(! DocumentStatusEnum.UNSENT.getCode().equals(qualification.getDocumentStatus())) {
			throw new CommonException("不是未发送单据，不能修改");
		}
		// 判断供应商有没有改变，如果变了，看有没有重复
		Qualification cation = this.qualificationMapper.selectByPrimaryKey(qualification.getId());
		if (!cation.getSupCompanySrmCode().equals(qualification.getSupCompanySrmCode())) {
			QualificationExample qualificationExample = new QualificationExample();
			qualificationExample.createCriteria()
				.andSupCompanySrmCodeEqualTo(qualification.getSupCompanySrmCode())
				.andDocumentStatusNotEqualTo(DocumentStatusEnum.TERMINATE_AUDIT.getCode())
				.andIsEnableEqualTo(Constant.YES_INT);
			if (this.qualificationMapper.countByExample(qualificationExample) > 0) {
				throw new CommonException("编码为：" +qualification.getSupCompanySrmCode() +"的供应商已存在单据，不能再创建单据");
			}
		}
		// 设置供应商信息
		this.setSupCompanyInfo(qualification);
		// 更新日期
    	qualification.setUpdateTime(new Date());
    	//资质审核附件不能为空
    	Assert.isNotEmpty(qualification.getQualificationItemList(), "资质审核附件不能为空");
    	//执行修改操作
    	this.qualificationMapper.updateByPrimaryKeySelective(qualification);
    	//对附件表执行修改操作,先删除原来的，再添加进去
    	QualificationItemExample example = new QualificationItemExample();
    	example.createCriteria().andQualificationReviewIdEqualTo(qualification.getId())
    		.andCompanyTypeEqualTo(Constant.NO_INT);
    	this.qualificationItemService.deleteByExample(example);
    	//添加
    	qualification.getQualificationItemList().forEach(item ->{
    		Assert.isNotBlank(item.getPurFile(), "采购方上传附件不能为空");
    		if (item.getPurFile().length() > 3900) {
    			throw new CommonException("上传的附件个数超长，请减少附件个数后再保存");
    		}
    		// 资质审核表ID
    		item.setQualificationReviewId(qualification.getId());
    		// 采购方上传时间
    		item.setPurUploadTime(new Date());
    		// 创建时间
    		item.setCreateTime(new Date());
    		//更新时间
    		item.setUpdateTime(new Date());
    		//属于采购商上传附件
    		item.setCompanyType(Constant.NO_INT);
    		//执行保存操作
    		this.qualificationItemService.addObj(item);
    	});
	}
    
    @CacheEvict(value={"qualification"}, allEntries=true)
    @Override
	public void supModifyObj(Qualification qualification) {
		if (DocumentStatusEnum.WAIT_UPLOAD.getCode().equals(qualification.getDocumentStatus()) ||
				DocumentStatusEnum.REJECT.getCode().equals(qualification.getDocumentStatus())) {
			
			Qualification quali = this.queryObjById(qualification.getId());
			quali.setSupExplain(qualification.getSupExplain());
			this.qualificationMapper.updateByPrimaryKeySelective(quali);
			for( QualificationItem item : qualification.getQualificationItemList() ){
				//如果是采购商上传的附件，跳出循环
				if (Constant.NO_INT == item.getCompanyType()) {
					continue;
				}
				//删除原来的数据
				QualificationItemExample example = new QualificationItemExample();
				example.createCriteria().andQualificationReviewIdEqualTo(qualification.getId())
					.andCompanyTypeEqualTo(Constant.YES_INT);
				this.qualificationItemService.deleteByExample(example);
				if (StringUtils.isNotBlank(item.getSupFile()) && item.getSupFile().length() > 3900) {
					throw new CommonException("上传的附件个数超长，请减少附件个数后再保存");
				}
				//设置供应商上传附件的值并保存
				item.setCompanyType(Constant.YES_INT);
				item.setQualificationReviewId(qualification.getId());
				item.setSupUploadTime(new Date());
				item.setCreateTime(new Date());
				item.setUpdateTime(new Date());
				this.qualificationItemService.addObj(item);
			}
		} else {
			throw new CommonException("所选单据不是待上传或驳回状态，不能执行修改操作");
		}
		
	}
    
    @Cacheable(value="qualification", keyGenerator="redisKeyGenerator")
    @Override
    public Qualification queryObjById(String id) {
        return this.qualificationMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value="qualification", keyGenerator="redisKeyGenerator")
    @Override
    public List<Qualification> queryAllObjByExample(QualificationExample example) {
        return this.qualificationMapper.selectByExample(example);
    }

    @Cacheable(value="qualification", keyGenerator="redisKeyGenerator")
    @Override
    public PageView<Qualification> queryObjByPage(QualificationExample example) {
        PageView<Qualification> pageView = example.getPageView();
        pageView.setQueryResult(this.qualificationMapper.selectByExampleByPage(example));
        return pageView;
    }
    
    @CacheEvict(value={"qualification"}, allEntries=true)
	@Override
	public void purSendToSup(List<String> ids) {
		//只有未发送的单据才能发送
		QualificationExample example = new QualificationExample();
		example.createCriteria().andIdIn(ids)
			.andIsEnableEqualTo(Constant.YES_INT)
			.andDocumentStatusNotEqualTo(DocumentStatusEnum.UNSENT.getCode());
		if (qualificationMapper.countByExample(example) > 0) {
			throw new CommonException("所选单据不全是未发送单据，不能执行发送操作");
		}
		//发送成功后单据状态改为待上传
		Qualification qualification = new Qualification();
		qualification.setDocumentStatus(DocumentStatusEnum.WAIT_UPLOAD.getCode());
		qualification.setUpdateTime(new Date());
		qualification.setBeforeDocumentStatus(DocumentStatusEnum.UNSENT.getCode());
		example.clear();
		example.createCriteria().andIdIn(ids)
			.andIsEnableEqualTo(Constant.YES_INT);
		this.qualificationMapper.updateByExampleSelective(qualification, example);
		// 采购商给供应商发送单据成功后，向供应商发送消息
		List<Qualification> qualificationList = this.queryAllObjByExample(example);
		Assert.isNotEmpty(qualificationList, "采购商向供应商发送单据是根据单据ID查询出的数据为空");
		this.defaultThreadPool.execute( new Runnable(){
			@Override
			public void run() {
				qualificationList.forEach(quali ->{
					QualificationImTemplet templet = constructImInfo(1,"已发送",quali);
					sendMessagesToSup(templet,quali);
				});
			}
		});
		
	}
    
	@Transactional
    @CacheEvict(value={"qualification"}, allEntries=true)
	@Override
	public void supSendToPur(List<String> ids) {
		//只有待上传和驳回的单据才能发送
    	List<String> list = new ArrayList<>();
		list.add(DocumentStatusEnum.WAIT_REVIEW.getCode());
		list.add(DocumentStatusEnum.THROUGH_AUDIT.getCode());
		list.add(DocumentStatusEnum.TERMINATE_AUDIT.getCode());
		list.add(DocumentStatusEnum.SUPPLEMENTARY_INFORMATION.getCode());
		QualificationExample example = new QualificationExample();
		example.createCriteria().andIdIn(ids)
			.andIsEnableEqualTo(Constant.YES_INT)
			.andDocumentStatusIn(list);
		if (qualificationMapper.countByExample(example) > 0) { 
			throw new CommonException("所选单据包含待审核、通过审核、终止审核或已补资料单据，不能执行发送操作");
		}
    	
    	//判断供应商是否上传附件
		QualificationItemExample itemExample = new QualificationItemExample();
		itemExample.createCriteria().andQualificationReviewIdIn(ids)
			.andCompanyTypeEqualTo(Constant.YES_INT);
		List<QualificationItem> itemList = this.qualificationItemService.queryAllObjByExample(itemExample);
		Assert.isNotEmpty(itemList, "发送的单据有供应商附件没有上传");
		itemList.forEach(item ->{
			if (StringUtils.isBlank(item.getSupFile()) || "[]".equals(item.getSupFile())) {
				throw new CommonException("发送的单据中有的供应商附件没有上传");
			}
		});
		//修改单据状态，单据状态由待上传变为待审核，驳回变为已补资料
		example.clear();
		example.createCriteria().andIdIn(ids)
			.andIsEnableEqualTo(Constant.YES_INT);
		List<Qualification> qualificationList = this.queryAllObjByExample(example);
		Assert.isNotEmpty(qualificationList, "根据单据ID查询的单据数据为空");
		qualificationList.forEach(qualification ->{
			if (DocumentStatusEnum.WAIT_UPLOAD.getCode().equals(qualification.getDocumentStatus())) {
				qualification.setDocumentStatus(DocumentStatusEnum.WAIT_REVIEW.getCode());
			} else {
				qualification.setDocumentStatus(DocumentStatusEnum.SUPPLEMENTARY_INFORMATION.getCode());
			}
			this.qualificationMapper.updateByPrimaryKeySelective(qualification);
		});
    	
    	// 供应商向采购商发送消息
		this.defaultThreadPool.execute( new Runnable(){
			@Override
			public void run() {
				qualificationList.forEach(qualification ->{
					QualificationImTemplet templet = constructImInfo(2,"已发送",qualification);
					sendMessagesToPur(templet,qualification);
				});
			}
		});
	}
    
	@Transactional
    @CacheEvict(value={"qualification"}, allEntries=true)
	@Override
	public void checkObjByIds(List<String> ids) {
		//只有待审核、已补资料的状态才能审核
    	List<String> list = new ArrayList<>();
		list.add(DocumentStatusEnum.UNSENT.getCode());
		list.add(DocumentStatusEnum.THROUGH_AUDIT.getCode());
		list.add(DocumentStatusEnum.TERMINATE_AUDIT.getCode());
		list.add(DocumentStatusEnum.REJECT.getCode());
		list.add(DocumentStatusEnum.WAIT_UPLOAD.getCode());
		QualificationExample example = new QualificationExample();
		example.createCriteria().andIdIn(ids)
			.andIsEnableEqualTo(Constant.YES_INT)
			.andDocumentStatusIn(list);
		if (qualificationMapper.countByExample(example) > 0) {
			throw new CommonException("所选单据不全是待审核和已补资料单据，不能执行审核操作");
		}
		example.clear();
		// 查询单据信息
		example.createCriteria().andIdIn(ids)
			.andIsEnableEqualTo(Constant.YES_INT);
		List<Qualification> quaList = this.qualificationMapper.selectByExample(example);
		//修改状态为审核通过
		quaList.stream().forEach(qualifi ->{
			qualifi.setBeforeDocumentStatus(qualifi.getDocumentStatus());
			qualifi.setDocumentStatus(DocumentStatusEnum.THROUGH_AUDIT.getCode());
			this.qualificationMapper.updateByPrimaryKeySelective(qualifi);
		});
    	// 审核通过后修改准入流程中的状态为通过
    	List<Qualification> qualificationList = this.queryAllObjByExample(example);
    	Assert.isNotEmpty(qualificationList, "根据ID查询出的数据为空");
    	qualificationList.forEach(qualifications ->{
    		Process process = this.processService.queryObjByCompanyId(qualifications.getSupCompanyId());
    		process.setQualificationAudit(ReviewResultEnum.ADOPT.getCode());
    		process.setUpdateTime(new Date());
    		this.processService.modifyObj(process);
    	});
    	// 审核通过后给供应商发送消息
    	this.defaultThreadPool.execute( new Runnable(){
			@Override
			public void run() {
				qualificationList.forEach(quali ->{
					QualificationImTemplet templet = constructImInfo(1,"审核通过",quali);
					sendMessagesToSup(templet,quali);
				});
			}
    	});
	}
    
    @CacheEvict(value={"qualification"}, allEntries=true)
	@Override
	public void rejectObjByIds(Qualification qualification) {
    	//只有待审核、已补资料的状态才能驳回
    	List<String> list = new ArrayList<>();
		list.add(DocumentStatusEnum.UNSENT.getCode());
		list.add(DocumentStatusEnum.THROUGH_AUDIT.getCode());
		list.add(DocumentStatusEnum.TERMINATE_AUDIT.getCode());
		list.add(DocumentStatusEnum.REJECT.getCode());
		list.add(DocumentStatusEnum.WAIT_UPLOAD.getCode());
		QualificationExample example = new QualificationExample();
		example.createCriteria().andIdEqualTo(qualification.getId())
			.andIsEnableEqualTo(Constant.YES_INT)
			.andDocumentStatusIn(list);
		if (qualificationMapper.countByExample(example) > 0) {
			throw new CommonException("所选单据不是待审核和已补资料单据，不能执行驳回操作");
		}
    	// 根据ID查询单据信息
		Qualification qualifi = this.qualificationMapper.selectByPrimaryKey(qualification.getId());
    	// 修改状态为驳回
		qualification.setBeforeDocumentStatus(qualifi.getDocumentStatus());
		qualification.setDocumentStatus(DocumentStatusEnum.REJECT.getCode());
    	this.qualificationMapper.updateByPrimaryKeySelective(qualification);
    	
    	// 驳回后给供应商发送消息
    	List<Qualification> qualificationList = this.queryAllObjByExample(example);
    	Assert.isNotEmpty(qualificationList, "驳回是根据单据ID查询出的单据数据为空");
		this.defaultThreadPool.execute( new Runnable(){
			@Override
			public void run() {
				qualificationList.forEach(quali ->{
					QualificationImTemplet templet = constructImInfo(1,"已驳回",quali);
					sendMessagesToSup(templet,quali);
				});
			}
		});
	}
    
    @Transactional
    @CacheEvict(value={"qualification"}, allEntries=true)
    @Override
	public void revokeObjByIds(List<String> ids) {
		// 未发送、待审核、已补资料的单据，采购商不需要撤回到上一次的单据状态
    	List<String> list = new ArrayList<String>();
    	list.add(DocumentStatusEnum.SUPPLEMENTARY_INFORMATION.getCode());
    	list.add(DocumentStatusEnum.UNSENT.getCode());
    	list.add(DocumentStatusEnum.WAIT_REVIEW.getCode());
    	QualificationExample example = new QualificationExample();
    	example.createCriteria().andIdIn(ids)
    			.andIsEnableEqualTo(Constant.YES_INT)
    			.andDocumentStatusIn(list);
    	if (this.qualificationMapper.countByExample(example) > 0) {
    		throw new CommonException("所选单据中包含未发送、待审核、已补资料的单据，操作失败");
    	}
    	
    	// 查询单据信息
    	example.clear();
    	example.createCriteria().andIdIn(ids)
    			.andIsEnableEqualTo(Constant.YES_INT);
    	List<Qualification> quaList = this.qualificationMapper.selectByExample(example);
    	quaList.stream().forEach(qualifi ->{
    		// 如果单据的上次状态为空，就不能再执行撤回操作
    		if (StringUtils.isBlank(qualifi.getBeforeDocumentStatus())) {
    			throw new CommonException("单据号为："+ qualifi.getDocumentNo() +"的单据已执行一次撤回操作，不能再重复操作");
    		}
    		/**
        	 * 1.如果供应商已发送认证审批，不能撤回操作
        	 * 2.如果准入流程是资质审核->现场考核，并且现场考察已开始，不能再执行撤回操作
        	 * 3.如果单据状态为终止审核，并且此供应商重新创建了单据，那么终止审核状态的单据不能再撤回操作
        	 */
    		Process process = this.processService.queryObjByCompanyId(qualifi.getSupCompanyId());
    		if (process.getIsSend() != AuthenticationEnum.UNCERTIFIED.getCode()) {
    			throw new CommonException("供应商:"+ qualifi.getSupCompanySrmCode() +"已发送审批认证，不能再执行撤回操作");
    		}
    		if (process.getAccessProcess().equals(AccessProcessEnum.QUALIFICATION_AND_SCENE_REVIEW.getCode()) && 
    				StringUtils.isNotBlank(process.getSceneInvestigate())) {
    			throw new CommonException("供应商:"+ qualifi.getSupCompanySrmCode() +"的现场考察已开始，不能再执行撤回操作");
    		}
    		if (qualifi.getDocumentStatus().equals(DocumentStatusEnum.TERMINATE_AUDIT.getCode())) {
    			example.clear();
    			example.createCriteria()
    					.andDocumentStatusNotEqualTo(DocumentStatusEnum.TERMINATE_AUDIT.getCode())
    					.andIsEnableEqualTo(Constant.YES_INT)
    					.andSupCompanySrmCodeEqualTo(qualifi.getSupCompanySrmCode());
    			if (this.qualificationMapper.countByExample(example) > 0) {
    				throw new CommonException("供应商："+ qualifi.getSupCompanySrmCode() +"已重新创建了单据，不能再执行撤回操作");
    			}
    		}
    		
    		// 如果单据的状态是通过审核、终止审核的单据，撤回操作时准入流程的状态也应撤回
    		if (qualifi.getDocumentStatus().equals(DocumentStatusEnum.THROUGH_AUDIT.getCode()) ||
    				qualifi.getDocumentStatus().equals(DocumentStatusEnum.TERMINATE_AUDIT.getCode())) {
    			process.setQualificationAudit(ReviewResultEnum.AUDIT.getCode());
    			process.setUpdateTime(new Date());
    			this.processService.modifyObj(process);
    		}
    		// 单据状态返回到上次的状态
    		qualifi.setDocumentStatus(qualifi.getBeforeDocumentStatus());
    		qualifi.setBeforeDocumentStatus("");
    		this.qualificationMapper.updateByPrimaryKeySelective(qualifi);
    	});
		
	}
    
    @Transactional
    @CacheEvict(value={"qualification"}, allEntries=true)
	@Override
	public void terminationAuditObjByIds(List<String> ids) {
    	//只有待审核、已补资料的状态才能终止审核
    	List<String> list = new ArrayList<>();
		list.add(DocumentStatusEnum.UNSENT.getCode());
		list.add(DocumentStatusEnum.THROUGH_AUDIT.getCode());
		list.add(DocumentStatusEnum.TERMINATE_AUDIT.getCode());
		list.add(DocumentStatusEnum.REJECT.getCode());
		list.add(DocumentStatusEnum.WAIT_UPLOAD.getCode());
		QualificationExample example = new QualificationExample();
		example.createCriteria().andIdIn(ids)
			.andIsEnableEqualTo(Constant.YES_INT)
			.andDocumentStatusIn(list);
		if (qualificationMapper.countByExample(example) > 0) {
			throw new CommonException("所选单据不全是待审核和已补资料单据，不能执行终止审核操作");
		}
		example.clear();
		// 查询单据信息
		example.createCriteria().andIdIn(ids)
			.andIsEnableEqualTo(Constant.YES_INT);
		List<Qualification> quaList = this.qualificationMapper.selectByExample(example);
		// 修改状态为终止审核
		quaList.stream().forEach(qualifi ->{
			qualifi.setBeforeDocumentStatus(qualifi.getDocumentStatus());
			qualifi.setDocumentStatus(DocumentStatusEnum.TERMINATE_AUDIT.getCode());
			this.qualificationMapper.updateByPrimaryKeySelective(qualifi);
		});
    	// 准入流程的状态改为终止
		List<Qualification> qualificationList = this.queryAllObjByExample(example);
		Assert.isNotEmpty(qualificationList, "根据单据ID查询的单据为空");
		qualificationList.forEach(qualifications ->{
			Process process = this.processService.queryObjByCompanyId(qualifications.getSupCompanyId());
			process.setQualificationAudit(ReviewResultEnum.TERMINATION.getCode());
			process.setUpdateTime(new Date());
			this.processService.modifyObj(process);
		});
    	// 终止审核后给供应商发送消息
		this.defaultThreadPool.execute( new Runnable(){
			@Override
			public void run() {
				qualificationList.forEach(quali ->{
					QualificationImTemplet templet = constructImInfo(1,"终止开发",quali);
					sendMessagesToSup(templet,quali);
				});
			}
		});
	}
    
    /**
     * 采购商发消息给供应商
     * @param templet
     * @param qualification
     */
    protected void sendMessagesToSup(QualificationImTemplet templet, Qualification qualification) {
    	Message<QualificationImTemplet> message = Message.init(templet) // 发送数据
				.setBusinessTypeCode(IM_CODE) // 业务编码
				.setCompanyCode(qualification.getPurCompanySrmCode()) // 发送公司编码
				.setMsgLevel(MessageLevelEnum.HIGH) // 消息等级
				.setSenderId(qualification.getPurCompanyUserId()) // 发送方用户ID
				.addReceiverId(qualification.getSupCompanyUserId()); // 接收方用户ID
		MessageSendUtils.sendMessage(message);
		
	}
    
    /**
     * 供应商给采购商发送消息
     * @param templet
     * @param qualification
     */
    protected void sendMessagesToPur(QualificationImTemplet templet, Qualification qualification) {
    	Message<QualificationImTemplet> message = Message.init(templet) // 发送数据
				.setBusinessTypeCode(IM_CODE) // 业务编码
				.setCompanyCode(qualification.getSupCompanySrmCode()) // 发送人公司编码
				.setMsgLevel(MessageLevelEnum.HIGH) // 消息等级
				.setSenderId(qualification.getSupCompanyUserId()) // 发送人用户ID
				.addReceiverId(qualification.getPurCompanyUserId()); // 接收人用户ID
		MessageSendUtils.sendMessage(message);
	}

	protected QualificationImTemplet constructImInfo(int role, String operate, Qualification qualification) {
		Assert.isNotNull(qualification, "发送的数据为空");
		QualificationImTemplet qualificationImTemplet = new QualificationImTemplet();
		// 设置角色信息，1为采购商，其他为供应商
		if (1 == role) {
			qualificationImTemplet.setRole("采购商");
			if (StringUtils.isNotBlank(qualification.getPurCompanyName())) {
				qualificationImTemplet.setCompanyName(qualification.getPurCompanyName());
			} else {
				qualificationImTemplet.setCompanyName("");
			}
		} else {
			qualificationImTemplet.setRole("供应商");
			if (StringUtils.isNotBlank(qualification.getSupCompanyName())) {
				qualificationImTemplet.setCompanyName(qualification.getSupCompanyName());
			} else {
				qualificationImTemplet.setCompanyName("");
			}
		}
		// 设置操作
		Assert.isNotBlank(operate, "操作字段为空，不能发送消息");
		qualificationImTemplet.setOperate(operate);
		// 设置其他信息
		qualificationImTemplet.setOtherInfo("了资质审核单据，单据号为：");
		// 设置单据号
		qualificationImTemplet.setBillList(qualification.getDocumentNo());
		return qualificationImTemplet;
	}

}