package com.els.base.certification.suppliercategory.service.impl;

import com.els.base.certification.suppliercategory.dao.SupplierCategoryMapper;
import com.els.base.certification.suppliercategory.entity.SupplierCategory;
import com.els.base.certification.suppliercategory.entity.SupplierCategoryExample;
import com.els.base.certification.suppliercategory.service.SupplierCategoryService;
import com.els.base.company.dao.CompanyMapper;
import com.els.base.company.entity.Company;
import com.els.base.company.service.CompanyService;
import com.els.base.core.entity.PageView;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;

import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

@Service("defaultSupplierCategoryService")
public class SupplierCategoryServiceImpl implements SupplierCategoryService {
    @Resource
    protected SupplierCategoryMapper supplierCategoryMapper;
    @Resource
    private CompanyMapper companyMapper;
    @Resource
    private CompanyService companyServic;

    @CacheEvict(value = {"supplierCategory"}, allEntries = true)
    @Override
    public void addObj(SupplierCategory t) {
        SupplierCategoryExample example = new SupplierCategoryExample();
        example.createCriteria().andCategoryCodeEqualTo(t.getCategoryCode());
        if (this.supplierCategoryMapper.countByExample(example) > 0) {
            throw new CommonException("新增的供应商分类编码不能相同");
        }
        this.supplierCategoryMapper.insertSelective(t);
    }

    @CacheEvict(value = {"supplierCategory"}, allEntries = true)
    @Override
    public void deleteObjById(String id) {
        this.supplierCategoryMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value = {"supplierCategory"}, allEntries = true)
    @Override
    public void modifyObj(SupplierCategory t) {
        if (StringUtils.isBlank(t.getId())) {
            throw new NullPointerException("id 为空，无法更新");
        }
        this.supplierCategoryMapper.updateByPrimaryKeySelective(t);
    }

    @Cacheable(value = "supplierCategory", keyGenerator = "redisKeyGenerator")
    @Override
    public SupplierCategory queryObjById(String id) {
        return this.supplierCategoryMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value = "supplierCategory", keyGenerator = "redisKeyGenerator")
    @Override
    public List<SupplierCategory> queryAllObjByExample(SupplierCategoryExample example) {
        return this.supplierCategoryMapper.selectByExample(example);
    }

    @Cacheable(value = "supplierCategory", keyGenerator = "redisKeyGenerator")
    @Override
    public PageView<SupplierCategory> queryObjByPage(SupplierCategoryExample example) {
        PageView<SupplierCategory> pageView = example.getPageView();
        pageView.setQueryResult(this.supplierCategoryMapper.selectByExampleByPage(example));
        return pageView;
    }

    @CacheEvict(value = {"supplierCategory"}, allEntries = true)
    @Override
    public void deleteCascadedMaterialCategoryInfo(String parentId) {
        if (StringUtils.isBlank(parentId)) {
            throw new CommonException("上一级传的id为空!");
        }
        // 先删除
        this.supplierCategoryMapper.deleteByPrimaryKey(parentId);

        SupplierCategoryExample materialCategoryExample = new SupplierCategoryExample();
        materialCategoryExample.createCriteria().andParentIdEqualTo(parentId).andIsEnableEqualTo(Constant.YES_INT);
        List<SupplierCategory> list = this.supplierCategoryMapper.selectByExample(materialCategoryExample);
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        for (SupplierCategory supplierCategory : list) {
            this.deleteCascadedMaterialCategoryInfo(supplierCategory.getId());
        }
        return;
    }

    @Cacheable(value = "supplierCategory", keyGenerator = "redisKeyGenerator")
    @Override
    public List<SupplierCategory> queryByParentId(String parentId) {
        if (StringUtils.isBlank(parentId)) {
            return null;
        }
        SupplierCategoryExample example = new SupplierCategoryExample();
        example.createCriteria().andParentIdEqualTo(parentId).andIsEnableEqualTo(Constant.YES_INT);
        List<SupplierCategory> list = this.supplierCategoryMapper.selectByExample(example);

        if (CollectionUtils.isEmpty(list)) {
            return null;
        }
        for (SupplierCategory materialCategory : list) {
            materialCategory.setSupplierCategoryList(this.queryByParentId(materialCategory.getId()));
        }
        return list;
    }


    @CacheEvict(value = {"supplierCategory"}, allEntries = true)
    @Override
    public void addSupCategory(Company c, String supplierCategoryId) {
        if (StringUtils.isBlank(c.getId())) {
            throw new NullPointerException("id 为空，供应商不存在无法更新");
        }
        SupplierCategory supplierCategory = this.supplierCategoryMapper.selectByPrimaryKey(supplierCategoryId);

        if (supplierCategory == null) {
            throw new CommonException("该分类不存在，无法保存");
        }
        c.setCategoryName(supplierCategory.getCompanyCategory());
        c.setCategoryCode(supplierCategory.getCategoryCode());
        this.companyServic.modifyObj(c);
    }

    @CacheEvict(value = {"supplierCategory"}, allEntries = true)
    @Override
    public void addSupCategory(Company company, String supplierCategoryId, String type) {
        switch (type) {
            //默认添加供应商分类
            case "0":
                this.addSupCategory(company, supplierCategoryId);
                break;

            //添加供应商，产品重要度： (A B C)
            case "1":
                this.addProductImportanceName(company, supplierCategoryId);
                break;

            //添加供应商，合作关系
            case "2":
                this.addSymbiosis(company, supplierCategoryId);
                break;

            //添加供应商， 关键程度：   关键、非关键
            case "3":
                this.addCriticalityName(company, supplierCategoryId);
                break;
            //默认添加供应商分类
            default:
                this.addSupCategory(company, supplierCategoryId);
                break;
        }
    }

    private void addCriticalityName(Company company, String supplierCategoryId) {
        if (StringUtils.isBlank(company.getId())) {
            throw new NullPointerException("id 为空，供应商不存在无法更新");
        }
        SupplierCategory supplierCategory = this.supplierCategoryMapper.selectByPrimaryKey(supplierCategoryId);
        if (supplierCategory == null) {
            throw new CommonException("该分类不存在，无法保存");
        }
        company.setCriticalityId(supplierCategoryId);
        company.setCriticalityName(supplierCategory.getCompanyCategory());
        this.companyServic.modifyObj(company);
    }

    private void addSymbiosis(Company company, String supplierCategoryId) {
        if (StringUtils.isBlank(company.getId())) {
            throw new NullPointerException("id 为空，供应商不存在无法更新");
        }
        SupplierCategory supplierCategory = this.supplierCategoryMapper.selectByPrimaryKey(supplierCategoryId);
        if (supplierCategory == null) {
            throw new CommonException("该分类不存在，无法保存");
        }
        company.setSymbiosisId(supplierCategoryId);
        company.setSymbiosis(supplierCategory.getCompanyCategory());
        this.companyServic.modifyObj(company);
    }

    private void addProductImportanceName(Company company, String supplierCategoryId) {
        if (StringUtils.isBlank(company.getId())) {
            throw new NullPointerException("id 为空，供应商不存在无法更新");
        }
        SupplierCategory supplierCategory = this.supplierCategoryMapper.selectByPrimaryKey(supplierCategoryId);
        if (supplierCategory == null) {
            throw new CommonException("该分类不存在，无法保存");
        }
        company.setProductImportanceId(supplierCategoryId);
        company.setProductImportanceName(supplierCategory.getCompanyCategory());
        this.companyServic.modifyObj(company);
    }

    @CacheEvict(value = {"supplierCategory"}, allEntries = true)
    @Override
    public void deleteByExample(SupplierCategoryExample example) {
        Assert.isNotEmpty(example.getOredCriteria(), "不能进行全表删除！");
        this.supplierCategoryMapper.deleteByExample(example);
    }
}