package com.els.base.certification.process.service.impl;

import com.els.base.certification.process.dao.ProcessMapper;
import com.els.base.certification.process.entity.Process;
import com.els.base.certification.process.entity.ProcessExample;
import com.els.base.certification.process.service.ProcessService;
import com.els.base.certification.process.util.AccessProcessEnum;
import com.els.base.certification.process.util.AuthenticationEnum;
import com.els.base.company.entity.Company;
import com.els.base.core.entity.PageView;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.utils.excel.DateConverter;
import com.els.base.utils.excel.ExcelUtils;
import com.els.base.utils.excel.TitleAndModelKey;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;

@Service("defaultProcessService")
public class ProcessServiceImpl implements ProcessService {
    @Resource
    protected ProcessMapper processMapper;

    @CacheEvict(value={"process"}, allEntries=true)
    @Override
    public void addObj(Process t) {
    	// 根据选择的准入流程，判断模版ID不能为空
    	Assert.isNotBlank(t.getProcessId(), "准入流程ID不能为空");
    	t.setIsEnable(Constant.YES_INT);
    	// 供应商未发送审核
    	t.setIsSend(AuthenticationEnum.UNCERTIFIED.getCode());
    	t.setCreateTime(new Date());
    	t.setUpdateTime(new Date());
    	// 根据供应商ID查询流程信息，看原来有没有指定流程，如果有，删除原先的再插入进去
    	Process process = this.queryObjByCompanyId(t.getCompanyId());
    	if (null == process) {
    		this.processMapper.insertSelective(t);
    	} else {
    		// 如果原来有指定流程，看资质审核、现场考察是否在进行，在进行就不能再修改流程了
    		if (StringUtils.isNotBlank(process.getQualificationAudit()) || 
    				StringUtils.isNotBlank(process.getSceneInvestigate())) {
    			throw new CommonException("资质审核或现场考察已在进行中，不能再修改流程");
    		}
    		// 如果流程是不需要审核，并且供应商已发送审批，此时流程不能修改
    		if (AccessProcessEnum.NOT_REVIEW.getCode().equals(process.getAccessProcess()) && 
    				AuthenticationEnum.UNCERTIFIED.getCode() != process.getIsSend()) {
    			throw new CommonException("供应商已在认证中，不能修改流程");
    		}
    		// 删除原来的流程,再添加
    		this.deleteObjById(process.getId());
    		this.processMapper.insertSelective(t);
    	}
    }

    @CacheEvict(value={"process"}, allEntries=true)
    @Override
    public void deleteObjById(String id) {
        this.processMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value={"process"}, allEntries=true)
    @Override
    public void modifyObj(Process t) {
        if (StringUtils.isBlank(t.getId())) {
            throw new NullPointerException("id 为空，无法更新");
        }
        this.processMapper.updateByPrimaryKeySelective(t);
    }
    
    @CacheEvict(value={"process"}, allEntries=true)
    @Override
	public void updateById(Process process) {
    	if (StringUtils.isBlank(process.getId())) {
            throw new NullPointerException("id 为空，无法更新");
        }
    	this.processMapper.updateByPrimaryKey(process);
	}

    @Cacheable(value="process", keyGenerator="redisKeyGenerator")
    @Override
    public Process queryObjById(String id) {
        return this.processMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value="process", keyGenerator="redisKeyGenerator")
    @Override
    public List<Process> queryAllObjByExample(ProcessExample example) {
        return this.processMapper.selectByExample(example);
    }

    @Cacheable(value="process", keyGenerator="redisKeyGenerator")
    @Override
    public PageView<Process> queryObjByPage(ProcessExample example) {
        PageView<Process> pageView = example.getPageView();
        pageView.setQueryResult(this.processMapper.selectByExampleByPage(example));
        return pageView;
    }
    
    @Cacheable(value="process", keyGenerator="redisKeyGenerator")
	@Override
	public Process queryObjByCompanyId(String companyId) {
		Assert.isNotBlank(companyId, "供应商的ID不能为空");
		ProcessExample example = new ProcessExample();
		example.createCriteria().andCompanyIdEqualTo(companyId);
		List<Process> list = this.processMapper.selectByExample(example);
		return CollectionUtils.isNotEmpty(list) ? list.get(0) : null;
	}
    
	@Override
	public PageView<Company> queryAllCompnayByPage(String purCompanyId, String roleEnum, Map<String, Object> queryMap) {
    	PageView<Company> pageView = new PageView<>(MapUtils.getIntValue(queryMap, "pageNo"),
				MapUtils.getIntValue(queryMap, "pageSize"));
    	queryMap.put("offset", (MapUtils.getIntValue(queryMap, "pageNo") - 1) * MapUtils.getIntValue(queryMap, "pageSize"));
		List<Company> list = this.processMapper.selectCompanyAndProcessInfoByPage(purCompanyId, roleEnum, queryMap);
		pageView.setQueryResult(list);
		pageView.setRowCount(this.processMapper.selectCompanyAndProcessInfoCount(purCompanyId, roleEnum, queryMap));
		return pageView;
	}
    
    @CacheEvict(value={"process"}, allEntries=true)
	@Override
	public int countByExample(ProcessExample example) {
		return this.processMapper.countByExample(example);
	}
    
    @CacheEvict(value={"process"}, allEntries=true)
	@Override
	public List<TitleAndModelKey> createExcelHeaderForPur() {
    	DateConverter dateConverter =new DateConverter();
		dateConverter.setDateFormat("yyyy-MM-dd");
		List<TitleAndModelKey> titleAndModelKeys = new ArrayList<TitleAndModelKey>();
		titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("SRM供应商编码", "companyCode"));
//    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("SAP供应商编码", "companySapCode"));
    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("简称", "companyName"));
    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("认证状态", "isSendName"));
    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("准入流程名称", "processName"));
    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("准入流程", "accessProcessName"));
    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("资质考察", "qualificationAuditName"));
    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("现场考察", "sceneInvestigateName"));
    	
    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("产品大类1", "secondCompanyTypeDesc1"));
    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("产品大类2", "secondCompanyTypeDesc2"));
    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("产品大类3", "secondCompanyTypeDesc3"));
    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("产品大类4", "secondCompanyTypeDesc4"));
    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("产品大类5", "secondCompanyTypeDesc5"));
    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("供应商服务类", "firstCompanyTypeDesc"));
//    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("供应商风险类", "businessTypeDesc"));
//    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("供应商资查类", "classificationThreeDesc"));
    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("英文简称", "companyEnglishAbbreviation"));
    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("供应商全称", "companyFullName"));
    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("英文全称", "englishCompanyName"));
//    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("代理品牌", "actingBrand"));
    	
    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("ISO9001体系证书", "ISO9001FileName"));
    	TitleAndModelKey ISO9001EndTime=ExcelUtils.createTitleAndModelKey("ISO9001体系证书有效期", "ISO9001EndTime");
    	ISO9001EndTime.setToObjConverter(dateConverter);
    	ISO9001EndTime.setToStrConverter(dateConverter);
    	titleAndModelKeys.add(ISO9001EndTime);
    	
    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("IATF16949体系证书", "IATF16949FileName"));
    	TitleAndModelKey IATF16949EndTime=ExcelUtils.createTitleAndModelKey("IATF16949体系证书有效期", "IATF16949EndTime");
    	IATF16949EndTime.setToObjConverter(dateConverter);
    	IATF16949EndTime.setToStrConverter(dateConverter);
    	titleAndModelKeys.add(IATF16949EndTime);
    	
    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("ISO14000体系证书", "ISO14000FileName"));
    	TitleAndModelKey ISO14000EndTime=ExcelUtils.createTitleAndModelKey("ISO14000证书有效期", "ISO14000EndTime");
    	ISO14000EndTime.setToObjConverter(dateConverter);
    	ISO14000EndTime.setToStrConverter(dateConverter);
    	titleAndModelKeys.add(ISO14000EndTime);
    	
    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("负责人", "contacts"));
    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("E-MAIL", "email"));
    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("TEL(一)", "telephone"));
    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("FAXNO", "fax"));
    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("联系地址(中文)", "address"));
    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("英文地址", "englishAddress"));
    	titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("经营范围", "businessScope"));
    	
    	TitleAndModelKey createTime=ExcelUtils.createTitleAndModelKey("注册时间", "createTime");
    	createTime.setToObjConverter(dateConverter);
    	createTime.setToStrConverter(dateConverter);
    	titleAndModelKeys.add(createTime);
    	
		return titleAndModelKeys;
	}

}