package com.els.base.certification.qualification.web.controller;


import com.els.base.auth.utils.SpringSecurityUtils;
import com.els.base.certification.qualification.entity.Qualification;
import com.els.base.certification.qualification.entity.QualificationExample;
import com.els.base.certification.qualification.entity.QualificationItem;
import com.els.base.certification.qualification.entity.QualificationItemExample;
import com.els.base.certification.qualification.service.QualificationItemService;
import com.els.base.certification.qualification.service.QualificationService;
import com.els.base.certification.qualification.util.DocumentStatusEnum;
import com.els.base.company.utils.CompanyUtils;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.entity.user.User;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.query.QueryParamWapper;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;

import java.util.List;
import javax.annotation.Resource;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

@Api(value="供应商准入-资质审核单")
@Controller
@RequestMapping("qualification")
public class QualificationController {
    @Resource
    protected QualificationService qualificationService;
    
    @Resource
    protected QualificationItemService qualificationItemService;

    @ApiOperation(httpMethod="POST", value="创建供应商准入-资质审核单")
    @RequestMapping("service/create")
    @ResponseBody
    public ResponseResult<String> create(@RequestBody Qualification qualification) {
        Assert.isNotNull(qualification, "保存的数据不能为空");
    	Assert.isNotBlank(qualification.getSupCompanySrmCode(), "供应商SRM编码不能为空");
    	//判断资质审核附件是否为空
    	Assert.isNotEmpty(qualification.getQualificationItemList(), "资质审核文件信息不能为空");
    	User user=SpringSecurityUtils.getLoginUser();
    	this.qualificationService.insert(user,qualification,CompanyUtils.currentCompany());
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod="POST", value="编辑供应商准入-资质审核单")
    @RequestMapping("service/edit")
    @ResponseBody
    public ResponseResult<String> edit(@RequestBody Qualification qualification) {
        Assert.isNotBlank(qualification.getId(), "id 为空，保存失败");
        this.qualificationService.modifyObj(qualification);
        return ResponseResult.success();
    }
    
    @ApiOperation(httpMethod="POST", value="采购方编辑供应商准入-资质审核单")
    @RequestMapping("service/purEdit")
    @ResponseBody
    public ResponseResult<String> purEdit(@RequestBody Qualification qualification) {
        Assert.isNotBlank(qualification.getId(), "id 为空，保存失败");
        Assert.isNotBlank(qualification.getSupCompanySrmCode(), "供应商SRM编码不能为空");
        this.qualificationService.purModifyObj(qualification);
        return ResponseResult.success();
    }
    
    @ApiOperation(httpMethod="POST", value="供应商修改供应商准入-资质审核单")
    @RequestMapping("service/supEdit")
    @ResponseBody
    public ResponseResult<String> supEdit(@RequestBody Qualification qualification) {
        Assert.isNotBlank(qualification.getId(), "id 为空，保存失败");
        Assert.isNotBlank(qualification.getSupCompanySrmCode(), "供应商SRM编码不能为空");
        this.qualificationService.supModifyObj(qualification);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod="POST", value="采购方批量删除供应商准入-资质审核单")
    @RequestMapping("service/deleteById")
    @ResponseBody
    public ResponseResult<String> deleteById(@RequestBody(required=true) List<String> ids) {
        Assert.isNotEmpty(ids, "传入的单据ID不能为空");
        this.qualificationService.deleteObjByIds(ids);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod="POST", value="采购商查询供应商准入-资质审核单")
    @ApiImplicitParams({ 
		@ApiImplicitParam( name = "pageNo",required = false,value = "所在页", paramType = "query", dataType = "String", defaultValue = "0" ),  
		@ApiImplicitParam( name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10" ),  
		@ApiImplicitParam( name = "wapper", required = false, value = "查询条件,属性名请参考 Qualification", paramType = "body", dataType = "QueryParamWapper" )  
	}) 
    @RequestMapping("service/pur/findByPage")
    @ResponseBody
    public ResponseResult<PageView<Qualification>> purFindByPage( 
		@RequestParam(defaultValue="0") int pageNo,  
		@RequestParam(defaultValue="10") int pageSize, 
		@RequestBody(required=false) QueryParamWapper wapper) {
        QualificationExample example = new QualificationExample();
        example.setPageView(new PageView<Qualification>(pageNo, pageSize));
        example.setOrderByClause("CREATE_TIME DESC");
        
        QualificationExample.Criteria criteria = example.createCriteria();
        //填写具体的查询条件
        criteria.andIsEnableEqualTo(Constant.YES_INT)
        	.andPurCompanyUserIdEqualTo(SpringSecurityUtils.getLoginUser().getId());
        
        if (wapper != null) {
            CriteriaUtils.addExample(example, wapper);
        }
        
        PageView<Qualification> pageData = this.qualificationService.queryObjByPage(example);
        // 查询附件信息
        pageData.getQueryResult().forEach(qualification ->{
        	QualificationItemExample itemExample = new QualificationItemExample();
        	itemExample.createCriteria()
        		.andQualificationReviewIdEqualTo(qualification.getId());
        	List<QualificationItem> itemList = this.qualificationItemService.queryAllObjByExample(itemExample);
        	qualification.setQualificationItemList(itemList);
        });
        return ResponseResult.success(pageData);
    }
    
    @ApiOperation(httpMethod="POST", value="供应商查询供应商准入-资质审核单")
    @ApiImplicitParams({ 
		@ApiImplicitParam( name = "pageNo",required = false,value = "所在页", paramType = "query", dataType = "String", defaultValue = "0" ),  
		@ApiImplicitParam( name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10" ),  
		@ApiImplicitParam( name = "wapper", required = false, value = "查询条件,属性名请参考 Qualification", paramType = "body", dataType = "QueryParamWapper" )  
	}) 
    @RequestMapping("service/sup/findByPage")
    @ResponseBody
    public ResponseResult<PageView<Qualification>> supFindByPage( 
		@RequestParam(defaultValue="0") int pageNo,  
		@RequestParam(defaultValue="10") int pageSize, 
		@RequestBody(required=false) QueryParamWapper wapper) {
        QualificationExample example = new QualificationExample();
        example.setPageView(new PageView<Qualification>(pageNo, pageSize));
        example.setOrderByClause("CREATE_TIME DESC");
        
        QualificationExample.Criteria criteria = example.createCriteria();
        //填写具体的查询条件
        criteria.andIsEnableEqualTo(Constant.YES_INT)
        	.andSupCompanyIdEqualTo(CompanyUtils.currentCompanyId())
        	.andDocumentStatusNotEqualTo(DocumentStatusEnum.UNSENT.getCode());
        
        if (wapper != null) {
            CriteriaUtils.addExample(example, wapper);
        }
        
        PageView<Qualification> pageData = this.qualificationService.queryObjByPage(example);
        pageData.getQueryResult().forEach(data ->{
        	QualificationItemExample itemExample = new QualificationItemExample();
        	itemExample.createCriteria()
        		.andQualificationReviewIdEqualTo(data.getId());
        	List<QualificationItem> itemList = this.qualificationItemService.queryAllObjByExample(itemExample);
        	data.setQualificationItemList(itemList);
        });
        return ResponseResult.success(pageData);
    }
    
    @ApiOperation(httpMethod="POST", value="采购商查看单据详情-资质审核单")
	@RequestMapping("service/pur/findQualificationById")
	@ResponseBody
	public ResponseResult<Qualification> purFindQualificationById(@RequestParam(required=true) String id){
		Assert.isNotBlank(id, "单据ID不能为空");
		Qualification qualification=this.qualificationService.queryObjById(id);
		Assert.isNotNull(qualification, "根据ID查询的数据为空");
		
		QualificationItemExample qualificationItemExample = new QualificationItemExample();
		qualificationItemExample.setOrderByClause("CREATE_TIME DESC");
		//如果单据供应商没有发送到采购方，采购商不能看到供应商上传的附件资料，发送了才能看到
		if (DocumentStatusEnum.UNSENT.getCode().equals(qualification.getDocumentStatus()) || 
				DocumentStatusEnum.WAIT_UPLOAD.getCode().equals(qualification.getDocumentStatus()) ||
				DocumentStatusEnum.REJECT.getCode().equals(qualification.getDocumentStatus())) {
			//只set采购商上传的附件数据
			qualificationItemExample.createCriteria().andQualificationReviewIdEqualTo(id)
				.andCompanyTypeEqualTo(Constant.NO_INT);
			qualification.setQualificationItemList(qualificationItemService.queryAllObjByExample(qualificationItemExample));
		} else {
			//采购商和供应商上传的附件数据都set进去
			qualificationItemExample.createCriteria().andQualificationReviewIdEqualTo(id);
			qualification.setQualificationItemList(qualificationItemService.queryAllObjByExample(qualificationItemExample));
		}
		return ResponseResult.success(qualification);
	}
    
    @ApiOperation(httpMethod="POST", value="供应商查看单据详情-资质审核单")
	@RequestMapping("service/sup/findQualificationById")
	@ResponseBody
	public ResponseResult<Qualification> supFindQualificationById(@RequestParam(required=true) String id){
		Assert.isNotBlank(id, "单据ID不能为空");
		Qualification qualification = this.qualificationService.queryObjById(id);
		Assert.isNotNull(qualification, "根据ID查询的数据为空");
		//查询附件表数据
		QualificationItemExample qualificationItemExample = new QualificationItemExample();
		qualificationItemExample.setOrderByClause("CREATE_TIME DESC");
		qualificationItemExample.createCriteria().andQualificationReviewIdEqualTo(id);
		List<QualificationItem> itemList = this.qualificationItemService.queryAllObjByExample(qualificationItemExample);
		qualification.setQualificationItemList(itemList);
		return ResponseResult.success(qualification);
	}
    
    @ApiOperation(httpMethod = "POST", value = "采购商批量发送供应商准入-资质审核单")
	@RequestMapping("service/purSendByIds")
	@ResponseBody
	public ResponseResult<String> purSendByIds(@RequestBody(required = true) List<String> ids){
		Assert.isNotEmpty(ids, "发送单据的ID不能为空");
		this.qualificationService.purSendToSup(ids);
		return ResponseResult.success();
	}
    
    @ApiOperation(httpMethod = "POST", value = "供应商批量发送供应商准入-资质审核单")
	@RequestMapping("service/supSendByIds")
	@ResponseBody
	public ResponseResult<String> supSendByIds(@RequestBody(required = true) List<String> ids){
		Assert.isNotEmpty(ids, "发送单据的ID不能为空");
		this.qualificationService.supSendToPur(ids);
		return ResponseResult.success();
	}
    
    @ApiOperation(httpMethod = "POST", value = "采购商批量审核供应商准入-资质审核单")
	@RequestMapping("service/checkByIds")
	@ResponseBody
	public ResponseResult<String> checkByIds(@RequestBody(required = true) List<String> ids){
		Assert.isNotEmpty(ids, "审核的单据ID不能为空");
		this.qualificationService.checkObjByIds(ids);
		return ResponseResult.success();
	}
    
    @ApiOperation(httpMethod = "POST", value = "采购商驳回供应商准入-资质审核单")
	@RequestMapping("service/rejectById")
	@ResponseBody
	public ResponseResult<String> rejectById(@RequestBody(required = true)Qualification qualification ){
		Assert.isNotNull(qualification, "驳回发送单据的数据为空");
    	Assert.isNotBlank(qualification.getId(), "要驳回的单据ID不能为空");
		Assert.isNotBlank(qualification.getRejectReason(), "驳回原因不能为空");
		this.qualificationService.rejectObjByIds(qualification);
		return ResponseResult.success();
	}
    
    @ApiOperation(httpMethod = "POST", value = "采购商批量终止审核供应商准入-资质审核单")
	@RequestMapping("service/terminationAuditByIds")
	@ResponseBody
	public ResponseResult<String> terminationAuditByIds(@RequestBody(required = true) List<String> ids){
		Assert.isNotEmpty(ids, "要终止的单据ID不能为空");
		this.qualificationService.terminationAuditObjByIds(ids);
		return ResponseResult.success();
	}
    
    @ApiOperation(httpMethod = "POST", value = "采购商批量撤回供应商准入-资质审核单")
	@RequestMapping("service/revokeByIds")
	@ResponseBody
	public ResponseResult<String> revokeByIds(@RequestBody(required = true) List<String> ids){
		Assert.isNotEmpty(ids, "要撤回的单据ID不能为空");
		this.qualificationService.revokeObjByIds(ids);
		return ResponseResult.success();
	}
    
}