package com.els.tso.srm.core.service.impl;

import com.els.tso.core.util.IdWorker;
import com.els.tso.srm.core.configuration.SrmSystemConfig;
import com.els.tso.srm.core.exception.BusinessException;
import com.els.tso.srm.core.model.MultipartFileParam;
import com.els.tso.srm.core.service.FileChunkUploadService;
import com.els.tso.srm.core.service.RedisCacheService;
import org.apache.commons.io.IOUtils;
import org.assertj.core.util.Sets;
import org.springframework.stereotype.Service;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.time.LocalDate;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * @author daipj
 */
@Service
public class FileChunkUploadServiceImpl implements FileChunkUploadService {

    private static final Pattern PATTERN = Pattern.compile("\\s|'");
    private final RedisCacheService redisCacheService;
    private final SrmSystemConfig systemConfig;

    public FileChunkUploadServiceImpl(RedisCacheService redisCacheService, SrmSystemConfig systemConfig) {
        this.redisCacheService = redisCacheService;
        this.systemConfig = systemConfig;
    }

    @Override
    public MultipartFileParam fileChunkUpload(MultipartFileParam multipartFileParam, String userId) throws IOException {
        //暂存redis
        redisCacheService.hset(multipartFileParam.getIdentifier(), multipartFileParam.getChunkNumber() + "", IOUtils.toByteArray(multipartFileParam.getAttachment()));
        //所有分片接受完-
        if (redisCacheService.hlen(multipartFileParam.getIdentifier()) == multipartFileParam.getTotalChunks()) {
            //取出所有缓存,合并到一起
            try (ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream()) {
                for (int i = 1; i <= multipartFileParam.getTotalChunks(); i++) {
                    //从redis读出
                    byte[] bytes = (byte[]) redisCacheService.hget(multipartFileParam.getIdentifier(), i + "");
                    byteArrayOutputStream.write(bytes, 0, bytes.length);
                }
                String fileName = multipartFileParam.getFilename();

                String fileNewName = this.getNewFileName(fileName);
                String savePath = getSavePath(userId, fileNewName);
                //整合完,写入文件
                try (FileOutputStream fileOutputStream = new FileOutputStream(savePath)) {
                    byte[] bytes = byteArrayOutputStream.toByteArray();
                    fileOutputStream.write(bytes);
                    fileOutputStream.flush();
                }
                multipartFileParam.setFileStorePath(savePath);
                //删除缓存
                redisCacheService.del(multipartFileParam.getIdentifier());

            }
        }
        multipartFileParam.setUploaded(this.getUploadedChunks(multipartFileParam.getIdentifier()));
        multipartFileParam.setAttachment(null);
        return multipartFileParam;
    }

    @Override
    public Set<Integer> getUploadedChunks(String identifier) {

        if (identifier == null) {
            throw new BusinessException("identifier不能为空");
        }
        MultipartFileParam multipartFileParam = new MultipartFileParam();
        multipartFileParam.setIdentifier(identifier);
        Map<Object, Object> map = redisCacheService.hgetAll(multipartFileParam.getIdentifier());
        if (map != null) {
            return map.keySet()
                .stream().map((key) -> Integer.parseInt(key.toString())).collect(Collectors.toSet());
        }
        return Sets.newHashSet();
    }


    private String getFileExt(String fileName) {

        return fileName.substring(fileName.lastIndexOf(".") + 1).toLowerCase();

    }

    private String getFilePath(String elsAccount) {

        return "upload" + File.separator + elsAccount + File.separator + LocalDate.now().toString();

    }

    private String getNewFileName(String fileName) {

        String fileShortName = fileName.substring(0, fileName.lastIndexOf("."));
        //空白符号或者单引号

        Matcher matcher = PATTERN.matcher(fileShortName);
        //替换文件名中的空白符号或者单引号为空
        fileShortName = matcher.replaceAll("");
        // 检查扩展名
        String fileExt = getFileExt(fileName);
        return fileShortName + "_" + IdWorker.getId() + "." + fileExt;

    }

    private String getSavePath(String elsAccount, String fileNewName) {

        String savePath = systemConfig.getFilePath() + getFilePath(elsAccount);
        //创建文件夹
        File saveDirFile = new File(savePath);
        if (!saveDirFile.exists()) {
            if (!saveDirFile.mkdirs()) {
                throw new BusinessException("文件夹创建失败,请确认操作系统及配置参数");
            }
        }
        return savePath + File.separator + fileNewName;
    }


}
