package com.els.tso.srm.core.service.impl;

import com.els.tso.srm.core.service.RedisCacheService;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * @author daipj
 * @date 2020/9/22 11:37
 */
@Service
@Slf4j
public class RedisCacheCacheServiceImpl implements RedisCacheService {

    private static final String LOCK_PRE = "LOCK:";

    private static final String CACHE_PRE = "CACHE:";

    private final RedisTemplate<Object, Object> redisTemplate;

    private final RedissonClient redissonClient;

    public RedisCacheCacheServiceImpl(RedisTemplate<Object, Object> redisTemplate, RedissonClient redissonClient) {
        this.redisTemplate = redisTemplate;
        this.redissonClient = redissonClient;
    }

    @Override
    public Object get(Object key) {
        return redisTemplate.boundValueOps(CACHE_PRE + key).get();
    }

    @Override
    public void set(Object key, Object value) {
        redisTemplate.boundValueOps(CACHE_PRE + key).set(value);
    }

    @Override
    public boolean getLock(String key) throws InterruptedException {

        final RLock lock = redissonClient.getLock(LOCK_PRE + key);
        return lock.tryLock(30, TimeUnit.SECONDS);
    }

    @Override
    public boolean getLock(String key, long timeout) throws InterruptedException {

        final RLock lock = redissonClient.getLock(LOCK_PRE + key);
        return lock.tryLock(timeout, TimeUnit.SECONDS);
    }

    @Override
    public void releaseLock(String key) {

        final RLock lock = redissonClient.getLock(LOCK_PRE + key);
        if (lock.isHeldByCurrentThread()) {
            lock.unlock();
        }
    }


    /**
     * @param key
     * @param value
     * @param expire 毫秒
     * @return
     */

    @Override
    public void setWithExpire(String key, String value, long expire) {

        redisTemplate.boundValueOps(CACHE_PRE + key).set(value, expire, TimeUnit.MICROSECONDS);
    }


    @Override
    public void lset(String key, long index, Object value) {

        redisTemplate.boundListOps(key).set(index, value);
    }


    @Override
    public Boolean del(String key) {

        return redisTemplate.delete(key);
    }

    @Override
    public Long lpush(String key, Object value) {

        return redisTemplate.boundListOps(key).leftPush(value);
    }

    /**
     * @param key
     * @param count
     * @param value 删除redis list 的某个元素
     */
    @Override
    public Long lremove(String key, int count, String value) {

        return redisTemplate.boundListOps(key).remove(count, value);
    }


    @Override
    public Object rpop(String key) {

        return redisTemplate.boundListOps(key).rightPop();
    }

    @Override
    public List<Object> brpop(String key) {

        return redisTemplate.boundListOps(key).range(0, -1);
    }


    @Override
    public List<Object> lrange(String key, long start, long end) {

        return redisTemplate.boundListOps(key).range(start, end);
    }


    @Override
    public Long addSet(String key, Set<String> set) {

        return redisTemplate.boundSetOps(key).add(set);
    }

    @Override
    public Boolean addZset(String key, Double score, String member) {

        return redisTemplate.boundZSetOps(key).add(member, score);
    }

    @Override
    public Long deleteZset(String key, String member) {

        return redisTemplate.boundZSetOps(key).remove(member);
    }

    @Override
    public Set<Object> getZset(String key, long start, long end) {

        return redisTemplate.boundZSetOps(key).range(start, end);
    }


    @Override
    public Long incr(String key) {

        return redisTemplate.boundValueOps(key).increment();
    }

    /**
     * hash set
     *
     * @param key
     * @return
     */
    @Override
    public void hset(String key, String field, byte[] value) {

        redisTemplate.boundHashOps(key).put(field, value);
    }

    /**
     * hash get
     *
     * @param key
     * @return
     */
    @Override
    public Object hget(String key, String field) {

        return redisTemplate.boundHashOps(key).get(field);

    }

    /**
     * hash getAll
     *
     * @param key
     * @return
     */
    @Override
    public Map<Object, Object> hgetAll(String key) {

        return redisTemplate.boundHashOps(key).entries();
    }

    /**
     * hash hdel
     *
     * @param key
     * @return
     */
    @Override
    public Boolean hdel(String key) {
        return redisTemplate.delete(key);
    }

    /**
     * hash hdel
     *
     * @param key
     * @return
     */
    @Override
    public Long hlen(String key) {
        return redisTemplate.boundHashOps(key).size();
    }

}
