/*
 * Decompiled with CFR 0.152.
 */
package timeseries.models.arima;

import data.DoubleFunctions;
import data.Operators;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.util.Arrays;
import linear.doubles.Matrix;
import linear.doubles.Vector;
import math.function.AbstractMultivariateFunction;
import optim.BFGS;
import stats.Statistics;
import stats.distributions.Distribution;
import stats.distributions.Normal;
import timeseries.TimePeriod;
import timeseries.TimeSeries;
import timeseries.models.Forecast;
import timeseries.models.Model;
import timeseries.models.arima.ArimaForecast;
import timeseries.models.arima.ArmaKalmanFilter;
import timeseries.models.arima.StateSpaceARMA;
import timeseries.operators.LagPolynomial;

public final class Arima
implements Model {
    private static final double MACHINE_EPSILON = Math.ulp(1.0);
    private static final double DEFAULT_TOLERANCE = Math.sqrt(MACHINE_EPSILON);
    private final TimeSeries observations;
    private final TimeSeries differencedSeries;
    private final TimeSeries fittedSeries;
    private final TimeSeries residuals;
    private final int seasonalFrequency;
    private final ModelOrder order;
    private final ModelInformation modelInfo;
    private final ModelCoefficients modelCoefficients;
    private final double mean;
    private final double[] arSarCoeffs;
    private final double[] maSmaCoeffs;
    private final double[] stdErrors;

    private Arima(TimeSeries observations, ModelOrder order, TimePeriod seasonalCycle, FittingStrategy fittingStrategy) {
        Matrix initHessian;
        Vector initParams;
        double meanParScale;
        this.observations = observations;
        this.order = order;
        this.seasonalFrequency = (int)observations.timePeriod().frequencyPer(seasonalCycle);
        this.differencedSeries = observations.difference(1, order.d).difference(this.seasonalFrequency, order.D);
        double d = meanParScale = order.constant == 1 ? 10.0 * this.differencedSeries.stdDeviation() / Math.sqrt(this.differencedSeries.n()) : 1.0;
        if (fittingStrategy == FittingStrategy.CSSML || fittingStrategy == FittingStrategy.USSML) {
            FittingStrategy subStrategy = fittingStrategy == FittingStrategy.CSSML ? FittingStrategy.CSS : FittingStrategy.USS;
            Arima firstModel = new Arima(observations, order, seasonalCycle, subStrategy);
            initParams = new Vector(firstModel.coefficients().getAllCoeffs());
            initHessian = this.getInitialHessian(firstModel);
        } else {
            initParams = new Vector(this.getInitialParameters(meanParScale));
            initHessian = this.getInitialHessian(initParams.size());
        }
        OptimFunction function = new OptimFunction(this.differencedSeries, order, fittingStrategy, this.seasonalFrequency, meanParScale);
        BFGS optimizer = new BFGS(function, initParams, DEFAULT_TOLERANCE, DEFAULT_TOLERANCE, initHessian);
        Vector optimizedParams = optimizer.parameters();
        Matrix inverseHessian = optimizer.inverseHessian();
        this.stdErrors = DoubleFunctions.sqrt(Operators.scale(inverseHessian.diagonal(), 1.0 / (double)this.differencedSeries.n()));
        if (order.constant == 1) {
            int n = this.stdErrors.length - 1;
            this.stdErrors[n] = this.stdErrors[n] * meanParScale;
        }
        double[] arCoeffs = this.getArCoeffs(optimizedParams);
        double[] maCoeffs = this.getMaCoeffs(optimizedParams);
        double[] sarCoeffs = this.getSarCoeffs(optimizedParams);
        double[] smaCoeffs = this.getSmaCoeffs(optimizedParams);
        this.arSarCoeffs = Arima.expandArCoefficients(arCoeffs, sarCoeffs, this.seasonalFrequency);
        this.maSmaCoeffs = Arima.expandMaCoefficients(maCoeffs, smaCoeffs, this.seasonalFrequency);
        this.mean = order.constant == 1 ? meanParScale * optimizedParams.at(order.p + order.q + order.P + order.Q) : 0.0;
        this.modelCoefficients = new ModelCoefficients(arCoeffs, maCoeffs, sarCoeffs, smaCoeffs, order.d, order.D, this.mean);
        if (fittingStrategy == FittingStrategy.CSS) {
            this.modelInfo = Arima.fitCSS(this.differencedSeries, this.arSarCoeffs, this.maSmaCoeffs, this.mean, order);
            double[] residuals = this.modelInfo.residuals;
            double[] fittedArray = this.integrate(Operators.differenceOf(this.differencedSeries.asArray(), residuals));
            this.fittedSeries = new TimeSeries(observations.timePeriod(), observations.observationTimes(), fittedArray);
            this.residuals = this.observations.minus(this.fittedSeries);
        } else if (fittingStrategy == FittingStrategy.USS) {
            this.modelInfo = Arima.fitUSS(this.differencedSeries, this.arSarCoeffs, this.maSmaCoeffs, this.mean, order);
            double[] residuals = this.modelInfo.residuals;
            double[] fittedArray = this.integrate(Operators.differenceOf(this.differencedSeries.asArray(), DoubleFunctions.slice(residuals, 2 * this.arSarCoeffs.length, residuals.length)));
            int diffs = order.d + order.D * this.seasonalFrequency;
            int offset = 2 * this.arSarCoeffs.length;
            for (int i = 0; i < diffs && i >= diffs - offset; ++i) {
                int n = i;
                fittedArray[n] = fittedArray[n] - residuals[offset - diffs + i];
            }
            this.fittedSeries = new TimeSeries(observations.timePeriod(), observations.observationTimes(), fittedArray);
            this.residuals = this.observations.minus(this.fittedSeries);
        } else {
            this.modelInfo = Arima.fitML(this.differencedSeries, this.arSarCoeffs, this.maSmaCoeffs, this.mean, order);
            double[] residuals = this.modelInfo.residuals;
            double[] fittedArray = this.integrate(Operators.differenceOf(this.differencedSeries.asArray(), residuals));
            this.fittedSeries = new TimeSeries(observations.timePeriod(), observations.observationTimes(), fittedArray);
            this.residuals = this.observations.minus(this.fittedSeries);
        }
    }

    private Arima(TimeSeries observations, ModelCoefficients coeffs, TimePeriod seasonalCycle, FittingStrategy fittingStrategy) {
        this.observations = observations;
        this.modelCoefficients = coeffs;
        this.order = coeffs.extractModelOrder();
        this.seasonalFrequency = (int)observations.timePeriod().frequencyPer(seasonalCycle);
        this.differencedSeries = observations.difference(1, this.order.d).difference(this.seasonalFrequency, this.order.D);
        this.arSarCoeffs = Arima.expandArCoefficients(coeffs.arCoeffs, coeffs.sarCoeffs, this.seasonalFrequency);
        this.maSmaCoeffs = Arima.expandMaCoefficients(coeffs.maCoeffs, coeffs.smaCoeffs, this.seasonalFrequency);
        this.mean = coeffs.mean;
        this.stdErrors = DoubleFunctions.fill(this.order.sumARMA() + this.order.constant, Double.POSITIVE_INFINITY);
        if (fittingStrategy == FittingStrategy.CSS) {
            this.modelInfo = Arima.fitCSS(this.differencedSeries, this.arSarCoeffs, this.maSmaCoeffs, this.mean, this.order);
            double[] residuals = this.modelInfo.residuals;
            double[] fittedArray = this.integrate(Operators.differenceOf(this.differencedSeries.asArray(), residuals));
            this.fittedSeries = new TimeSeries(observations.timePeriod(), observations.observationTimes(), fittedArray);
            this.residuals = this.observations.minus(this.fittedSeries);
        } else if (fittingStrategy == FittingStrategy.USS) {
            this.modelInfo = Arima.fitUSS(this.differencedSeries, this.arSarCoeffs, this.maSmaCoeffs, this.mean, this.order);
            double[] residuals = this.modelInfo.residuals;
            double[] fittedArray = this.integrate(Operators.differenceOf(this.differencedSeries.asArray(), DoubleFunctions.slice(residuals, 2 * this.arSarCoeffs.length, residuals.length)));
            int diffs = this.order.d + this.order.D * this.seasonalFrequency;
            int offset = 2 * this.arSarCoeffs.length;
            for (int i = 0; i < diffs && i >= diffs - offset; ++i) {
                int n = i;
                fittedArray[n] = fittedArray[n] - residuals[offset - diffs + i];
            }
            this.fittedSeries = new TimeSeries(observations.timePeriod(), observations.observationTimes(), fittedArray);
            this.residuals = this.observations.minus(this.fittedSeries);
        } else {
            this.modelInfo = Arima.fitML(this.differencedSeries, this.arSarCoeffs, this.maSmaCoeffs, this.mean, this.order);
            double[] residuals = this.modelInfo.residuals;
            double[] fittedArray = this.integrate(Operators.differenceOf(this.differencedSeries.asArray(), residuals));
            this.fittedSeries = new TimeSeries(observations.timePeriod(), observations.observationTimes(), fittedArray);
            this.residuals = this.observations.minus(this.fittedSeries);
        }
    }

    public static Arima model(TimeSeries observations, ModelOrder order) {
        return new Arima(observations, order, TimePeriod.oneYear(), FittingStrategy.USS);
    }

    public static Arima model(TimeSeries observations, ModelOrder order, TimePeriod seasonalCycle) {
        return new Arima(observations, order, seasonalCycle, FittingStrategy.USS);
    }

    public static Arima model(TimeSeries observations, ModelOrder order, FittingStrategy fittingStrategy) {
        return new Arima(observations, order, TimePeriod.oneYear(), fittingStrategy);
    }

    public static Arima model(TimeSeries observations, ModelOrder order, TimePeriod seasonalCycle, FittingStrategy fittingStrategy) {
        return new Arima(observations, order, seasonalCycle, fittingStrategy);
    }

    public static Arima model(TimeSeries observations, ModelCoefficients coeffs, FittingStrategy fittingStrategy) {
        return new Arima(observations, coeffs, TimePeriod.oneYear(), fittingStrategy);
    }

    public static Arima model(TimeSeries observations, ModelCoefficients coeffs, TimePeriod seasonalCycle) {
        return new Arima(observations, coeffs, seasonalCycle, FittingStrategy.USS);
    }

    public static Arima model(TimeSeries observations, ModelCoefficients coeffs, TimePeriod seasonalCycle, FittingStrategy fittingStrategy) {
        return new Arima(observations, coeffs, seasonalCycle, fittingStrategy);
    }

    private static ModelInformation fitCSS(TimeSeries differencedSeries, double[] arCoeffs, double[] maCoeffs, double mean, ModelOrder order) {
        int offset = arCoeffs.length;
        int n = differencedSeries.n();
        double[] fitted = new double[n];
        double[] residuals = new double[n];
        for (int t = offset; t < fitted.length; ++t) {
            fitted[t] = mean;
            for (int i = 0; i < arCoeffs.length; ++i) {
                int n2 = t;
                fitted[n2] = fitted[n2] + arCoeffs[i] * (differencedSeries.at(t - i - 1) - mean);
            }
            for (int j = 0; j < Math.min(t, maCoeffs.length); ++j) {
                int n3 = t;
                fitted[n3] = fitted[n3] + maCoeffs[j] * residuals[t - j - 1];
            }
            residuals[t] = differencedSeries.at(t) - fitted[t];
        }
        int npar = order.sumARMA() + order.constant;
        int m = differencedSeries.n() - npar;
        double sigma2 = Statistics.sumOfSquared(residuals) / (double)m;
        double logLikelihood = (double)(-n) / 2.0 * (Math.log(Math.PI * 2 * sigma2) + 1.0);
        return new ModelInformation(npar, sigma2, logLikelihood, residuals, fitted);
    }

    private static ModelInformation fitUSS(TimeSeries differencedSeries, double[] arCoeffs, double[] maCoeffs, double mean, ModelOrder order) {
        int i;
        int t;
        int i2;
        int n = differencedSeries.n();
        int m = arCoeffs.length;
        double[] extendedFit = new double[2 * m + n];
        double[] extendedSeries = new double[2 * m + n];
        double[] residuals = new double[2 * m + n];
        for (i2 = 0; i2 < differencedSeries.n(); ++i2) {
            extendedSeries[i2] = differencedSeries.at(--n);
        }
        for (t = n = differencedSeries.n(); t < n + 2 * m; ++t) {
            extendedSeries[t] = mean;
            for (i = 0; i < arCoeffs.length; ++i) {
                if (!(Math.abs(arCoeffs[i]) > 0.0)) continue;
                int n2 = t;
                extendedSeries[n2] = extendedSeries[n2] + arCoeffs[i] * (extendedSeries[t - i - 1] - mean);
            }
        }
        n = extendedSeries.length;
        for (i2 = 0; i2 < m; ++i2) {
            extendedFit[i2] = extendedSeries[n - i2 - 1];
        }
        for (t = m; t < n; ++t) {
            extendedFit[t] = mean;
            for (i = 0; i < arCoeffs.length; ++i) {
                if (!(Math.abs(arCoeffs[i]) > 0.0)) continue;
                int n3 = t;
                extendedFit[n3] = extendedFit[n3] + arCoeffs[i] * (extendedSeries[n - t + i] - mean);
            }
            for (int j = 0; j < Math.min(t, maCoeffs.length); ++j) {
                if (!(Math.abs(maCoeffs[j]) > 0.0)) continue;
                int n4 = t;
                extendedFit[n4] = extendedFit[n4] + maCoeffs[j] * residuals[t - j - 1];
            }
            residuals[t] = extendedSeries[n - t - 1] - extendedFit[t];
        }
        n = residuals.length;
        int npar = order.sumARMA() + order.constant;
        double sigma2 = Statistics.sumOfSquared(residuals) / (double)(n - npar);
        double logLikelihood = (double)(-n) / 2.0 * (Math.log(Math.PI * 2 * sigma2) + 1.0);
        return new ModelInformation(npar, sigma2, logLikelihood, residuals, extendedFit);
    }

    private static ModelInformation fitML(TimeSeries differencedSeries, double[] arCoeffs, double[] maCoeffs, double mean, ModelOrder order) {
        double[] series = Operators.subtract(differencedSeries.asArray(), mean);
        ArmaKalmanFilter.KalmanOutput output = Arima.kalmanFit(differencedSeries, arCoeffs, maCoeffs, mean);
        double sigma2 = output.sigma2();
        double logLikelihood = output.logLikelihood();
        double[] residuals = output.residuals();
        double[] fitted = Operators.differenceOf(series, residuals);
        int npar = order.sumARMA() + order.constant + 1;
        return new ModelInformation(npar, sigma2, logLikelihood, residuals, fitted);
    }

    private static ArmaKalmanFilter.KalmanOutput kalmanFit(TimeSeries differencedSeries, double[] arCoeffs, double[] maCoeffs, double mean) {
        double[] series = Operators.subtract(differencedSeries.asArray(), mean);
        StateSpaceARMA ss = new StateSpaceARMA(series, arCoeffs, maCoeffs);
        ArmaKalmanFilter kalmanFilter = new ArmaKalmanFilter(ss);
        return kalmanFilter.output();
    }

    public static ModelOrder order(int p, int d, int q) {
        return new ModelOrder(p, d, q, 0, 0, 0, d == 0);
    }

    public static ModelOrder order(int p, int d, int q, boolean constant) {
        return new ModelOrder(p, d, q, 0, 0, 0, constant);
    }

    public static ModelOrder order(int p, int d, int q, int P, int D, int Q) {
        return new ModelOrder(p, d, q, P, D, Q, d == 0 && D == 0);
    }

    public static ModelOrder order(int p, int d, int q, int P, int D, int Q, boolean constant) {
        return new ModelOrder(p, d, q, P, D, Q, constant);
    }

    private static double[] expandArCoefficients(double[] arCoeffs, double[] sarCoeffs, int seasonalFrequency) {
        double[] arSarCoeffs = new double[arCoeffs.length + sarCoeffs.length * seasonalFrequency];
        System.arraycopy(arCoeffs, 0, arSarCoeffs, 0, arCoeffs.length);
        for (int i = 0; i < sarCoeffs.length; ++i) {
            arSarCoeffs[(i + 1) * seasonalFrequency - 1] = sarCoeffs[i];
            for (int j = 0; j < arCoeffs.length; ++j) {
                arSarCoeffs[(i + 1) * seasonalFrequency + j] = -sarCoeffs[i] * arCoeffs[j];
            }
        }
        return arSarCoeffs;
    }

    private static double[] expandMaCoefficients(double[] maCoeffs, double[] smaCoeffs, int seasonalFrequency) {
        double[] maSmaCoeffs = new double[maCoeffs.length + smaCoeffs.length * seasonalFrequency];
        System.arraycopy(maCoeffs, 0, maSmaCoeffs, 0, maCoeffs.length);
        for (int i = 0; i < smaCoeffs.length; ++i) {
            maSmaCoeffs[(i + 1) * seasonalFrequency - 1] = smaCoeffs[i];
            for (int j = 0; j < maCoeffs.length; ++j) {
                maSmaCoeffs[(i + 1) * seasonalFrequency + j] = smaCoeffs[i] * maCoeffs[j];
            }
        }
        return maSmaCoeffs;
    }

    public double[] fcst(int steps) {
        int d = this.order.d;
        int D = this.order.D;
        int n = this.differencedSeries.n();
        int m = this.observations.n();
        double[] resid = this.residuals.asArray();
        double[] diffedFcst = new double[n + steps];
        double[] fcst = new double[m + steps];
        System.arraycopy(this.differencedSeries.asArray(), 0, diffedFcst, 0, n);
        System.arraycopy(this.observations.asArray(), 0, fcst, 0, m);
        LagPolynomial diffPolynomial = LagPolynomial.differences(d);
        LagPolynomial seasDiffPolynomial = LagPolynomial.seasonalDifferences(this.seasonalFrequency, D);
        LagPolynomial lagPolyomial = diffPolynomial.times(seasDiffPolynomial);
        for (int t = 0; t < steps; ++t) {
            diffedFcst[n + t] = this.mean;
            fcst[m + t] = this.mean;
            int n2 = m + t;
            fcst[n2] = fcst[n2] + lagPolyomial.fit(fcst, m + t);
            for (int i = 0; i < this.arSarCoeffs.length; ++i) {
                int n3 = n + t;
                diffedFcst[n3] = diffedFcst[n3] + this.arSarCoeffs[i] * (diffedFcst[n + t - i - 1] - this.mean);
                int n4 = m + t;
                fcst[n4] = fcst[n4] + this.arSarCoeffs[i] * (diffedFcst[n + t - i - 1] - this.mean);
            }
            for (int j = this.maSmaCoeffs.length; j > 0 && t - j < 0; --j) {
                int n5 = n + t;
                diffedFcst[n5] = diffedFcst[n5] + this.maSmaCoeffs[j - 1] * resid[m + t - j];
                int n6 = m + t;
                fcst[n6] = fcst[n6] + this.maSmaCoeffs[j - 1] * resid[m + t - j];
            }
        }
        return DoubleFunctions.slice(fcst, m, m + steps);
    }

    @Override
    public TimeSeries pointForecast(int steps) {
        int n = this.observations.n();
        double[] fcst = this.fcst(steps);
        TimePeriod timePeriod = this.observations.timePeriod();
        OffsetDateTime startTime = this.observations.observationTimes().get(n - 1).plus(timePeriod.periodLength() * timePeriod.timeUnit().unitLength(), timePeriod.timeUnit().temporalUnit());
        return new TimeSeries(timePeriod, startTime, fcst);
    }

    @Override
    public Forecast forecast(int steps, double alpha) {
        return ArimaForecast.forecast(this, steps, alpha);
    }

    @Override
    public Forecast forecast(int steps) {
        return this.forecast(steps, 0.05);
    }

    private double[] integrate(double[] fitted) {
        int t;
        int offset = this.order.d + this.order.D * this.seasonalFrequency;
        double[] integrated = new double[this.observations.n()];
        for (t = 0; t < offset; ++t) {
            integrated[t] = this.observations.at(t);
        }
        for (t = offset; t < this.observations.n(); ++t) {
            integrated[t] = this.observations.at(t) - this.differencedSeries.at(t - offset) + fitted[t - offset];
        }
        return integrated;
    }

    private Matrix getInitialHessian(int n) {
        return new Matrix.IdentityBuilder(n).build();
    }

    private Matrix getInitialHessian(Arima model) {
        double[] stdErrors = model.stdErrors;
        Matrix.IdentityBuilder builder = new Matrix.IdentityBuilder(stdErrors.length);
        for (int i = 0; i < stdErrors.length; ++i) {
            builder.set(i, i, stdErrors[i] * stdErrors[i] * (double)this.differencedSeries.n());
        }
        return builder.build();
    }

    private double[] getInitialParameters(double meanParScale) {
        double[] initParams = new double[this.order.sumARMA() + this.order.constant];
        if (this.order.constant == 1 && Math.abs(meanParScale) > Math.ulp(1.0)) {
            initParams[initParams.length - 1] = this.differencedSeries.mean() / meanParScale;
        }
        return initParams;
    }

    private double[] getSarCoeffs(Vector optimizedParams) {
        double[] sarCoeffs = new double[this.order.P];
        for (int i = 0; i < this.order.P; ++i) {
            sarCoeffs[i] = optimizedParams.at(i + this.order.p + this.order.q);
        }
        return sarCoeffs;
    }

    private double[] getSmaCoeffs(Vector optimizedParams) {
        double[] smaCoeffs = new double[this.order.Q];
        for (int i = 0; i < this.order.Q; ++i) {
            smaCoeffs[i] = optimizedParams.at(i + this.order.p + this.order.q + this.order.P);
        }
        return smaCoeffs;
    }

    private double[] getArCoeffs(Vector optimizedParams) {
        double[] arCoeffs = new double[this.order.p];
        for (int i = 0; i < this.order.p; ++i) {
            arCoeffs[i] = optimizedParams.at(i);
        }
        return arCoeffs;
    }

    private double[] getMaCoeffs(Vector optimizedParams) {
        double[] arCoeffs = new double[this.order.q];
        for (int i = 0; i < this.order.q; ++i) {
            arCoeffs[i] = optimizedParams.at(i + this.order.p);
        }
        return arCoeffs;
    }

    @Override
    public TimeSeries timeSeries() {
        return this.observations;
    }

    @Override
    public TimeSeries fittedSeries() {
        return this.fittedSeries;
    }

    @Override
    public TimeSeries residuals() {
        return this.residuals;
    }

    public double sigma2() {
        return this.modelInfo.sigma2;
    }

    public int seasonalFrequency() {
        return this.seasonalFrequency;
    }

    public double[] stdErrors() {
        return (double[])this.stdErrors.clone();
    }

    public ModelCoefficients coefficients() {
        return this.modelCoefficients;
    }

    public ModelOrder order() {
        return this.order;
    }

    public double logLikelihood() {
        return this.modelInfo.logLikelihood;
    }

    public double aic() {
        return this.modelInfo.aic;
    }

    double[] arSarCoefficients() {
        return (double[])this.arSarCoeffs.clone();
    }

    double[] maSmaCoefficients() {
        return (double[])this.maSmaCoeffs.clone();
    }

    public String toString() {
        return "\norder: " + this.order + "\nmodelInfo: " + this.modelInfo + "\nmodelCoefficients: " + this.modelCoefficients;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        Arima arima = (Arima)o;
        if (this.seasonalFrequency != arima.seasonalFrequency) {
            return false;
        }
        if (Double.compare(arima.mean, this.mean) != 0) {
            return false;
        }
        if (!this.observations.equals(arima.observations)) {
            return false;
        }
        if (!this.differencedSeries.equals(arima.differencedSeries)) {
            return false;
        }
        if (!this.fittedSeries.equals(arima.fittedSeries)) {
            return false;
        }
        if (!this.residuals.equals(arima.residuals)) {
            return false;
        }
        if (!this.order.equals(arima.order)) {
            return false;
        }
        if (!this.modelInfo.equals(arima.modelInfo)) {
            return false;
        }
        if (!this.modelCoefficients.equals(arima.modelCoefficients)) {
            return false;
        }
        if (!Arrays.equals(this.arSarCoeffs, arima.arSarCoeffs)) {
            return false;
        }
        if (!Arrays.equals(this.maSmaCoeffs, arima.maSmaCoeffs)) {
            return false;
        }
        return Arrays.equals(this.stdErrors, arima.stdErrors);
    }

    public int hashCode() {
        int result = this.observations.hashCode();
        result = 31 * result + this.differencedSeries.hashCode();
        result = 31 * result + this.fittedSeries.hashCode();
        result = 31 * result + this.residuals.hashCode();
        result = 31 * result + this.seasonalFrequency;
        result = 31 * result + this.order.hashCode();
        result = 31 * result + this.modelInfo.hashCode();
        result = 31 * result + this.modelCoefficients.hashCode();
        long temp = Double.doubleToLongBits(this.mean);
        result = 31 * result + (int)(temp ^ temp >>> 32);
        result = 31 * result + Arrays.hashCode(this.arSarCoeffs);
        result = 31 * result + Arrays.hashCode(this.maSmaCoeffs);
        result = 31 * result + Arrays.hashCode(this.stdErrors);
        return result;
    }

    public static class Simulation {
        private final ModelCoefficients coefficients;
        private final Distribution distribution;
        private final TimePeriod period;
        private final TimePeriod seasonalCycle;
        private final int n;

        private Simulation(Builder builder) {
            this.coefficients = builder.coefficients;
            this.distribution = builder.distribution;
            this.period = builder.period;
            this.seasonalCycle = builder.seasonalCycle;
            this.n = builder.n;
        }

        public static Builder newBuilder() {
            return new Builder();
        }

        public TimeSeries sim() {
            int end;
            int burnin = (int)((double)this.n / 2.0);
            int seasonalFrequency = (int)this.period.frequencyPer(this.seasonalCycle);
            double[] arSarCoeffs = Arima.expandArCoefficients(this.coefficients.arCoeffs, this.coefficients.sarCoeffs, seasonalFrequency);
            double[] maSmaCoeffs = Arima.expandMaCoefficients(this.coefficients.maCoeffs, this.coefficients.smaCoeffs, seasonalFrequency);
            int diffOffset = this.coefficients.d + this.coefficients.D * seasonalFrequency;
            int offset = Math.min(this.n, arSarCoeffs.length);
            double[] series = new double[this.n + burnin];
            double[] errors = new double[this.n + burnin];
            for (int t = 0; t < offset; ++t) {
                series[t] = errors[t] = this.distribution.rand();
                int n = t;
                series[n] = series[n] + this.coefficients.mean;
                for (int j = 0; j < Math.min(t, maSmaCoeffs.length); ++j) {
                    int n2 = t;
                    series[n2] = series[n2] + maSmaCoeffs[j] * errors[t - j - 1];
                }
            }
            for (int t = offset; t < this.n + burnin; ++t) {
                int j;
                series[t] = errors[t] = this.distribution.rand();
                int n = t;
                series[n] = series[n] + this.coefficients.mean;
                end = Math.min(t, arSarCoeffs.length);
                for (j = 0; j < end; ++j) {
                    int n3 = t;
                    series[n3] = series[n3] + arSarCoeffs[j] * (series[t - j - 1] - this.coefficients.mean);
                }
                end = Math.min(t, maSmaCoeffs.length);
                for (j = 0; j < end; ++j) {
                    int n4 = t;
                    series[n4] = series[n4] + maSmaCoeffs[j] * errors[t - j - 1];
                }
            }
            LagPolynomial poly = LagPolynomial.differences(this.coefficients.d).times(LagPolynomial.seasonalDifferences(seasonalFrequency, this.coefficients.D));
            end = this.n + burnin;
            for (int t = diffOffset; t < end; ++t) {
                int n = t;
                series[n] = series[n] + poly.fit(series, t);
            }
            series = DoubleFunctions.slice(series, burnin, this.n + burnin);
            return new TimeSeries(this.period, OffsetDateTime.of(1, 1, 1, 0, 0, 0, 0, ZoneOffset.ofHours(0)), series);
        }

        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (!(o instanceof Simulation)) {
                return false;
            }
            Simulation other = (Simulation)o;
            if (!other.canEqual(this)) {
                return false;
            }
            ModelCoefficients this$coefficients = this.coefficients;
            ModelCoefficients other$coefficients = other.coefficients;
            if (this$coefficients == null ? other$coefficients != null : !((Object)this$coefficients).equals(other$coefficients)) {
                return false;
            }
            Distribution this$distribution = this.distribution;
            Distribution other$distribution = other.distribution;
            if (this$distribution == null ? other$distribution != null : !this$distribution.equals(other$distribution)) {
                return false;
            }
            TimePeriod this$period = this.period;
            TimePeriod other$period = other.period;
            if (this$period == null ? other$period != null : !((Object)this$period).equals(other$period)) {
                return false;
            }
            TimePeriod this$seasonalCycle = this.seasonalCycle;
            TimePeriod other$seasonalCycle = other.seasonalCycle;
            if (this$seasonalCycle == null ? other$seasonalCycle != null : !((Object)this$seasonalCycle).equals(other$seasonalCycle)) {
                return false;
            }
            return this.n == other.n;
        }

        protected boolean canEqual(Object other) {
            return other instanceof Simulation;
        }

        public int hashCode() {
            int PRIME = 59;
            int result = 1;
            ModelCoefficients $coefficients = this.coefficients;
            result = result * 59 + ($coefficients == null ? 43 : ((Object)$coefficients).hashCode());
            Distribution $distribution = this.distribution;
            result = result * 59 + ($distribution == null ? 43 : $distribution.hashCode());
            TimePeriod $period = this.period;
            result = result * 59 + ($period == null ? 43 : ((Object)$period).hashCode());
            TimePeriod $seasonalCycle = this.seasonalCycle;
            result = result * 59 + ($seasonalCycle == null ? 43 : ((Object)$seasonalCycle).hashCode());
            result = result * 59 + this.n;
            return result;
        }

        public String toString() {
            return "Arima.Simulation(coefficients=" + this.coefficients + ", distribution=" + this.distribution + ", period=" + this.period + ", seasonalCycle=" + this.seasonalCycle + ", n=" + this.n + ")";
        }

        public static class Builder {
            private ModelCoefficients coefficients = ModelCoefficients.newBuilder().build();
            private Distribution distribution = new Normal();
            private TimePeriod period = this.coefficients.isSeasonal() ? TimePeriod.oneMonth() : TimePeriod.oneYear();
            private TimePeriod seasonalCycle = TimePeriod.oneYear();
            private int n = 500;
            private boolean periodSet = false;

            public Builder setCoefficients(ModelCoefficients coefficients) {
                if (coefficients == null) {
                    throw new NullPointerException("The model coefficients cannot be null.");
                }
                this.coefficients = coefficients;
                if (!this.periodSet) {
                    this.period = coefficients.isSeasonal() ? TimePeriod.oneMonth() : TimePeriod.oneYear();
                }
                return this;
            }

            public Builder setDistribution(Distribution distribution) {
                if (distribution == null) {
                    throw new NullPointerException("The distribution cannot be null.");
                }
                this.distribution = distribution;
                return this;
            }

            public Builder setPeriod(TimePeriod period) {
                if (period == null) {
                    throw new NullPointerException("The time period cannot be null.");
                }
                this.periodSet = true;
                this.period = period;
                return this;
            }

            public Builder setSeasonalCycle(TimePeriod seasonalCycle) {
                if (seasonalCycle == null) {
                    throw new NullPointerException("The seasonal cycle cannot be null.");
                }
                this.seasonalCycle = seasonalCycle;
                return this;
            }

            public Builder setN(int n) {
                if (n < 1) {
                    throw new IllegalArgumentException("the number of observations to simulate must be a positive integer.");
                }
                this.n = n;
                return this;
            }

            public TimeSeries sim() {
                return new Simulation(this).sim();
            }

            public Simulation build() {
                return new Simulation(this);
            }
        }
    }

    static class OptimFunction
    extends AbstractMultivariateFunction {
        private final TimeSeries differencedSeries;
        private final ModelOrder order;
        private final FittingStrategy fittingStrategy;
        private final int seasonalFrequency;
        private final double[] arParams;
        private final double[] maParams;
        private final double[] sarParams;
        private final double[] smaParams;
        private final double meanScale;

        OptimFunction(TimeSeries differencedSeries, ModelOrder order, FittingStrategy fittingStrategy, int seasonalFrequency, double meanScale) {
            this.differencedSeries = differencedSeries;
            this.order = order;
            this.fittingStrategy = fittingStrategy;
            this.seasonalFrequency = seasonalFrequency;
            this.arParams = new double[order.p];
            this.maParams = new double[order.q];
            this.sarParams = new double[order.P];
            this.smaParams = new double[order.Q];
            this.meanScale = meanScale;
        }

        @Override
        public final double at(Vector point) {
            double mean;
            ++this.functionEvaluations;
            double[] params = point.elements();
            System.arraycopy(params, 0, this.arParams, 0, this.order.p);
            System.arraycopy(params, this.order.p, this.maParams, 0, this.order.q);
            System.arraycopy(params, this.order.p + this.order.q, this.sarParams, 0, this.order.P);
            System.arraycopy(params, this.order.p + this.order.q + this.order.P, this.smaParams, 0, this.order.Q);
            double[] arCoeffs = Arima.expandArCoefficients(this.arParams, this.sarParams, this.seasonalFrequency);
            double[] maCoeffs = Arima.expandMaCoefficients(this.maParams, this.smaParams, this.seasonalFrequency);
            double d = mean = this.order.constant == 1 ? this.meanScale * params[params.length - 1] : 0.0;
            if (this.fittingStrategy == FittingStrategy.ML || this.fittingStrategy == FittingStrategy.CSSML || this.fittingStrategy == FittingStrategy.USSML) {
                int n = this.differencedSeries.n();
                ArmaKalmanFilter.KalmanOutput output = Arima.kalmanFit(this.differencedSeries, arCoeffs, maCoeffs, mean);
                return 0.5 * (Math.log(output.sigma2()) + output.sumLog() / (double)n);
            }
            ModelInformation info = this.fittingStrategy == FittingStrategy.CSS ? Arima.fitCSS(this.differencedSeries, arCoeffs, maCoeffs, mean, this.order) : Arima.fitUSS(this.differencedSeries, arCoeffs, maCoeffs, mean, this.order);
            return 0.5 * Math.log(info.sigma2);
        }

        public String toString() {
            return "differencedSeries: " + this.differencedSeries + "\norder: " + this.order + "\nfittingStrategy: " + (Object)((Object)this.fittingStrategy) + "\nseasonalFrequency: " + this.seasonalFrequency + "\narParams: " + Arrays.toString(this.arParams) + "\nmaParams: " + Arrays.toString(this.maParams) + "\nsarParams: " + Arrays.toString(this.sarParams) + "\nsmaParams: " + Arrays.toString(this.smaParams);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            OptimFunction that = (OptimFunction)o;
            if (this.seasonalFrequency != that.seasonalFrequency) {
                return false;
            }
            if (Double.compare(that.meanScale, this.meanScale) != 0) {
                return false;
            }
            if (!this.differencedSeries.equals(that.differencedSeries)) {
                return false;
            }
            if (!this.order.equals(that.order)) {
                return false;
            }
            if (this.fittingStrategy != that.fittingStrategy) {
                return false;
            }
            if (!Arrays.equals(this.arParams, that.arParams)) {
                return false;
            }
            if (!Arrays.equals(this.maParams, that.maParams)) {
                return false;
            }
            if (!Arrays.equals(this.sarParams, that.sarParams)) {
                return false;
            }
            return Arrays.equals(this.smaParams, that.smaParams);
        }

        public int hashCode() {
            int result = this.differencedSeries.hashCode();
            result = 31 * result + this.order.hashCode();
            result = 31 * result + this.fittingStrategy.hashCode();
            result = 31 * result + this.seasonalFrequency;
            result = 31 * result + Arrays.hashCode(this.arParams);
            result = 31 * result + Arrays.hashCode(this.maParams);
            result = 31 * result + Arrays.hashCode(this.sarParams);
            result = 31 * result + Arrays.hashCode(this.smaParams);
            long temp = Double.doubleToLongBits(this.meanScale);
            result = 31 * result + (int)(temp ^ temp >>> 32);
            return result;
        }
    }

    static class ModelInformation {
        private final double sigma2;
        private final double logLikelihood;
        private final double aic;
        private final double[] residuals;
        private final double[] fitted;

        ModelInformation(int npar, double sigma2, double logLikelihood, double[] residuals, double[] fitted) {
            this.sigma2 = sigma2;
            this.logLikelihood = logLikelihood;
            this.aic = (double)(2 * npar) - 2.0 * logLikelihood;
            this.residuals = (double[])residuals.clone();
            this.fitted = (double[])fitted.clone();
        }

        public String toString() {
            return "sigma2: " + this.sigma2 + "\nlogLikelihood: " + this.logLikelihood + "\nAIC: " + this.aic;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            ModelInformation that = (ModelInformation)o;
            if (Double.compare(that.sigma2, this.sigma2) != 0) {
                return false;
            }
            if (Double.compare(that.logLikelihood, this.logLikelihood) != 0) {
                return false;
            }
            if (Double.compare(that.aic, this.aic) != 0) {
                return false;
            }
            if (!Arrays.equals(this.residuals, that.residuals)) {
                return false;
            }
            return Arrays.equals(this.fitted, that.fitted);
        }

        public int hashCode() {
            long temp = Double.doubleToLongBits(this.sigma2);
            int result = (int)(temp ^ temp >>> 32);
            temp = Double.doubleToLongBits(this.logLikelihood);
            result = 31 * result + (int)(temp ^ temp >>> 32);
            temp = Double.doubleToLongBits(this.aic);
            result = 31 * result + (int)(temp ^ temp >>> 32);
            result = 31 * result + Arrays.hashCode(this.residuals);
            result = 31 * result + Arrays.hashCode(this.fitted);
            return result;
        }
    }

    public static class ModelCoefficients {
        private final double[] arCoeffs;
        private final double[] maCoeffs;
        private final double[] sarCoeffs;
        private final double[] smaCoeffs;
        private final int d;
        private final int D;
        private final double mean;
        private final double intercept;

        ModelCoefficients(double[] arCoeffs, double[] maCoeffs, double[] sarCoeffs, double[] smaCoeffs, int d, int D, double mean) {
            this.arCoeffs = (double[])arCoeffs.clone();
            this.maCoeffs = (double[])maCoeffs.clone();
            this.sarCoeffs = (double[])sarCoeffs.clone();
            this.smaCoeffs = (double[])smaCoeffs.clone();
            this.d = d;
            this.D = D;
            this.mean = mean;
            this.intercept = this.mean * (1.0 - Statistics.sumOf(arCoeffs) - Statistics.sumOf(sarCoeffs));
        }

        private ModelCoefficients(Builder builder) {
            this.arCoeffs = (double[])builder.arCoeffs.clone();
            this.maCoeffs = (double[])builder.maCoeffs.clone();
            this.sarCoeffs = (double[])builder.sarCoeffs.clone();
            this.smaCoeffs = (double[])builder.smaCoeffs.clone();
            this.d = builder.d;
            this.D = builder.D;
            this.mean = builder.mean;
            this.intercept = this.mean * (1.0 - Statistics.sumOf(this.arCoeffs) - Statistics.sumOf(this.sarCoeffs));
        }

        public static Builder newBuilder() {
            return new Builder();
        }

        public final double[] arCoeffs() {
            return (double[])this.arCoeffs.clone();
        }

        public final double[] maCoeffs() {
            return (double[])this.maCoeffs.clone();
        }

        public final double[] seasonalARCoeffs() {
            return (double[])this.sarCoeffs.clone();
        }

        public final double[] seasonalMACoeffs() {
            return (double[])this.smaCoeffs.clone();
        }

        public final int d() {
            return this.d;
        }

        public final int D() {
            return this.D;
        }

        public final double mean() {
            return this.mean;
        }

        public final double intercept() {
            return this.intercept;
        }

        public final double[] getAllCoeffs() {
            if (Math.abs(this.mean) < 1.0E-16) {
                return DoubleFunctions.combine(this.arCoeffs, this.maCoeffs, this.sarCoeffs, this.smaCoeffs);
            }
            return DoubleFunctions.append(DoubleFunctions.combine(this.arCoeffs, this.maCoeffs, this.sarCoeffs, this.smaCoeffs), this.mean);
        }

        final boolean isSeasonal() {
            return this.D > 0 || this.sarCoeffs.length > 0 || this.smaCoeffs.length > 0;
        }

        private ModelOrder extractModelOrder() {
            return new ModelOrder(this.arCoeffs.length, this.d, this.maCoeffs.length, this.sarCoeffs.length, this.D, this.smaCoeffs.length, Math.abs(this.mean) > 1.0E-16);
        }

        public String toString() {
            StringBuilder sb = new StringBuilder();
            if (this.arCoeffs.length > 0) {
                sb.append("\nautoregressive: ").append(Arrays.toString(this.arCoeffs));
            }
            if (this.maCoeffs.length > 0) {
                sb.append("\nmoving-average: ").append(Arrays.toString(this.maCoeffs));
            }
            if (this.sarCoeffs.length > 0) {
                sb.append("\nseasonal autoregressive: ").append(Arrays.toString(this.sarCoeffs));
            }
            if (this.smaCoeffs.length > 0) {
                sb.append("\nseasonal moving-average: ").append(Arrays.toString(this.smaCoeffs));
            }
            sb.append("\nmean: ").append(this.mean);
            sb.append("\nintercept: ").append(this.intercept);
            return sb.toString();
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            ModelCoefficients that = (ModelCoefficients)o;
            if (this.d != that.d) {
                return false;
            }
            if (this.D != that.D) {
                return false;
            }
            if (Double.compare(that.mean, this.mean) != 0) {
                return false;
            }
            if (Double.compare(that.intercept, this.intercept) != 0) {
                return false;
            }
            if (!Arrays.equals(this.arCoeffs, that.arCoeffs)) {
                return false;
            }
            if (!Arrays.equals(this.maCoeffs, that.maCoeffs)) {
                return false;
            }
            if (!Arrays.equals(this.sarCoeffs, that.sarCoeffs)) {
                return false;
            }
            return Arrays.equals(this.smaCoeffs, that.smaCoeffs);
        }

        public int hashCode() {
            int result = Arrays.hashCode(this.arCoeffs);
            result = 31 * result + Arrays.hashCode(this.maCoeffs);
            result = 31 * result + Arrays.hashCode(this.sarCoeffs);
            result = 31 * result + Arrays.hashCode(this.smaCoeffs);
            result = 31 * result + this.d;
            result = 31 * result + this.D;
            long temp = Double.doubleToLongBits(this.mean);
            result = 31 * result + (int)(temp ^ temp >>> 32);
            temp = Double.doubleToLongBits(this.intercept);
            result = 31 * result + (int)(temp ^ temp >>> 32);
            return result;
        }

        public static class Builder {
            private double[] arCoeffs = new double[0];
            private double[] maCoeffs = new double[0];
            private double[] sarCoeffs = new double[0];
            private double[] smaCoeffs = new double[0];
            private int d = 0;
            private int D = 0;
            private double mean = 0.0;

            private Builder() {
            }

            public Builder setARCoeffs(double ... arCoeffs) {
                this.arCoeffs = arCoeffs;
                return this;
            }

            public Builder setSeasonalARCoeffs(double ... sarCoeffs) {
                this.sarCoeffs = sarCoeffs;
                return this;
            }

            public Builder setMACoeffs(double ... maCoeffs) {
                this.maCoeffs = maCoeffs;
                return this;
            }

            public Builder setSeasonalMACoeffs(double ... smaCoeffs) {
                this.smaCoeffs = smaCoeffs;
                return this;
            }

            public Builder setDifferences(int d) {
                this.d = d;
                return this;
            }

            public Builder setSeasonalDifferences(int D) {
                this.D = D;
                return this;
            }

            public Builder setMean(double mean) {
                this.mean = mean;
                return this;
            }

            public ModelCoefficients build() {
                return new ModelCoefficients(this);
            }
        }
    }

    public static class ModelOrder {
        final int p;
        final int d;
        final int q;
        final int P;
        final int D;
        final int Q;
        final int constant;

        private ModelOrder(int p, int d, int q, int P, int D, int Q, boolean constant) {
            this.p = p;
            this.d = d;
            this.q = q;
            this.P = P;
            this.D = D;
            this.Q = Q;
            this.constant = constant ? 1 : 0;
        }

        private int sumARMA() {
            return this.p + this.q + this.P + this.Q;
        }

        public String toString() {
            boolean isSeasonal = this.P > 0 || this.Q > 0 || this.D > 0;
            StringBuilder builder = new StringBuilder();
            if (isSeasonal) {
                builder.append("Seasonal ");
            }
            builder.append("ARIMA (").append(this.p).append(", ").append(this.d).append(", ").append(this.q);
            if (isSeasonal) {
                builder.append(") x (").append(this.P).append(", ").append(this.D).append(", ").append(this.Q);
            }
            builder.append(") with").append(this.constant == 1 ? " a constant" : " no constant");
            return builder.toString();
        }

        public int hashCode() {
            int prime = 31;
            int result = 1;
            result = 31 * result + this.D;
            result = 31 * result + this.P;
            result = 31 * result + this.Q;
            result = 31 * result + this.constant;
            result = 31 * result + this.d;
            result = 31 * result + this.p;
            result = 31 * result + this.q;
            return result;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            ModelOrder other = (ModelOrder)obj;
            if (this.D != other.D) {
                return false;
            }
            if (this.P != other.P) {
                return false;
            }
            if (this.Q != other.Q) {
                return false;
            }
            if (this.constant != other.constant) {
                return false;
            }
            if (this.d != other.d) {
                return false;
            }
            return this.p == other.p && this.q == other.q;
        }
    }

    public static enum FittingStrategy {
        CSS,
        USS,
        ML,
        CSSML,
        USSML;

    }
}

