/*
 * Copyright (C)2016 - SMBJ Contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.hierynomus.msfscc.fileinformation;

import com.hierynomus.msdtyp.FileTime;
import com.hierynomus.msdtyp.MsDataTypes;
import com.hierynomus.protocol.commons.Charsets;
import com.hierynomus.protocol.commons.buffer.Buffer;
import com.hierynomus.protocol.commons.buffer.Buffer.BufferException;

/**
 * Class containing information about a volume
 */
public class VolumeInfo {

    /**
     * The time when the volume was created.
     */
    private final FileTime volumeCreationTime;

    /**
     * The serial number is an opaque value generated by the file system at format time, and is not necessarily related to any hardware serial number for the device on which the file system is located.
     * No specific format or content of this field is required for protocol interoperation. This value is not required to be unique.
     */
    private final int volumeSerialNumber;

    /**
     * Set to TRUE if the file system supports object-oriented file system objects; set to FALSE otherwise.
     */
    private final boolean supportsObjects;

    /**
     * A field containing the name of the volume.
     */
    private final String volumeLabel;

    /**
     * Parses the volume information from a given buffer
     * [MS-SMB2] 2.2.38 SMB2 QUERY_INFO Response, SMB2_0_INFO_FILESYSTEM/FileFsVolumeInformation
     * <p>
     * [MS-FSCC] 2.5.9 FileFsVolumeInformation for SMB2
     */
    public static VolumeInfo parseFileFsVolumeInformation(Buffer.PlainBuffer buffer) throws BufferException {
        final FileTime volumeCreationTime = MsDataTypes.readFileTime(buffer); // (8 bytes)
        final int volumeSerialNumber = buffer.readUInt32AsInt(); // 32-bit unsigned integer
        final long nameLen = buffer.readUInt32(); // 32-bit unsigned integer
        final boolean supportsObjects = buffer.readBoolean(); // boolean
        buffer.skip(1); // Reserved (1 byte)
        final String volumeLabel = buffer.readString(Charsets.UTF_16LE, (int) nameLen / 2); // variable-length Unicode field

        return new VolumeInfo(volumeCreationTime, volumeSerialNumber, supportsObjects, volumeLabel);
    }

    VolumeInfo(FileTime volumeCreationTime, int volumeSerialNumber, boolean supportsObjects, String volumeLabel) {
        this.volumeCreationTime = volumeCreationTime;
        this.volumeSerialNumber = volumeSerialNumber;
        this.supportsObjects = supportsObjects;
        this.volumeLabel = volumeLabel;
    }

    /**
     * Gets the time when the volume was created.
     *
     * @return the time when the volume was created
     */
    public FileTime getVolumeCreationTime() {
        return volumeCreationTime;
    }

    /**
     * Gets the serial number is an opaque value generated by the file system at format time, and is not necessarily related to any hardware serial number for the device on which the file system is located.
     * No specific format or content of this field is required for protocol interoperation. This value is not required to be unique.
     *
     * @return the volume serial number
     */
    public int getVolumeSerialNumber() {
        return volumeSerialNumber;
    }

    /**
     * Are object-oriented file system objects supported
     *
     * @return true if the file system supports object-oriented file system objects; false otherwise
     */
    public boolean isSupportsObjects() {
        return supportsObjects;
    }

    /**
     * Gets the field containing the name of the volume.
     *
     * @return the volume label
     */
    public String getVolumeLabel() {
        return volumeLabel;
    }

    @Override
    public String toString() {
        return "VolumeInfo{" + "volumeCreationTime=" + volumeCreationTime + ", volumeSerialNumber=" + volumeSerialNumber + ", supportsObjects=" + supportsObjects + ", volumeLabel='" + volumeLabel + '\'' + '}';
    }
}
