package com.rabbitmq.client.impl;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * This is a generic implementation of the <q>Channels</q> specification
 * in <i>Channeling Work</i>, Nov 2010 (<tt>channels.pdf</tt>).
 * <p/>
 * Objects of type <b>K</b> must be registered, with <code><b>registerKey(K)</b></code>,
 * and then they become <i>clients</i> and a queue of
 * items (type <b>W</b>) is stored for each client.
 * <p/>
 * Each client has a <i>state</i> which is exactly one of <i>dormant</i>,
 * <i>in progress</i> or <i>ready</i>. Immediately after registration a client is <i>dormant</i>.
 * <p/>
 * Items may be (singly) added to (the end of) a client&apos;s queue with <code><b>addWorkItem(K,W)</b></code>.
 * If the client is <i>dormant</i> it becomes <i>ready</i> thereby. All other states remain unchanged.
 * <p/>
 * The next <i>ready</i> client, together with a collection of its items,
 * may be retrieved with <code><b>nextWorkBlock(collection,max)</b></code>
 * (making that client <i>in progress</i>).
 * <p/>
 * An <i>in progress</i> client can finish (processing a batch of items) with <code><b>finishWorkBlock(K)</b></code>.
 * It then becomes either <i>dormant</i> or <i>ready</i>, depending if its queue of work items is empty or no.
 * <p/>
 * If a client has items queued, it is either <i>in progress</i> or <i>ready</i> but cannot be both.
 * When work is finished it may be marked <i>ready</i> if there is further work,
 * or <i>dormant</i> if there is not.
 * There is never any work for a <i>dormant</i> client.
 * <p/>
 * A client may be unregistered, with <code><b>unregisterKey(K)</b></code>, which removes the client from
 * all parts of the state, and any queue of items stored with it.
 * All clients may be unregistered with <code><b>unregisterAllKeys()</b></code>.
 * <p/>
 * <b>Concurrent Semantics</b><br/>
 * This implementation is thread-safe.
 * <p/>
 * <b>Implementation Notes</b><br/>
 * The state is, roughly, as follows:
 * <pre> pool :: <i>map</i>(K, <i>seq</i> W)
 * inProgress :: <i>set</i> K
 * ready :: <i>iseq</i> K</pre>
 * <p/>
 * where a <code><i>seq</i></code> is a sequence (queue or list) and an <code><i>iseq</i></code>
 * (<i>i</i> for <i>i</i>njective) is a sequence with no duplicates.
 * <p/>
 * <b>State transitions</b><br/><pre>
 *      finish(k)            -------------
 *             -----------> | (dormant)   |
 *            |              -------------
 *  -------------  next()        | add(item)
 * | in progress | <---------    |
 *  -------------            |   V
 *            |              -------------
 *             -----------> | ready       |
 *      finish(k)            -------------
 * </pre>
 * <i>dormant</i> is not represented in the implementation state, and adding items
 * when the client is <i>in progress</i> or <i>ready</i> does not change its state.
 * @param <K> Key -- type of client
 * @param <W> Work -- type of work item
 */
public class WorkPool<K, W> {
    private static final int MAX_QUEUE_LENGTH = 1000;

    // This is like a LinkedBlockingQueue of limited length except you can turn the limit
    // on and off. And it only has the methods we need.
    //
    // This class is partly synchronised because:
    //
    // a) we cannot make put(T) synchronised as it may block indefinitely. Therefore we
    //    only lock before modifying the list.
    // b) we don't want to make setUnlimited() synchronised as it is called frequently by
    //    the channel.
    // c) anyway the issue with setUnlimited() is not that it be synchronised itself but
    //    that calls to it should alternate between false and true. We assert this, but
    //    it should not be able to go wrong because the RPC calls in AMQChannel and
    //    ChannelN are all protected by the _channelMutex; we can't have more than one
    //    outstanding RPC or finish the same RPC twice.

    private class WorkQueue {
        private LinkedList<W> list;
        private boolean unlimited;
        private int maxLengthWhenLimited;

        private WorkQueue(int maxLengthWhenLimited) {
            this.list = new LinkedList<W>();
            this.unlimited = false; // Just for assertions
            this.maxLengthWhenLimited = maxLengthWhenLimited;
        }

        public void put(W w) throws InterruptedException {
            if (list.size() > maxLengthWhenLimited) {
                acquireSemaphore();
            }
            synchronized (this) {
                list.add(w);
            }
        }

        public synchronized W poll() {
            W res = list.poll();

            if (list.size() <= maxLengthWhenLimited) {
                releaseSemaphore();
            }

            return res;
        }

        public void setUnlimited(boolean unlimited) {
            assert this.unlimited != unlimited;
            this.unlimited = unlimited;
            if (unlimited) {
                increaseUnlimited();
            }
            else {
                decreaseUnlimited();
            }
        }

        public boolean isEmpty() {
            return list.isEmpty();
        }
    }

    /** An injective queue of <i>ready</i> clients. */
    private final SetQueue<K> ready = new SetQueue<K>();
    /** The set of clients which have work <i>in progress</i>. */
    private final Set<K> inProgress = new HashSet<K>();
    /** The pool of registered clients, with their work queues. */
    private final Map<K, WorkQueue> pool = new HashMap<K, WorkQueue>();

    // The semaphore should only be used when unlimitedQueues == 0, otherwise we ignore it and
    // thus don't block the connection.
    private Semaphore semaphore = new Semaphore(1);
    private AtomicInteger unlimitedQueues = new AtomicInteger(0);

    private void acquireSemaphore() throws InterruptedException {
        if (unlimitedQueues.get() == 0) {
            semaphore.acquire();
        }
    }

    private void releaseSemaphore() {
        semaphore.release();
    }

    private void increaseUnlimited() {
        unlimitedQueues.getAndIncrement();
        semaphore.release();
    }

    private void decreaseUnlimited() {
        unlimitedQueues.getAndDecrement();
    }

    /**
     * Add client <code><b>key</b></code> to pool of item queues, with an empty queue.
     * A client is initially <i>dormant</i>.
     * <p/>
     * No-op if <code><b>key</b></code> already present.
     * @param key client to add to pool
     */
    public void registerKey(K key) {
        synchronized (this) {
            if (!this.pool.containsKey(key)) {
                this.pool.put(key, new WorkQueue(MAX_QUEUE_LENGTH));
            }
        }
    }

    public void unlimit(K key, boolean unlimited) {
        synchronized (this) {
            WorkQueue queue = this.pool.get(key);
            if (queue != null) {
                queue.setUnlimited(unlimited);
            }
        }
    }

    /**
     * Remove client from pool and from any other state. Has no effect if client already absent.
     * @param key of client to unregister
     */
    public void unregisterKey(K key) {
        synchronized (this) {
            this.pool.remove(key);
            this.ready.remove(key);
            this.inProgress.remove(key);
        }
    }

    /**
     * Remove all clients from pool and from any other state.
     */
    public void unregisterAllKeys() {
        synchronized (this) {
            this.pool.clear();
            this.ready.clear();
            this.inProgress.clear();
        }
    }

    /**
     * Return the next <i>ready</i> client,
     * and transfer a collection of that client's items to process.
     * Mark client <i>in progress</i>.
     * <p/>
     * If there is no <i>ready</i> client, return <code><b>null</b></code>.
     * @param to collection object in which to transfer items
     * @param size max number of items to transfer
     * @return key of client to whom items belong, or <code><b>null</b></code> if there is none.
     */
    public K nextWorkBlock(Collection<W> to, int size) {
        synchronized (this) {
            K nextKey = readyToInProgress();
            if (nextKey != null) {
                WorkQueue queue = this.pool.get(nextKey);
                drainTo(queue, to, size);
            }
            return nextKey;
        }
    }

    /**
     * Private implementation of <code><b>drainTo</b></code> (not implemented for <code><b>LinkedList&lt;W&gt;</b></code>s).
     * @param deList to take (poll) elements from
     * @param c to add elements to
     * @param maxElements to take from deList
     * @return number of elements actually taken
     */
    private int drainTo(WorkQueue deList, Collection<W> c, int maxElements) {
        int n = 0;
        while (n < maxElements) {
            W first = deList.poll();
            if (first == null)
                break;
            c.add(first);
            ++n;
        }
        return n;
    }

    /**
     * Add (enqueue) an item for a specific client.
     * No change and returns <code><b>false</b></code> if client not registered.
     * If <i>dormant</i>, the client will be marked <i>ready</i>.
     * @param key the client to add to the work item to
     * @param item the work item to add to the client queue
     * @return <code><b>true</b></code> if and only if the client is marked <i>ready</i>
     * &mdash; <i>as a result of this work item</i>
     */
    public boolean addWorkItem(K key, W item) {
        WorkQueue queue;
        synchronized (this) {
            queue = this.pool.get(key);
        }
        // The put operation may block. We need to make sure we are not holding the lock while that happens.
        if (queue != null) {
            try {
                queue.put(item);
            } catch (InterruptedException e) {
                // ok
            }

            synchronized (this) {
                if (isDormant(key)) {
                    dormantToReady(key);
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Set client no longer <i>in progress</i>.
     * Ignore unknown clients (and return <code><b>false</b></code>).
     * @param key client that has finished work
     * @return <code><b>true</b></code> if and only if client becomes <i>ready</i>
     * @throws IllegalStateException if registered client not <i>in progress</i>
     */
    public boolean finishWorkBlock(K key) {
        synchronized (this) {
            if (!this.isRegistered(key))
                return false;
            if (!this.inProgress.contains(key)) {
                throw new IllegalStateException("Client " + key + " not in progress");
            }

            if (moreWorkItems(key)) {
                inProgressToReady(key);
                return true;
            } else {
                inProgressToDormant(key);
                return false;
            }
        }
    }

    private boolean moreWorkItems(K key) {
        WorkQueue leList = this.pool.get(key);
        return leList != null && !leList.isEmpty();
    }

    /* State identification functions */
    private boolean isInProgress(K key){ return this.inProgress.contains(key); }
    private boolean isReady(K key){ return this.ready.contains(key); }
    private boolean isRegistered(K key) { return this.pool.containsKey(key); }
    private boolean isDormant(K key){ return !isInProgress(key) && !isReady(key) && isRegistered(key); }

    /* State transition methods - all assume key registered */
    private void inProgressToReady(K key){ this.inProgress.remove(key); this.ready.addIfNotPresent(key); }
    private void inProgressToDormant(K key){ this.inProgress.remove(key); }
    private void dormantToReady(K key){ this.ready.addIfNotPresent(key); }

    /* Basic work selector and state transition step */
    private K readyToInProgress() {
        K key = this.ready.poll();
        if (key != null) {
            this.inProgress.add(key);
        }
        return key;
    }
}
