/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security.principal;

import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import org.apache.commons.lang3.StringUtils;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.cloudplatform.security.Audience;
import com.sap.cloud.sdk.cloudplatform.security.Authorization;

class AudienceAuthorizationUtil
{
    static Map<Audience, Set<Authorization>> getAuthorizationsByAudience(
        @Nonnull final Set<Audience> audiences,
        @Nonnull final Set<Authorization> authorizations )
    {
        final Set<Audience> modifiedAudiences;

        if( audiences.isEmpty() ) {
            modifiedAudiences =
                authorizations
                    .stream()
                    .map(Authorization::getName)
                    .map(
                        authorization -> authorization.contains(".")
                            ? authorization.substring(0, authorization.indexOf("."))
                            : authorization)
                    .map(Audience::new)
                    .collect(Collectors.toSet());
        } else {
            modifiedAudiences =
                audiences
                    .stream()
                    .map(Audience::getAudience)
                    .map(audience -> audience.contains(".") ? audience.substring(0, audience.indexOf(".")) : audience)
                    .map(Audience::new)
                    .collect(Collectors.toSet());
        }

        final Map<Audience, Set<Authorization>> authorizationsByAudience = Maps.newHashMap();

        modifiedAudiences.stream().map(Audience::getAudience).forEach(audience -> {
            final String authorizationPrefix = audience.endsWith(".") ? audience : audience + ".";

            final Set<Authorization> authorizationsForAudience =
                authorizations
                    .stream()
                    .map(Authorization::getName)
                    .filter(authorization -> authorization.startsWith(authorizationPrefix))
                    .map(authorization -> StringUtils.removeStart(authorization, audience + "."))
                    .map(Authorization::new)
                    .collect(Collectors.toSet());

            authorizationsByAudience.put(new Audience(audience), authorizationsForAudience);
        });

        return authorizationsByAudience;
    }
}
