/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security.principal;

import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.annotations.Beta;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.sap.cloud.sdk.cloudplatform.security.Audience;
import com.sap.cloud.sdk.cloudplatform.security.Authorization;
import com.sap.cloud.sdk.cloudplatform.security.principal.exception.PrincipalAttributeException;

import io.vavr.control.Try;
import lombok.Data;
import lombok.Getter;
import lombok.RequiredArgsConstructor;

/**
 * This implementation of {@link Principal} represents the information available for an authenticated entity.
 */
@Data
@RequiredArgsConstructor
public class DefaultPrincipal implements Principal
{
    /**
     * {@inheritDoc}
     */
    @Getter
    @Nonnull
    private final String principalId;

    @Nonnull
    private final Set<Authorization> localAuthorizations;

    @Nonnull
    private final Map<String, PrincipalAttribute> attributes;

    @Nonnull
    Map<Audience, Set<Authorization>> authorizationsByAudience = Maps.newHashMap();

    /**
     * Creates a new {@link DefaultPrincipal}.
     *
     * @param principalId
     *            The ID of the principal
     * @param localAuthorizations
     *            The local authorizations of the principal
     * @param allAuthorizations
     *            All authorizations of the principal
     * @param audiences
     *            The audiences of the principal
     * @param attributes
     *            The attributes of the principal
     */
    @Beta
    public DefaultPrincipal(
        @Nonnull final String principalId,
        @Nonnull final Set<Authorization> localAuthorizations,
        @Nonnull final Set<Authorization> allAuthorizations,
        @Nonnull final Set<Audience> audiences,
        @Nonnull final Map<String, PrincipalAttribute> attributes )
    {
        this.principalId = principalId;
        this.localAuthorizations = localAuthorizations;
        this.attributes = attributes;
        this.authorizationsByAudience =
            AudienceAuthorizationUtil.getAuthorizationsByAudience(audiences, allAuthorizations);
    }

    /**
     * {@inheritDoc}
     */
    @Nonnull
    @Override
    public Set<Authorization> getAuthorizations()
    {
        return Sets.newHashSet(localAuthorizations);
    }

    @Nonnull
    @Override
    public Map<Audience, Set<Authorization>> getAuthorizationsByAudience()
    {
        return Maps.newHashMap(authorizationsByAudience);
    }

    /**
     * {@inheritDoc}
     */
    @Nonnull
    @Override
    public Try<PrincipalAttribute> getAttribute( @Nonnull final String attributeName )
    {
        @Nullable
        final PrincipalAttribute attribute = attributes.get(attributeName);

        if( attribute == null ) {
            return Try
                .failure(new PrincipalAttributeException("No attribute found with name '" + attributeName + "'."));
        }

        return Try.success(attribute);
    }
}
