/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security.principal;

import java.util.concurrent.Callable;
import java.util.function.Supplier;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.sap.cloud.sdk.cloudplatform.security.principal.exception.PrincipalAccessException;
import com.sap.cloud.sdk.cloudplatform.thread.DefaultThreadContext;
import com.sap.cloud.sdk.cloudplatform.thread.Executable;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContextExecutor;
import com.sap.cloud.sdk.cloudplatform.thread.exception.ThreadContextExecutionException;
import com.sap.cloud.sdk.cloudplatform.util.FacadeLocator;

import io.vavr.control.Try;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

/**
 * Accessor for retrieving the current {@link Principal}.
 */
@NoArgsConstructor( access = AccessLevel.PRIVATE )
@Slf4j
public final class PrincipalAccessor
{
    @Nonnull
    private static Try<PrincipalFacade> principalFacade = FacadeLocator.getFacade(PrincipalFacade.class);

    /**
     * Returns the {@link PrincipalFacade} instance.
     *
     * @return The {@link PrincipalFacade} instance, or {@code null}.
     */
    @Nullable
    public static PrincipalFacade getPrincipalFacade()
    {
        return principalFacade.getOrNull();
    }

    /**
     * Returns a {@link Try} of the {@link PrincipalFacade} instance.
     *
     * @return A {@link Try} of the {@link PrincipalFacade} instance.
     */
    @Nonnull
    public static Try<PrincipalFacade> tryGetPrincipalFacade()
    {
        return principalFacade;
    }

    /**
     * Replaces the default {@link PrincipalFacade} instance.
     *
     * @param principalFacade
     *            An instance of {@link PrincipalFacade}. Use {@code null} to reset the facade.
     */
    public static void setPrincipalFacade( @Nullable final PrincipalFacade principalFacade )
    {
        if( principalFacade == null ) {
            PrincipalAccessor.principalFacade = FacadeLocator.getFacade(PrincipalFacade.class);
        } else {
            PrincipalAccessor.principalFacade = Try.success(principalFacade);
        }
    }

    /**
     * Global fallback {@link Principal}. By default, no fallback is used, i.e., the fallback is {@code null}. A global
     * fallback can be useful to ensure a safe fallback or to ease testing with a mocked principal.
     */
    @Getter
    @Setter
    @Nullable
    private static Supplier<Principal> fallbackPrincipal = null;

    /**
     * Returns the current {@code Principal}.
     *
     * @return The current {@code Principal}.
     *
     * @throws PrincipalAccessException
     *             If there is an issue while accessing the {@code Principal}.
     */
    @Nonnull
    public static Principal getCurrentPrincipal()
        throws PrincipalAccessException
    {
        return tryGetCurrentPrincipal().getOrElseThrow(failure -> {
            if( failure instanceof PrincipalAccessException ) {
                throw (PrincipalAccessException) failure;
            } else {
                throw new PrincipalAccessException("Failed to get current principal.", failure);
            }
        });
    }

    /**
     * Returns a {@link Try} of the current {@link Principal}, or, if the {@link Try} is a failure, the global fallback.
     *
     * @return A {@link Try} of the current {@link Principal}.
     */
    @Nonnull
    public static Try<Principal> tryGetCurrentPrincipal()
    {
        final Try<Principal> principalTry = principalFacade.flatMap(PrincipalFacade::tryGetCurrentPrincipal);
        if( principalTry.isSuccess() || fallbackPrincipal == null ) {
            return principalTry;
        }

        @Nullable
        final Principal fallback = fallbackPrincipal.get();
        if( fallback == null ) {
            return Try.failure(new PrincipalAccessException());
        }

        return principalTry.recover(failure -> {
            log.warn("Recovering with fallback principal: {}.", fallback, failure);
            return fallback;
        });
    }

    /**
     * Execute the given {@link Callable} on behalf of a given principal.
     *
     * @param principal
     *            The principal to execute on behalf of.
     * @param callable
     *            The callable to execute.
     *
     * @param <T>
     *            The type of the callable.
     *
     * @return The value computed by the callable.
     *
     * @throws ThreadContextExecutionException
     *             If there is an issue while running the code on behalf of the principal.
     */
    @Nullable
    public static <T> T executeWithPrincipal( @Nonnull final Principal principal, @Nonnull final Callable<T> callable )
        throws ThreadContextExecutionException
    {
        return new ThreadContextExecutor()
            .withThreadContext(new DefaultThreadContext())
            .withListeners(new PrincipalThreadContextListener(principal))
            .execute(callable);
    }

    /**
     * Execute the given {@link Executable} on behalf of a given principal.
     *
     * @param principal
     *            The principal to execute on behalf of.
     * @param executable
     *            The executable to execute.
     *
     * @throws ThreadContextExecutionException
     *             If there is an issue while running the code on behalf of the principal.
     */
    public static void executeWithPrincipal( @Nonnull final Principal principal, @Nonnull final Executable executable )
        throws ThreadContextExecutionException
    {
        executeWithPrincipal(principal, () -> {
            executable.execute();
            return null;
        });
    }

    /**
     * Execute the given {@link Callable}, using the given principal as fallback if there is no other principal
     * available.
     *
     * @param fallbackPrincipal
     *            The principal to fall back to.
     * @param callable
     *            The callable to execute.
     *
     * @param <T>
     *            The type of the callable.
     *
     * @return The value computed by the callable.
     *
     * @throws ThreadContextExecutionException
     *             If there is an issue while running the code on behalf of the principal.
     */
    @Nullable
    public static <T> T executeWithFallbackPrincipal(
        @Nonnull final Supplier<Principal> fallbackPrincipal,
        @Nonnull final Callable<T> callable )
        throws ThreadContextExecutionException
    {
        final Try<Principal> principalTry = tryGetCurrentPrincipal();

        if( principalTry.isSuccess() ) {
            try {
                return callable.call();
            }
            catch( final ThreadContextExecutionException e ) {
                throw e;
            }
            catch( final Exception e ) {
                throw new ThreadContextExecutionException(e);
            }
        }

        return executeWithPrincipal(fallbackPrincipal.get(), callable);
    }

    /**
     * Execute the given {@link Executable}, using the given principal as fallback if there is no other principal
     * available.
     *
     * @param fallbackPrincipal
     *            The principal to fall back to.
     * @param executable
     *            The executable to execute.
     *
     * @throws ThreadContextExecutionException
     *             If there is an issue while running the code on behalf of the principal.
     */
    public static void executeWithFallbackPrincipal(
        @Nonnull final Supplier<Principal> fallbackPrincipal,
        @Nonnull final Executable executable )
        throws ThreadContextExecutionException
    {
        executeWithFallbackPrincipal(fallbackPrincipal, () -> {
            executable.execute();
            return null;
        });
    }
}
