/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.security.principal;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.sap.cloud.sdk.cloudplatform.thread.Property;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContext;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContextListener;

import io.vavr.control.Try;

/**
 * Implementation of {@link ThreadContextListener} that ensures the correct initialization of {@link Principal}s when
 * working with non-container managed threads on all supported Cloud platforms.
 */
public class PrincipalThreadContextListener implements ThreadContextListener
{
    /**
     * The ThreadContext key.
     */
    public static final String PROPERTY_PRINCIPAL = PrincipalThreadContextListener.class.getName() + ":principal";

    /**
     * The {@link Principal} to be used by this listener.
     */
    @Nullable
    private final Principal principal;

    /**
     * Default constructor.
     */
    public PrincipalThreadContextListener()
    {
        principal = null;
    }

    /**
     * Constructor for providing a {@link Principal} to be returned by this listener.
     *
     * @param principal
     *            The {@link Principal} to be used by this listener.
     */
    public PrincipalThreadContextListener( @Nonnull final Principal principal )
    {
        this.principal = principal;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int getPriority()
    {
        return DefaultPriorities.PRINCIPAL_LISTENER;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public
        void
        afterInitialize( @Nonnull final ThreadContext threadContext, @Nullable final ThreadContext parentThreadContext )
    {
        final Try<Principal> principalTry;

        if( principal != null ) {
            principalTry = Try.success(principal);
        } else {
            if( parentThreadContext != null ) {
                principalTry =
                    parentThreadContext.<Principal> getProperty(PROPERTY_PRINCIPAL).map(Property::getValue).orElse(
                        PrincipalAccessor::tryGetCurrentPrincipal);
            } else {
                principalTry = PrincipalAccessor.tryGetCurrentPrincipal();
            }
        }

        threadContext.setPropertyIfAbsent(PROPERTY_PRINCIPAL, Property.ofTry(principalTry));
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public
        void
        beforeDestroy( @Nonnull final ThreadContext threadContext, @Nullable final ThreadContext parentThreadContext )
    {
        threadContext.removeProperty(PROPERTY_PRINCIPAL);
    }
}
