/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.service.prov.api;

import java.util.List;
import java.util.Map;

import com.sap.cloud.sdk.service.prov.api.exception.DatasourceException;


public interface DataSourceHandler {
     
	/**
	 * Creates an entity instance.
	 * 
	 * @param entityData Represents the entity to be created
	 * @param returnCreatedEntity Indicates whether the created entity needs to be returned
	 * @return Based on the value of <code>returnCreatedEntity</code>, either the created entity
	 * or null is returned.
	 * @throws DatasourceException If an issue with the data source occurs
	 */
	public EntityData executeInsert(EntityData entityData, boolean returnCreatedEntity) throws DatasourceException;
	
	/**
	 * Creates an entity instance.
	 * 
	 * @param pojoData Represents the entity to be created
	 * @param entityName Name of the entity
	 * @param returnCreatedEntity Indicates whether the created entity needs to be returned
	 * @return Based on the value of <code>returnCreatedEntity</code>, either the created entity
	 * or null is returned.
	 * @throws DatasourceException If an issue with the data source occurs
	 */
	<T> T executeInsert(T pojoData, String entityName, boolean returnCreatedEntity) throws DatasourceException;
	
	/**
	 * Creates an entity instance with associated entities.
	 * 
	 * @param entityData Represents the entity and its associated entities to be created
	 * @param returnCreatedEntity Indicates whether the created entity needs to be returned
	 * @return Based on the value of <code>returnCreatedEntity</code>, either the created entity
	 * or null is returned.
	 * @throws DatasourceException If an issue with the data source occurs
	 */
	public EntityData executeInsertWithAssociations(EntityData entityData, boolean returnCreatedEntity) throws DatasourceException;
	
	/**
	 * Updates an entity instance.
	 * 
	 * @param entityData Represents the entity to be updated
	 * @param returnUpdatedEntity Indicates whether the updated entity needs to be returned
	 * @return Based on the value of <code>returnUpdatedEntity</code>, either the updated entity
	 * or null is returned.
	 * @throws DatasourceException If an issue with the data source occurs
	 */
	public EntityData executeUpdate(EntityData entityData, Map<String, Object> keys, boolean returnUpdatedEntity) throws DatasourceException;
	
	/**
	 * Deletes an entity instance.
	 * 
	 * @param entityName Name of the entity to be deleted
	 * @param keys Key-value pair identifying a particular entity instance
	 * @return <code>true</code> if the entity is deleted, <code>false</code> otherwise
	 * @throws DatasourceException If an issue with the data source occurs
	 */
	public boolean executeDelete(String entityName, Map<String, Object> keys) throws DatasourceException;
	
	/**
	 * Reads an entity instance.
	 * 
	 * @param entityName Name of the entity to be read
	 * @param keys Key-value pair identifying a particular entity instance
	 * @param flattenedElementNames Names of elements that have to be read. 
	 * For elements in a structured type, the flattened name has to be given.
	 * 
	 * @return entity read from the data source
	 * @throws DatasourceException If an issue with the data source occurs
	 */
	public EntityData executeRead(String entityName, Map<String, Object> keys, List<String> flattenedElementNames) throws DatasourceException;

}
