/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.service.prov.api;

import java.sql.Time;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.annotation.JsonAutoDetect.Visibility;
import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.sap.cloud.sdk.service.prov.api.exception.DataConversionException;
import com.sap.cloud.sdk.service.prov.api.internal.CSNUtil;
import com.sap.cloud.sdk.service.prov.api.internal.DefaultEntityDataBuilder;
import com.sap.cloud.sdk.service.prov.api.util.PojoUtil;

public abstract class EntityData implements Cloneable {
	
	protected abstract Object getPropertyValue(String name);
	/**
	 * Checks whether a particular element is present.
	 *  
	 * @param name The name of element to check.
	 * @return true if the element is present, false otherwise.
	 */
	public abstract boolean contains(String name);
	
	/**
	 * Returns the value of a specific association
	 * If this method returns null, then it means that the association is not present. 
	 * 
	 * @param name Name of the association
	 * @return if association is 1:1 then the returned object is EntityData, else if association is 1:n then the returned object is a List<EntityData> 
	 */
	public abstract Object getAssociationValue(String name);
	
	/**
	 * Returns the collection of all associations.
	 * If this method returns null, it means that there are no associations for the entity. 
	 * 
	 * @return if association is 1:1 then the returned object is EntityData, else if association is 1:n then the returned object is a List<EntityData> 
	 */
	public abstract Map<String,Object> getAssociations();

	@SuppressWarnings("unchecked")
	protected static Object cloneProperty(Object property) {

		if(property instanceof Calendar)
			return ((Calendar) property).clone();
		if(property instanceof Timestamp)
			return Timestamp.from(((Timestamp)property).toInstant());
		if(property instanceof Time)
			return new java.sql.Time(((java.sql.Time)property).getTime());
		if(property instanceof java.sql.Date)
			return new java.sql.Date(((java.sql.Date)property).getTime());
		if(property instanceof Date)
			return new java.util.Date(((Date) property).getTime());
		if(property instanceof Map)
			return cloneMap((Map<String, Object>)property);
		if(property instanceof List)
			return cloneList((List<Map<String, Object>>)property);
		return property;

	}

	protected static Object cloneMap(Map<String, Object> property) {
		HashMap<String, Object> cloned = new HashMap<>();
		for(Map.Entry<String, Object> entry : property.entrySet()) {
			cloned.put(entry.getKey(), cloneProperty(property.get(entry.getKey())));
		}
		return cloned;
	}
	
	@SuppressWarnings("unchecked")
	protected static Object cloneList(List<Map<String, Object>> mapList) {
		List<Map<String, Object>> cloned = new ArrayList<>();
		for(Map<String, Object> map : mapList) {
			cloned.add((Map<String, Object>) cloneMap(map));
		}
		return cloned;
	}
	
	/**
	 * Returns the value of a particular element.
	 * 
	 * If this method returns null it can mean either of 2 things. 
	 * 1) The element is not present. 
	 * 2) The element is present but the value is explicitly set as null.
	 * 
	 * Use the contains method to check whether an element exists in the entityData.
	 * 
	 * @param name Name of the element.
	 * @return value of the element if the element is present, null otherwise.
	 */
	public Object getElementValue(String name) {
		return cloneProperty(getPropertyValue(name));
	}
	
	
	/**
	 * Returns an EntityDataBuilder instance which can be used to construct an entityData from scratch.
	 * @return An instance of EntityDataBuilder.
	 */
	public static EntityDataBuilder getBuilder() {
		return new DefaultEntityDataBuilder();
	}
	
	/**
	 * Returns an EntityDataBuilder instance which is partially initialized based on the passed entityData.
	 * This method should be used if you need to construct an entityData starting from an existing entityData.
	 * 
	 * For example:- You have an entityData for an entity Customer. But you would like to add an extra element to this entityData for the name of the Customer's company.
	 * The first step you have to do is to get a builder by passing the existing entitydata.
	 * EntityDataBuilder builder = EntityData.getBuilder(customerEntityData);
	 * 
	 * Once you have the builder you can add the extra element, by calling the addElement method.
	 * 
	 * 
	 * @param entityData The entityData from which a new EntityData is to be constructed, for example by adding an extra element.
	 * @return An instance of EntityDataBuilder which is initialized based on the passed EntityData.
	 */
	public static EntityDataBuilder getBuilder(EntityData entityData) {
		return new DefaultEntityDataBuilder(entityData);
	}
	
	/**
	 * Returns an EntityData instance which is based on a list of properties
	 * 
	 * @param propertiesMap List of properties used to construct a new instance of EntityData
	 * @param keys List of key names
	 * @param entityName Name of the newly created entity
	 * @return An instance of EntityData
	 */
	public static EntityData createFromMap(Map<String, Object> propertiesMap,List<String> keys,String entityName){
		
		@SuppressWarnings("unchecked")
		Map<String, Object> cloned = (Map<String, Object>) cloneMap(propertiesMap);		
		DefaultEntityDataBuilder entityDataBuilder = new DefaultEntityDataBuilder();
		cloned.forEach((key,value)-> entityDataBuilder.addElement(key,value));
		if(keys!=null){
		for( String key : keys){
				entityDataBuilder.addKeyElement(key,propertiesMap.get(key));
		}
		}
		return entityDataBuilder.buildEntityData(entityName);
	}
	
	/**
	 * Returns an EntityData instance which is based on a list of properties
	 * 
	 * @param propertiesMap List of properties and associations used to construct a new instance of EntityData 
	 * @param keys Map of List of key based on entity name
	 * @param entityName Fully qualified entityname along with service name
	 * @return An instance of EntityData
	 */
	@SuppressWarnings("unchecked")
	public static EntityData createFromDeepMap(Map<String, Object> propertiesMap, Map<String, List<String>> keys, String entityName) {
		int index = entityName.lastIndexOf('.');		
		String serviceName = entityName.substring(0, index);
		String parentEntityName = entityName.substring(index+1, entityName.length());
		//Parent entity association name is kept same initially for looking up from keys for entityBuilder
		//During recursion association name will get replaced based on available associations
		return createFromDeepMap(serviceName, propertiesMap, keys, parentEntityName, parentEntityName);
	}
	
	private static EntityData createFromDeepMap(String serviceName, Map<String, Object> propertiesMap, Map<String, List<String>> keys, String entityName, String associationName) {
		Map<String, Object> cloned = (Map<String, Object>) cloneMap(propertiesMap);
		DefaultEntityDataBuilder entityDataBuilder = new DefaultEntityDataBuilder();
		cloned.forEach((key,value)-> {
					if(value instanceof Map) {
						String associatedEntity = CSNUtil.getEntityName(serviceName, entityName, key);
						if(associatedEntity != null && !"".equals(associatedEntity.trim())) {
							entityDataBuilder.addAssociationElement(key, createFromDeepMap(serviceName, (Map<String, Object>)value, keys, associatedEntity, key));
						}else {
							throw new IllegalStateException("Association with the name "+key+" is not found.");
						}
					}else if(value instanceof List){
						String associatedEntity = CSNUtil.getEntityName(serviceName, entityName, key);						
						List<Map<String, Object>> valueList = (List<Map<String, Object>>)value;
						List<EntityData> edList = new ArrayList<>();
						for(Map<String, Object> valueMap : valueList){
							edList.add(createFromDeepMap(serviceName, (Map<String, Object>)valueMap, keys, associatedEntity, key));
						}
						entityDataBuilder.addAssociationElement(key, edList);
					}else{
						entityDataBuilder.addElement(key,value);
					}
				}
			);

		if(keys.get(associationName) != null){
			for( String key : keys.get(associationName)){
				entityDataBuilder.addKeyElement(key, propertiesMap.get(key));
			}
		}
		return entityDataBuilder.buildEntityData(associationName);
	}
	
	/**
	 * Returns the map of all the properties of a given EntityData instance. * 
	 *
	 * @return A map instance containing all the properties of an EntityData instance.
	 */
	public abstract Map<String, Object> asMap();
	
	/**
	 * Returns an EntityData instance which is based on the POJO passed as parameter
	 * 
	 * @param data A POJO instance 
	 * @param entityName Name of the newly created entity
	 * @return An instance of EntityData
	 */
	@SuppressWarnings("unchecked")
	public  static EntityData createFrom(Object data, String entityName){
		Map<String, Object> tempdata = getMapFromSinglePojo(data);
		List<String> keys = PojoUtil.getKeyProperties(data);
		if(keys.isEmpty()){
			throw new DataConversionException("The POJO class is not having any keys");
		}
		Map<String, Object> copyData = (Map<String, Object>) cloneMap(tempdata);
		if (copyData.size() > 0)
			return EntityData.createFromMap(copyData,keys, entityName);
		else
			return null;

	}

	private static Map<String, Object> getMapFromSinglePojo(
			Object pojoData) {
		ObjectMapper mapper = new ObjectMapper();
		/*-- Ignore Null Values --*/
		mapper.configure(SerializationFeature.WRITE_NULL_MAP_VALUES, false);
		/*-- do not fail on empty beans  --*/
		mapper.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false);
		/*-- First: hide all elements in the class ; make property,getter,setter hidden  --*/
		mapper.setVisibility(PropertyAccessor.ALL, Visibility.NONE);
		/*-- Second: now only make the properties discoverable   --*/
		mapper.setVisibility(PropertyAccessor.FIELD, Visibility.ANY);
		mapper.setSerializationInclusion(Include.NON_NULL);
		@SuppressWarnings("unchecked")	
		Map<String, Object> pojoInMap = mapper.convertValue(pojoData,
				HashMap.class);

		return pojoInMap;
	}
	
	
	/**
	 * Returns a POJO based on EntityData
	 * 
	 * @param T A POJO class 
	 * @return A POJO based on EntityData
	 */
	public abstract <T> T as (Class<T>  t)throws DataConversionException  ; 
	
	/**
	 * Returns the properties of the entity.
	 * @return A Map containing the property names and values of an entity
	 */
	public abstract Map<String, Object> getMap();

}
