package com.sap.cloud.sdk.service.prov.api.connection;

import java.io.ByteArrayOutputStream;
import java.sql.Connection;
import java.time.Instant;

import javax.servlet.ServletRequestEvent;
import javax.servlet.ServletRequestListener;
import javax.servlet.annotation.WebListener;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cloud.sdk.service.prov.api.transaction.impl.TransactionHandler;
import com.sap.cloud.sdk.service.prov.api.util.RequestProcessingHelper;

/**
 * Utility class for storing thread specific objects with the guarantee that these objects will be
 * removed at the end of the http call. (So that same objects are not accidentally re-used when
 * different http calls come in the same thread)
 */
@WebListener
public class ThreadSafeObjectStore implements ServletRequestListener {

final static Logger logger = LoggerFactory.getLogger(ThreadSafeObjectStore.class);
	
	static ThreadLocal<Connection> connectionForThread = new ThreadLocal<>();
	static ThreadLocal<Instant> metadataLastModifiedTime = new ThreadLocal<>();
	static ThreadLocal<Connection> connectionForCDSHandlerFactory = new ThreadLocal<>();
	static ThreadLocal<HashedData> csnForThread = new ThreadLocal<>();
	static ThreadLocal<ByteArrayOutputStream> metadataForThread = new ThreadLocal<>();
	private static ThreadLocal<String> tenantIdForExtensibility = new ThreadLocal<>();
	
	public static Connection getConnection() {
		try {
			if(connectionForThread.get() == null || connectionForThread.get().isClosed() == true)
				initializeConnection();
		}catch(Exception e) {
			logger.error("error getting connection from provider",e);
		}
		return connectionForThread.get();
	}

	private static void initializeConnection() {
		DataSourceParams connectionProvider = RequestProcessingHelper.getConnectionProvider();
		if(connectionProvider!=null && connectionProvider instanceof JDBCDataSourceParams) {
			connectionForThread.set(((JDBCDataSourceParams) connectionProvider).getConnection());
			/*
			 * We are currently a little uncertain as to the best way to set the 
			 * user information. Hence not setting anything until the uncertainty
			 * is gone.
			 */
		}
	}
	
	public static Connection getConnectionForCDSHandlerFactory() {
		
		try {
			if(connectionForCDSHandlerFactory.get() == null || connectionForCDSHandlerFactory.get().isClosed() == true) {
				DataSourceParams connectionProvider = RequestProcessingHelper.getConnectionProvider();
				if(connectionProvider!=null && connectionProvider instanceof JDBCDataSourceParams) {
					connectionForCDSHandlerFactory.set(((JDBCDataSourceParams) connectionProvider).getConnection());
				}
			}
		}catch(Exception e) {
			logger.error("error getting connection from provider",e);
		}
		return connectionForCDSHandlerFactory.get();
	}

	public static void setCsnForCurrentThread(HashedData csn) {
			csnForThread.set(csn);
			logger.debug("Csn is being set for the current thread.");
	}
	
	public static HashedData getCsnForCurrentThread() {
		return csnForThread.get();
	}
	
	public static Instant getMetadataLastModifiedTime() {
		return metadataLastModifiedTime.get();
	}
	
	public static void setMetadataLastModifiedTime(Instant metadataLastModified) {
		metadataLastModifiedTime.set(metadataLastModified);
	}
	
	public static ByteArrayOutputStream getMetadataForCurrentThread() {
		return metadataForThread.get();
	}
	
	public static void setMetadataForCurrentThread(ByteArrayOutputStream baos) {
		metadataForThread.set(baos);
	}
	
	public static String getTenantIdForExtensibility() {
		return tenantIdForExtensibility.get();
	}
	
	public static void setTenantIdForExtensibility(String tenantId) {
		tenantIdForExtensibility.set(tenantId);
	}
		
	private static void removeConnectionForCurrentThread() {
		if(connectionForCDSHandlerFactory.get() != null) {
			Connection conn = connectionForCDSHandlerFactory.get();
			try {
				if(conn!=null && !conn.isClosed())
				   conn.close();
			} catch (Exception e) {
				logger.error("Error while closing connection", e);
			}
			connectionForCDSHandlerFactory.remove();
		}
	}

	@Override
	public void requestDestroyed(ServletRequestEvent sre) {
		/*
		 * We remove the connection and not just close it because,
		 * the current thread may not die with the current request
		 * if thread is being pooled by the servlet container. 
		 * We do not want that the same thread when used in 
		 * different requests get the same connection.
		 */
		try {
			if(connectionForThread.get() != null)
				TransactionHandler.cleanupTransaction();
		}catch(Exception e) { logger.error(e.getMessage(), e);}
		removeConnectionForCurrentThread();
		connectionForThread.remove();
		csnForThread.remove();
		metadataForThread.remove();
		metadataLastModifiedTime.remove();
		tenantIdForExtensibility.remove();
	}
	

	@Override
	public void requestInitialized(ServletRequestEvent sre) {
		//Nothing to be done right now
		
	}
	
}
