/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.service.prov.api.filter;

import java.util.ArrayList;
import java.util.List;

import com.sap.cloud.sdk.service.prov.api.filter.ExpressionOperatorTypes.FUNCTION;
import com.sap.cloud.sdk.service.prov.api.filter.ExpressionOperatorTypes.NODE_KIND;
import com.sap.cloud.sdk.service.prov.api.filter.ExpressionOperatorTypes.OPERATOR;

/**
 * Provides methods for working with expression trees.
 */
public class ExpressionAPIUtility {
	private List<ExpressionNode> nodes = null;

	/**
	 * Gets all the leaf units of the expression tree. A leaf unit is a node with at least one leaf node as its child.
	 * For example, <code>Name eq 'John'</code> is a leaf unit within a bigger filter expression tree.
	 * 
	 * @param filterTreeNode Represents the filter expression tree
	 * @return A list of leaf units from the filter expression tree
	 */
	public List<ExpressionNode> getLeafUnits(Expression filterTreeNode) {
		if (nodes == null)
			nodes = new ArrayList<>();
		traverseFilterTree((ExpressionNode) filterTreeNode, true);
		return nodes;

	}

	private void traverseFilterTree(ExpressionNode filterTreeNode, boolean leafUnitFlag) {
		List<ExpressionNode> childrenNodes = getChildren(filterTreeNode);

		if (leafUnitFlag) {
			if (isLeafUnit(childrenNodes)) {
				nodes.add(filterTreeNode);
			}
		} else
			nodes.add(filterTreeNode);

		if (childrenNodes == null || childrenNodes.isEmpty())
			return;
		childrenNodes.forEach(node -> traverseFilterTree(node, leafUnitFlag));
	}

	private boolean isLeafUnit(List<ExpressionNode> filterTreeNodes) {
		// LeafUnit = LeafUnit is a FilterNode which has at least one Leaf
		// Node(i.e. Simple Property /Literal)
		// if at least one child is a property or a literal

		if (filterTreeNodes == null || filterTreeNodes.isEmpty())
			return false;
		return filterTreeNodes.stream().filter(node -> {
			return ((node.getKind() == NODE_KIND.PROPERTY || node.getKind() == NODE_KIND.LITERAL) ? true : false);
		}).findAny().isPresent();
	}

	/**
	 * Gets all the nodes from a filter expression tree.
	 * @param filterTree An <code>ExpressionNode</code> object which is the root node of a filter expression tree
	 * @return A list of <code>ExpressionNode</code> objects containing all nodes of a given filter expression tree
	 */
	public List<ExpressionNode> getNodes(Expression filterTreeNode) {
		if (nodes == null)
			nodes = new ArrayList<>();
		traverseFilterTree((ExpressionNode) filterTreeNode, false);
		return nodes;
	}

	/**
	 * Gets all the children of a filter expression tree. Each child could be a node or an entire filter expression tree.
	 * @param filterTreeNode A filter expression tree 
	 * @return A list of <code>ExpressionNode</code> objects containing the children of the filter expression
	 */
	public List<ExpressionNode> getChildren(ExpressionNode filterTreeNode) {
		List<ExpressionNode> childrenNodes = new ArrayList<>();
		if (filterTreeNode instanceof BinaryExpressionNode) {
			childrenNodes.add(((BinaryExpressionNode) filterTreeNode).getFirstChild());
			childrenNodes.add(((BinaryExpressionNode) filterTreeNode).getSecondChild());
		} else if (filterTreeNode instanceof UnaryExpressionNode) {
			childrenNodes.add(((UnaryExpressionNode) filterTreeNode).getChild());
		} else if (filterTreeNode instanceof FunctionNode) {
			childrenNodes.addAll(((FunctionNode) filterTreeNode).getParameters());
		} else
			childrenNodes = null;

		return childrenNodes;
	}

	/**
	 * Checks whether an operator is used in a filter expression.
	 * @param currentNode <code>ExpressionNode</code> containing the filter expression
	 * @param operator Operator being searched in the filter expression
	 * @return A boolean indicating whether the operator is present in the filter expression
	 */
	public boolean hasOperator(ExpressionNode currentNode, OPERATOR operator) {
		if (currentNode instanceof BinaryExpressionNode) {
			return ((BinaryExpressionNode) currentNode).getOperator().equals(operator.name());

		} else if (currentNode instanceof UnaryExpressionNode) {

			return ((UnaryExpressionNode) currentNode).getOperator().equals(operator.name());
		} else
			return false;

	}
	 

	/**
	 * Checks whether a function is used in a filter expression. Examples of functions include Concat, Contains, and so on.
	 * @param currentNode <code>ExpressionNode</code> containing the filter expression
	 * @param function Function being searched in the filter expression
	 * @return A boolean indicating whether the function is present in the filter expression
	 */

	public boolean hasFunction(ExpressionNode currentNode, FUNCTION function) {

		  if (currentNode instanceof FunctionNode) {
			return hasStringFunction((FunctionNode) currentNode, function);
		  }

		else
			return false;
	}

	private boolean hasStringFunction(FunctionNode currentNode, FUNCTION function) {
		return currentNode.getFunctionName().equals(function.toString());
	}

 
}
