/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.service.prov.api.filter;

import java.util.List;

import com.sap.cloud.sdk.service.prov.api.filter.ExpressionOperatorTypes.FUNCTION;
import com.sap.cloud.sdk.service.prov.api.filter.ExpressionOperatorTypes.NODE_KIND;
import com.sap.cloud.sdk.service.prov.api.filter.ExpressionOperatorTypes.OPERATOR;

 
public interface ExpressionNode extends Expression{
	
	/**
	 * Checks whether the current <code>ExpressionNode</code> is a leaf unit. A leaf unit is a node with at least one leaf node as its child.  
	 * @return A <code>boolean</code> value indicating whether the <code>ExpressionNode</code> is a leaf unit
	 */
	boolean isLeafUnit();
	
	/**
	 * Returns the type of the filter expression. Some examples of filter expression types are unary, binary, literal, and so on.
	 * @return A enumeration value of type <code>NODE_KIND</code> indicating the type of the current node
	 */
	NODE_KIND getKind();
	
	/**
	 * Checks whether the current node is a literal node.
	 * @return A <code>boolean</code> value indicating whether the current node is a literal
	 */
	default boolean isLiteralNode() {
		if ((this instanceof LiteralNode )) return true; else return false;
	}
	
	/**
	 * Checks whether the current node is a property. For example, in the expression <code>Name eq 'John'</code>, <code>Name</code> is the property.
	 * @return A <code>boolean</code> value indicating whether the current node is a property
	 */
	
	default boolean isPropertyNode() {
		if ((this instanceof PropertyNode )) return true; else return false;
	}
	
	/**
	 * Checks whether the current <code>ExpressionNode</code> represents a filter on a certain property. 
	 * @param propertyName Name of the property 
	 * @return A <code>boolean</code> value indicating whether the current <code>ExpressionNode</code> represents a filter on the specified property
	 */
	default boolean hasProperty(String propertyName) {
		List<ExpressionNode> children = new ExpressionAPIUtility().getChildren(this);
		if(children == null) return false;
		return children.stream().filter(node -> {
			return ((node.getKind() == NODE_KIND.PROPERTY) ? true : false);
		}).filter(node -> {
			PropertyNode memberNode = (PropertyNode) node;
			return memberNode.getPath().equals(propertyName);
		}).findAny().isPresent();
	}
	
	/**
	 * Checks whether an operator is used in the current filter expression.
	 * @param operator Operator being searched in the filter expression
	 * @return A boolean indicating whether the operator is present in the current filter expression
	 */
	default boolean hasOperator( OPERATOR operator) {
		return new ExpressionAPIUtility().hasOperator(this, operator);
	}
	
	/**
	 * Checks whether the current filter expression contains a function.
	 * @param function Function being searched in the filter expression 
	 * @return A <code>boolean</code> value indicating whether the current filter expression contains the specified function
	 */
	default boolean hasFunction(FUNCTION function)  
	{		return new ExpressionAPIUtility().hasFunction(this, function);
	}	

}
