/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.service.prov.api.internal;

import static com.sap.cloud.sdk.service.prov.api.util.GenericUtilityMethods.getOutputStreamFromInputStream;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.commons.codec.digest.DigestUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.TextNode;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.sap.cloud.sdk.service.prov.api.connection.HashedData;
import com.sap.cloud.sdk.service.prov.api.connection.HashedDataImpl;
import com.sap.cloud.sdk.service.prov.api.connection.ThreadSafeObjectStore;
import com.sap.cloud.sdk.service.prov.api.security.AuthorizationDetails;
import com.sap.cloud.sdk.service.prov.model.internal.CSNConstants;


/**
 *
 * Utility class for obtaining information from the CSN(Internal).
 *
 */
public class CSNUtil {
	private static final String DEFINITIONS = "definitions";
	private static final String ELEMENTS = "elements";
	private static final String ON = "on";
	private static final String REF = "ref";
	private static final String NO_REF_CONSTRAINT = "Unable to create with navigation because the corresponding referential constraint is missing in the csn file";
	private static final String PARAMS = "params";
	private static final String TYPE = "type";
	private static final String SQL_MAPPING = "@sql_mapping";
	private static final String ODATA_INSERTED = "@odata.on.insert";
	private static final String ODATA_UPDATED = "@odata.on.update";
	private static final String CDS_INSERTED = "@cds.on.insert";
	private static final String CDS_UPDATED = "@cds.on.update";
	private static final String ODATA_COMPUTED = "@Core.Computed";
	private static final String ODATA_IMMUTABLE="@Core.Immutable";
	private static final String ORDER_DEFAULT = "@Order.default";
	private static final String ODATA_SEARCHABLE = "@Capabilities.SearchRestrictions.Searchable";
	private static final String ODATA_DEFAULT_SEARCH_ELEMENT = "@Search.defaultSearchElement";
	private static final String ODATA_DEFAULT_SEARCH_ELEMENT_WITH_FUZZY = "@Search.fuzzinessThreshold";
	private static final String ODATA_NOW = "now";
	private static final String ODATA_USER = "user";
	private static final String ODATA_HASH = "#";
	private static final String CDS_PERSISTENCE_NAME = "@cds.persistence.name";
	private static final String CSN_NOT_FOUND = "CSN file not found.";
	private static final String DRAFT_ENABLED = "@odata.draft.enabled";
	private static final String KIND = "kind";
	private static final String RESTRICT = "@restrict";
	private static final String GRANT = "grant";
	private static final String CDS_QUERY_LIMIT = "@cds.query.limit";
	private static final String CDS_CONSISTENT_PAGING = "@cds.consistent.paging";
	private static final String SNAPSHOT_STORAGE_DAYS = "@snapshot.storage.days";
	private static volatile JsonNode staticCsn;
	private static volatile HashedData staticHashedCSN;
	private static boolean initialized = false;
	private static final String DRAFT_NODE = "@Common.DraftNode.PreparationAction";
	private static final String TARGET = "target";
	private static final String REQUIRES = "requires";
	private static final String LOCALIZED = "localized";
	private static final String CDS_TEMPORAL = "@cds.temporal";
	private static final String CDS_VALID = "@cds.valid";
	private static final String CDS_VALID_FROM = "@cds.valid.from";
	private static final String CDS_AUTOEXPOSE = "@cds.autoexpose";
	private static final String CDS_AUTOEXPOSED = "@cds.autoexposed";
	private static final String DELTA_ENABLED = "@odata.delta.enabled";
	private static final String DOT = ".";
	private static final String AGGREGATED_ENTITY = "@Aggregation.ApplySupported.PropertyRestrictions";
	private static final String CDS_COMPOSITION = "cds.Composition";
	private static final String CDS_ASSOCIATION = "cds.Association";
	private static final ObjectMapper objectMapper = new ObjectMapper();
	private static final String ODATA_ETAG = "@odata.etag";
	private static final String CDS_ETAG = "@cds.etag";
	public static final String entitySemantics = "@PersonalData.EntitySemantics";
	public static final String auditLogOperation = "@AuditLog.Operation";
	public static final String Semantics = "@PersonalData.Semantics";
	public static final String dataChangeLogOperation = "@PersonalDataChangeLog.Operation";
	private static final String VIRTUAL = "virtual";
	static Logger logger = LoggerFactory.getLogger(CSNUtil.class);
	private static final LoadingCache<HashedData, JsonNode> parsedCsnModels = CacheBuilder.newBuilder().maximumSize(1000)
			.build(new CacheLoader<HashedData, JsonNode>() {
				public JsonNode load(HashedData key) throws IOException {
					return objectMapper.readTree(key.getData());
				}
			});

	public static boolean isCreatable(String serviceName, String entityName) throws CSNNotSetException {
		JsonNode csn = getCsn();
		if (csn == null)
			return true;
		JsonNode insertable = csn.get(DEFINITIONS).get(serviceName + "." + entityName)
				.get("@Capabilities.InsertRestrictions.Insertable");
		return insertable == null || insertable.asText().equals("true");
	}

	public static InputStream getCsnAsInputStream() throws JsonProcessingException {
		return new ByteArrayInputStream(objectMapper.writeValueAsBytes(getCsn()));
	}
	
	public static InputStream getCsnAsInputStream(JsonNode csn) throws JsonProcessingException {
		return new ByteArrayInputStream(objectMapper.writeValueAsBytes(csn));
	}

	private static void checkIfCSNInitialized() throws CSNNotSetException {
		if (staticCsn == null) {
			// The below code is synchronized to avoid the race condition
			// when checkIfCSNInitialized is triggered by multiple threads
			synchronized (CSNUtil.class) {
				if (staticCsn == null) { // Again we check if the CSN is initialized for edge case of two threads
					// waiting for the locks to be released
					if (initialized) {
						logger.error("Already tried to initialize the CSN and the initialization failed");
						throw new CSNNotSetException();
					}
					else {
						initialized = true;
						logger.debug("Finding local static CSN for initialization");
						try (InputStream inputStreamCSN = CSNUtil.class.getClassLoader().getResourceAsStream("edmx/csn.json");) {
							if (inputStreamCSN == null) {
								logger.error("static csn file not found");
								throw new CSNNotSetException();
							}
							logger.debug("Starting local static CSN initialization");
							byte[] csnData = getOutputStreamFromInputStream(inputStreamCSN).toByteArray();
							String hash = DigestUtils.shaHex(csnData);
							staticHashedCSN = new HashedDataImpl(hash, csnData);

							staticCsn = objectMapper.readTree(csnData);
							parsedCsnModels.put(staticHashedCSN, staticCsn);
							logger.debug("Local static CSN initialization successfull");
						} catch (IOException e) {
							logger.error("Local static CSN initialization failed",e);
							throw new CSNNotSetException();
						}
					}
				}
			}
		}
	}

	public static boolean isUpdatable(String serviceName, String entityName) throws CSNNotSetException {
		JsonNode csn = getCsn();
		if (csn == null)
			return true;
		JsonNode updatable = csn.get(DEFINITIONS).get(serviceName + "." + entityName)
				.get("@Capabilities.UpdateRestrictions.Updatable");
		return updatable == null || updatable.asText().equals("true");
	}

	public static boolean isDeletable(String serviceName, String entityName) throws CSNNotSetException {
		JsonNode csn = getCsn();
		if (csn == null)
			return true;
		JsonNode deletable = csn.get(DEFINITIONS).get(serviceName + "." + entityName)
				.get("@Capabilities.DeleteRestrictions.Deletable");
		return deletable == null || deletable.asText().equals("true");

	}

	public static Map<String, AdminDataAnnotation> getAdminDataMapForCreate(String serviceName, String entityName,
																			boolean userContextPresent) {
		JsonNode csn = getCsn();
		if (csn == null)
			return new HashMap<>();
		Map<String, AdminDataAnnotation> adminDataMap = new HashMap<>();
		if ((csn.get(DEFINITIONS).get(serviceName + "." + entityName)) != null){
			JsonNode node = csn.get(DEFINITIONS).get(serviceName + "." + entityName).get(ELEMENTS);
			Iterator<String> it = node.fieldNames();
			while (it.hasNext()) {
				String colName = it.next();
				/*
				 * We check the core.computed annotation first, because core.computed has less
				 * precedence over other admin data annotations, and this way, during the
				 * further if checks, if its found that there are other annotations
				 * like @odata.on.insert, these will override the core.computed part.
				 */
				JsonNode insertNode = node.get(colName).get(ODATA_INSERTED);
				if (insertNode != null && insertNode.get(ODATA_HASH) != null) {
					adminDataMap.put(colName,
							getAdminDataAnnotation(insertNode.get(ODATA_HASH).asText(), userContextPresent));
				}
				// Now try with Prefix CDS
				JsonNode cdsinsertNode = node.get(colName).get(CDS_INSERTED);
				if (cdsinsertNode != null && cdsinsertNode.get(ODATA_HASH) != null) {
					adminDataMap.put(colName,
							getAdminDataAnnotation(cdsinsertNode.get(ODATA_HASH).asText(), userContextPresent));
				}

				JsonNode updateNode = node.get(colName).get(ODATA_UPDATED);
				if (updateNode != null && updateNode.get(ODATA_HASH) != null) {
					adminDataMap.put(colName,
							getAdminDataAnnotation(updateNode.get(ODATA_HASH).asText(), userContextPresent));
				}
				// Now try with Prefix CDS
				JsonNode cdsupdateNode = node.get(colName).get(CDS_UPDATED);
				if (cdsupdateNode != null && cdsupdateNode.get(ODATA_HASH) != null) {
					adminDataMap.put(colName,
							getAdminDataAnnotation(cdsupdateNode.get(ODATA_HASH).asText(), userContextPresent));
				}
			}
		}
		return adminDataMap;
	}


	public static Map<String, AdminDataAnnotation> getAdminDataMapForDeepCreate(String serviceName,String parentEntityName, String assoEntityName,
            String entityName,boolean userContextPresent) {
		JsonNode csn = getCsn();
		if (csn == null)
			return new HashMap<>();
		Map<String, AdminDataAnnotation> adminDataMap = new HashMap<>();
		if ((csn.get(DEFINITIONS).get(serviceName + "." + entityName)) != null){
			JsonNode entityNode = csn.get(DEFINITIONS).get(serviceName + "." + parentEntityName).get(ELEMENTS).get(assoEntityName);
			if (entityNode != null){
				JsonNode targetNode = entityNode.get("target");
				JsonNode childNode = csn.get(DEFINITIONS).get(targetNode.asText());
				if (childNode != null){
					JsonNode node = childNode.get(ELEMENTS);

					Iterator<String> it = node.fieldNames();
					while (it.hasNext()) {
						String colName = it.next();
						JsonNode insertNode = node.get(colName).get(ODATA_INSERTED);
						if (insertNode != null && insertNode.get(ODATA_HASH) != null) {
							adminDataMap.put(colName,
									getAdminDataAnnotation(insertNode.get(ODATA_HASH).asText(), userContextPresent));
						}
						//Now try with Prefix CDS
						JsonNode cdsinsertNode = node.get(colName).get(CDS_INSERTED);
						if (cdsinsertNode != null && cdsinsertNode.get(ODATA_HASH) != null) {
							adminDataMap.put(colName,
									getAdminDataAnnotation(cdsinsertNode.get(ODATA_HASH).asText(), userContextPresent));
						}


						JsonNode updateNode = node.get(colName).get(ODATA_UPDATED);
						if (updateNode != null && updateNode.get(ODATA_HASH) != null) {
							adminDataMap.put(colName,
									getAdminDataAnnotation(updateNode.get(ODATA_HASH).asText(), userContextPresent));
						}
						//Now try with Prefix CDS
						JsonNode cdsupdateNode = node.get(colName).get(CDS_UPDATED);
						if (cdsupdateNode != null && cdsupdateNode.get(ODATA_HASH) != null) {
							adminDataMap.put(colName,
									getAdminDataAnnotation(cdsupdateNode.get(ODATA_HASH).asText(), userContextPresent));
						}
					}
				}
			}
		}
		return adminDataMap;
	}

	public static Map<String, AdminDataAnnotation> getAdminDataMapForUpdate(String serviceName, String entityName,
																			boolean userContextPresent) {
		JsonNode csn = getCsn();
		if (csn == null)
			return new HashMap<>();
		Map<String, AdminDataAnnotation> adminDataMap = new HashMap<>();
		if ((csn.get(DEFINITIONS).get(serviceName + "." + entityName)) != null){
			JsonNode node = csn.get(DEFINITIONS).get(serviceName + "." + entityName).get(ELEMENTS);
			Iterator<String> it = node.fieldNames();
			while (it.hasNext()) {
				String colName = it.next();
				JsonNode updateNode = node.get(colName).get(ODATA_UPDATED);
				if (updateNode != null && updateNode.get(ODATA_HASH) != null) {
					adminDataMap.put(colName,
							getAdminDataAnnotation(updateNode.get(ODATA_HASH).asText(), userContextPresent));
				}

				JsonNode cdsupdateNode = node.get(colName).get(CDS_UPDATED);
				if (cdsupdateNode != null && cdsupdateNode.get(ODATA_HASH) != null) {
					adminDataMap.put(colName,
							getAdminDataAnnotation(cdsupdateNode.get(ODATA_HASH).asText(), userContextPresent));
				}

				JsonNode createNode = node.get(colName).get(ODATA_INSERTED);
				if (createNode != null && createNode.get(ODATA_HASH) != null) {
					adminDataMap.put(colName, AdminDataAnnotation.NO_CHANGE);
				}
				JsonNode cdscreateNode = node.get(colName).get(CDS_INSERTED);
				if (cdscreateNode != null && cdscreateNode.get(ODATA_HASH) != null) {
					adminDataMap.put(colName, AdminDataAnnotation.NO_CHANGE);
				}
				JsonNode computeNode = node.get(colName).get(ODATA_IMMUTABLE);
				if (computeNode != null && computeNode.get(ODATA_HASH) != null  && computeNode.get(ODATA_HASH).asText().equals("false")) {
					adminDataMap.put(colName,
							getAdminDataAnnotation(computeNode.get(ODATA_HASH).asText(), false));
				}
				else if (computeNode != null &&  computeNode.asBoolean()){
					adminDataMap.put(colName,AdminDataAnnotation.IMMUTABLE_FIELD);
				}
			}
		}
		return adminDataMap;
	}

	public static List<String> getSearchableProperties(String serviceName, String entityName) {
		JsonNode csn = getCsn();
		List<String> searchableProperties = new ArrayList<>();

		if (csn != null) {
			JsonNode entityNode = csn.get(DEFINITIONS).get(serviceName + "." + entityName);

			if (entityNode != null && entityNode.get(ODATA_SEARCHABLE) != null
					&& entityNode.get(ODATA_SEARCHABLE).asBoolean()) {
				JsonNode propertyNode = entityNode.get(ELEMENTS);
				Iterator<String> it = propertyNode.fieldNames();
				while (it.hasNext()) {
					String colName = it.next();
					JsonNode defaultSearchNode = propertyNode.get(colName).get(ODATA_DEFAULT_SEARCH_ELEMENT);
					if (defaultSearchNode != null && defaultSearchNode.asBoolean()) {
						searchableProperties.add(colName);
					}

				}
				return searchableProperties.isEmpty() ? null : searchableProperties;
			}
		}
		return null;
	}

	/*
	 * Return a map with key as properties or column name and values as the
	 * fuzziness threshold provided in the  service.csd
	 */

	public static Map<String, Double> getSearchablePropertiesWithFuzzy(String serviceName, String entityName) {
		Map<String, Double> searchableFuzzyProperties = new HashMap<>();
		JsonNode csn = getCsn();

		if (csn != null) {
			JsonNode entityNode = csn.get(DEFINITIONS).get(serviceName + "." + entityName);
			if (entityNode != null && entityNode.get(ODATA_SEARCHABLE) != null
					&& entityNode.get(ODATA_SEARCHABLE).asBoolean()) {
				JsonNode propertyNode = entityNode.get(ELEMENTS);
				Iterator<String> it = propertyNode.fieldNames();
				while (it.hasNext()) {
					String colName = it.next();
					JsonNode defaultSearchNode = propertyNode.get(colName).get(ODATA_DEFAULT_SEARCH_ELEMENT_WITH_FUZZY);
					if (defaultSearchNode != null) {
						searchableFuzzyProperties.put(colName.toLowerCase(),
								Double.parseDouble(defaultSearchNode.asText()));
					}

				}
				return searchableFuzzyProperties.isEmpty() ? null : searchableFuzzyProperties;
			}

			return null;
		}

		return null;
	}

	/*
	 * Returns all column name based on serviceName and entityName Reads details
	 * from csn.json
	 */
	public static Set<String> getProperties(String serviceName, String entityName) {
		LinkedHashSet<String> properties = new LinkedHashSet<>();
		JsonNode csn = getCsn();

		if (csn != null) {
			JsonNode entityNode = csn.get(DEFINITIONS).get(serviceName + "." + entityName);
			if (entityNode != null) {
				JsonNode propertyNode = entityNode.get(ELEMENTS);
				Iterator<String> it = propertyNode.fieldNames();
				while (it.hasNext()) {
					String colName = it.next();
					String elementType = propertyNode.get(colName).get("type").asText();
					if (elementType != null && !elementType.contains("Association")
							&& !elementType.contains("Composition")) {
						properties.add(colName);
					}
				}
				return properties.isEmpty() ? null : properties;
			}
		}

		return properties;
	}

	private static AdminDataAnnotation getAdminDataAnnotation(String annotation, boolean userContextPresent) {
		switch (annotation) {
			case ODATA_NOW:
				return AdminDataAnnotation.NOW;
			case ODATA_USER: {
				if (userContextPresent)
					return AdminDataAnnotation.APPLICATION_USER;
				else
					return AdminDataAnnotation.DB_USER;
			}
			default:
				return null;
		}
	}

	public static boolean isPlainSqlMapping() {
		JsonNode csn = getCsn();
		if (csn == null)
			return false;
		String sqlMapping = csn.get(SQL_MAPPING) == null ? null : csn.get(SQL_MAPPING).asText();

		if (sqlMapping != null)
			return sqlMapping.equals("plain");
		else {
			String csnVersion;
			if(csn.get("version") != null)
				csnVersion = csn.get("version").get("csn").asText();
			else
				csnVersion = csn.get("$version").asText();
			logger.debug("Csn version is " + csnVersion);

			int versionNumber = Integer.parseInt(csnVersion.replaceAll("\\.", ""));
			return versionNumber >= 10;
		}
	}

	public static boolean isHdbcdsMapping() {

		JsonNode csn = getCsn();
		if (csn == null)
			return false;
		String sqlMapping = csn.get(SQL_MAPPING) == null ? null : csn.get(SQL_MAPPING).asText();
		return sqlMapping != null && sqlMapping.equals("hdbcds");

	}

	/*
	 * Returns EntityName based on serviceName, parentEntityName and associationName
	 * Reads details from csn.json
	 */
	public static String getEntityName(String serviceName, String parentEntityName, String associationName) {
		JsonNode csn = getCsn();
		if (csn == null)
			return null;
		String entityName = null;
		JsonNode entityNode = csn.get(DEFINITIONS).get(serviceName + "." + parentEntityName);
		if (entityNode != null) {
			JsonNode propertyNode = entityNode.get(ELEMENTS);
			JsonNode associationNode = propertyNode.get(associationName);
			if (associationNode != null) {
				entityName = associationNode.get(TARGET).asText();
				if (entityName != null && entityName.contains(".")) {
					entityName = entityName.substring(entityName.lastIndexOf('.') + 1);
				}
			}
		}
		return entityName;
	}

	/**
	 * Reads authorization (@requires/@restrict) details of the given service/entity
	 * name from CSN.json.
	 *
	 * @throws IllegalArgumentException - If conversion fails due to incompatible
	 *                                  type; check CSN.json and
	 *                                  AuthorizationDetails- class.
	 * @param serviceOrEntityName
	 * @return AuthorizationDetails if found. Else returns null.
	 */
	public static AuthorizationDetails getAuthorizationDetails(String serviceOrEntityName) {
		JsonNode csn = getCsn();
		if (csn != null) {
			JsonNode entityNode = csn.get(DEFINITIONS).get(serviceOrEntityName);
			if (entityNode != null) {
				return getAuthorizationDetails(entityNode);
			}
		}

		return null;
	}

	private static AuthorizationDetails getAuthorizationDetails(JsonNode entityNode) {
		Map<String, Object> authDetails = new HashMap<>();
		if (entityNode.get(KIND) != null) {
			if (entityNode.get(KIND).asText().equals("service")) {
				authDetails.put(KIND, entityNode.get(KIND));
				authDetails.put("requires", entityNode.get("@requires"));
				return getAuthorizationDetailsPojo(authDetails);
			} else if (entityNode.get(KIND).asText().equals("entity") || entityNode.get(KIND).asText().equals("view") || entityNode.get(KIND).asText().equals("action") || entityNode.get(KIND).asText().equals("function")) {
				authDetails.put(KIND, entityNode.get(KIND));
				JsonNode restrictDetails = entityNode.get(RESTRICT);
				// handle to convert str to array to make it consistent.
				convertToArray(restrictDetails);
				authDetails.put("restrict", entityNode.get(RESTRICT));
				return getAuthorizationDetailsPojo(authDetails);
			}
		}
		return null;
	}

	private static void convertToArray(JsonNode restrictDetails) {
		if (null != restrictDetails) {
			Iterator<JsonNode> allRoles = restrictDetails.iterator();
			while (allRoles.hasNext()) {
				JsonNode currentRule = allRoles.next();
				if (currentRule.get(GRANT)!= null && !currentRule.get(GRANT).isArray()) {
					String grantStr = currentRule.get(GRANT).asText();
					((ObjectNode) currentRule).remove(GRANT);
					((ObjectNode) currentRule).putArray(GRANT).add(grantStr);
				}
			}
		}
	}

	private static AuthorizationDetails getAuthorizationDetailsPojo(Map<String, Object> authDetails) {
		ObjectMapper mapper = new ObjectMapper().configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
		if (authDetails.get(REQUIRES) instanceof TextNode) {
			authDetails.put(REQUIRES, Arrays.asList(authDetails.get(REQUIRES)));
		}
		AuthorizationDetails pojo = mapper.convertValue(authDetails, AuthorizationDetails.class);
		return pojo;
	}

	/**
	 * Returns the value @cds.persistence.name annotation if present in the csn for
	 * the passed element. If not present this method returns null.
	 *
	 * @param fqEntityName
	 * @param elementName
	 * @return
	 */
	public static String getPersistenceName(String fqEntityName, String elementName) {
		JsonNode csn = getCsn();
		if (csn == null)
			return null;
		JsonNode elementNode = csn.get(DEFINITIONS).get(fqEntityName);
		if (elementNode == null)
			return null;
		if (elementName != null)
			elementNode = elementNode.get(ELEMENTS).get(elementName);
		if (elementNode == null)
			return null;
		return elementNode.get(CDS_PERSISTENCE_NAME) == null ? null : elementNode.get(CDS_PERSISTENCE_NAME).asText();
	}

	public static boolean isDraftEnabledEntity(String entityName, String serviceName) {
		JsonNode csn = getCsn();
		if (csn == null)
			return false;
		if (csn.get(DEFINITIONS).get(serviceName + "." + entityName) == null) {
			return false;
		}
		JsonNode draftEnabled = csn.get(DEFINITIONS).get(serviceName + "." + entityName).get(DRAFT_ENABLED);
		return draftEnabled != null && "true".equals(draftEnabled.asText());
	}

	public static boolean isDefaultOrderEnabled(String entityName, String serviceName) {
		JsonNode csn = getCsn();
		if (csn == null)
			return false;
		JsonNode orderEnabled = csn.get(DEFINITIONS).get(serviceName + "." + entityName);
		if (orderEnabled != null) {
			return orderEnabled.get(ORDER_DEFAULT) != null && orderEnabled.get(ORDER_DEFAULT).asText().equals("true");
		} else {
			return false;
		}

	}

	public static String getCDSDataType(String entityName, String serviceName, String propName) {

		JsonNode entityNode = getCsn().get(DEFINITIONS).get(serviceName + "." + entityName);
		if (entityNode != null) {
			return entityNode.get(ELEMENTS).get(propName).get("type").asText();
		}
		return null;

	}

	/**
	 * Returns the csn for the current call. This method logs if the csn is null.
	 * Actual handling for the null csn scenario is to be done in the calling
	 * method.
	 */
	public static JsonNode getCsn() {
		HashedData csnForThread = ThreadSafeObjectStore.getCsnForCurrentThread();
		if (csnForThread != null) {
			return parsedCsnModels.getUnchecked(csnForThread);
		}
		try {
			checkIfCSNInitialized();
		} catch (CSNNotSetException e) {
			logger.error(CSN_NOT_FOUND, e);
		}
		return staticCsn;
	}
	
	public static HashedData getHashedCsn() {
		HashedData csnForThread = ThreadSafeObjectStore.getCsnForCurrentThread();
		if (csnForThread != null) {
			return csnForThread;
		}
		try {
			checkIfCSNInitialized();
		} catch (CSNNotSetException e) {
			logger.error(CSN_NOT_FOUND, e);
			throw new RuntimeException(CSN_NOT_FOUND);
		}
		return staticHashedCSN;
	}

	public static boolean isDraftNodeEntity(String entityName, String serviceName) {
		JsonNode csn = getCsn();
		if (csn == null) {
			return false;
		}
		if (csn.get(DEFINITIONS).get(serviceName + "." + entityName) == null) {
			return false;
		}
		JsonNode draftEnabled = csn.get(DEFINITIONS).get(serviceName + "." + entityName).get(DRAFT_NODE);
		return draftEnabled != null && "draftPrepare".equals(draftEnabled.asText());
	}

	public static Map<String, List<String>> getCompositionTree(String serviceName, String entityName) {
		JsonNode csn = getCsn();
		if (csn == null)
			return new HashMap<String, List<String>>();
		JsonNode entityNode = csn.get(DEFINITIONS).get(serviceName + "." + entityName);
		Map.Entry<String, JsonNode> rootEntry = new Entry<String, JsonNode>() {
			@Override
			public JsonNode setValue(JsonNode value) {
				return null;
			}

			@Override
			public JsonNode getValue() {
				return entityNode;
			}

			@Override
			public String getKey() {
				return serviceName + "." + entityName;
			}
		};
		return constructCompositionTree(rootEntry, serviceName);
	}

	public static Map<String, List<String>> getDraftTree(String serviceName) {

		Map<String, Map<String, List<String>>> draftTree = new HashMap<>();
		JsonNode csn = getCsn();
		if (csn == null)
			return new HashMap<>();

		// current assumption .. only one Draft Tree per Service
		Map.Entry<String, JsonNode> root = null;
		JsonNode alldefs = csn.get(DEFINITIONS);

		Iterator<Entry<String, JsonNode>> nodes = alldefs.fields();

		while (nodes.hasNext()) {
			Map.Entry<String, JsonNode> entry = (Map.Entry<String, JsonNode>) nodes.next();
			if (entry.getKey().startsWith(serviceName) && entry.getValue().get(DRAFT_ENABLED) != null
					&& entry.getValue().get(DRAFT_ENABLED).asText().equals("true")) {
				root = entry;
				break;
			}
		}

		if (root != null) {
			draftTree.put(serviceName, constructCompositionTree(root, serviceName));
		}
		return draftTree.get(serviceName);
	}

	private static Map<String, List<String>> constructCompositionTree(Map.Entry<String, JsonNode> currentNode,
																	  String serviceName) {
		Map<String, List<String>> draftTree = new HashMap<>();
		List<String> compositions = new ArrayList<>();
		JsonNode csn = getCsn();
		if (csn == null)
			return draftTree;
		for (JsonNode property : currentNode.getValue().get(ELEMENTS)) {
			if (property.get("type") != null && property.get("type").asText().equalsIgnoreCase(CDS_COMPOSITION)) {
				compositions.add(property.get(TARGET).asText().substring(serviceName.length() + 1));
				Map.Entry<String, JsonNode> compositionNode = new Entry<String, JsonNode>() {
					@Override
					public JsonNode setValue(JsonNode value) {
						return null;
					}

					@Override
					public JsonNode getValue() {
						return csn.get(DEFINITIONS).get(property.get(TARGET).asText());
					}

					@Override
					public String getKey() {
						return property.get(TARGET).asText();
					}
				};

				draftTree.putAll(constructCompositionTree(compositionNode, serviceName));
			}
		}

		draftTree.put(currentNode.getKey().substring(serviceName.length() + 1), compositions);

		return draftTree;
	}

	// Parse the @cds.query.limit annotation for the given serviceName
	public static Long getQueryLimit(String serviceName) {
		JsonNode csn = getCsn();
		JsonNode queryLimit = null;
		if (csn == null)
			return null;
		if(csn.get(DEFINITIONS).get(serviceName) !=null)
			queryLimit = csn.get(DEFINITIONS).get(serviceName).get(CDS_QUERY_LIMIT);
		if (queryLimit != null) {
			return queryLimit.asLong();
		} else {
			return null;
		}
	}

	// Parse the @cds.consistent.paging annotation for the given serviceName and
	// entityName
	public static boolean isConsistentPagingEnabled(String serviceName, String entityName) {

		JsonNode csn = getCsn();
		if (csn == null)
			return false;
		JsonNode orderEnabled = csn.get(DEFINITIONS).get(serviceName + "." + entityName);
		if (orderEnabled != null) {
			return orderEnabled.get(CDS_CONSISTENT_PAGING) != null
					&& orderEnabled.get(CDS_CONSISTENT_PAGING).asText().equals("true");
		} else {
			return false;
		}

	}

	// Parse the @cds.temporal annotation for Each Element
	public static List<String> getTemporalEntities(String serviceName) {
		JsonNode csn = getCsn();
		List<String> temoralList = new ArrayList<>();
		if (csn == null)
			return temoralList;

		Iterator<Entry<String, JsonNode>> nodes = csn.get(DEFINITIONS).fields();

		while (nodes.hasNext()) {
			Map.Entry<String, JsonNode> entry = (Map.Entry<String, JsonNode>) nodes.next();
			Iterator<Entry<String, JsonNode>> FiledsinEachField = entry.getValue().fields();
			while (FiledsinEachField.hasNext()) {
				Map.Entry<String, JsonNode> fieldEntry = (Map.Entry<String, JsonNode>) FiledsinEachField.next();
				if (fieldEntry.getKey().startsWith(ELEMENTS)) {
					Iterator<Entry<String, JsonNode>> elementsIterator = fieldEntry.getValue().fields();
					while (elementsIterator.hasNext()) {
						Entry<String, JsonNode> element = elementsIterator.next();
						if (element.getValue().toString().contains(CDS_VALID)
								&& entry.getKey().startsWith(serviceName)) {
							temoralList.add(entry.getKey());
							break;
						}

					}
				}
				if (fieldEntry.getKey().startsWith(CDS_TEMPORAL)) {
					if (entry.getKey().startsWith(serviceName))
						temoralList.add(entry.getKey());
					break;
				}
			}
		}
		return temoralList;
	}

	public static Boolean isLocalized(String serviceName, String entityName) {
		JsonNode csn = getCsn();
		if (csn == null)
			return false;
		if (isDraftEnabledEntity(entityName, serviceName))
			return false;
		if (isDraftNodeEntity(entityName, serviceName))
			return false;
		String fqEntityName = serviceName + "." + entityName;
		JsonNode elementNode = csn.get(DEFINITIONS).get(fqEntityName);
		if (elementNode == null)
			return false;
		String localizedEntityName = LOCALIZED + "." + serviceName + "." + entityName;
		if (csn.get(DEFINITIONS).get(localizedEntityName) != null)
			return true;
		return false;
	}

	public static Boolean isLocalizedForDraft(String serviceName, String entityName) {
		JsonNode csn = getCsn();
		if (csn == null)
			return false;
		/*
		 * if (isDraftEnabledEntity(entityName, serviceName)) return false; if
		 * (isDraftNodeEntity(entityName, serviceName)) return false;
		 */
		String fqEntityName = serviceName + "." + entityName;
		JsonNode elementNode = csn.get(DEFINITIONS).get(fqEntityName);
		if (elementNode == null)
			return false;
		String localizedEntityName = LOCALIZED + "." + serviceName + "." + entityName;
		if (csn.get(DEFINITIONS).get(localizedEntityName) == null || elementNode.get(ELEMENTS) == null || elementNode.get(ELEMENTS).size() == 0)
			return false;
		Iterator<JsonNode> elements = elementNode.get(ELEMENTS).elements();
		while (elements.hasNext()) {
			JsonNode property = elements.next();
			if (property.get(LOCALIZED) != null && property.get(LOCALIZED).asBoolean() == true) {
				if(elementNode.get(ELEMENTS).get(LOCALIZED)!=null && elementNode.get(ELEMENTS).get(LOCALIZED).get("target")!=null)
					return true;
			}
		}
		return false;
	}

	public static boolean isAnyTemporal(String serviceName, String entitySetName) {
		boolean temporal = false;

		List<String> temporalEntityList = getTemporalEntities(serviceName);
		Map<String, List<String>> compositionTree = getCompositionTree(serviceName, entitySetName);

		if (!compositionTree.isEmpty()) {
			Iterator<Entry<String, List<String>>> it = compositionTree.entrySet().iterator();

			while (it.hasNext()) {
				Entry<String, List<String>> id = it.next();
				if (id.getKey() != null) {
					for (int i = 0; i < temporalEntityList.size(); i++) {
						if (id.getKey().equalsIgnoreCase(temporalEntityList.get(i).split("\\.")[1])) {
							temporal = true;
							break;
						}
					}
				}
				if (!id.getValue().isEmpty()) {
					for (int i = 0; i < temporalEntityList.size(); i++) {
						for (int j = 0; j < id.getValue().size(); j++) {
							if (id.getValue().get(j).equalsIgnoreCase(temporalEntityList.get(i).split("\\.")[1])) {
								temporal = true;
								break;
							}
						}
					}
				}
			}
		}
		return temporal;
	}

	public static boolean isTemporal(String serviceName, String requestURL) {
		boolean temporal = false;
		List<String> temporalEntityList = getTemporalEntities(serviceName);
		for (int i = 0; i < temporalEntityList.size(); i++) {
			if (requestURL.toLowerCase().contains(temporalEntityList.get(i).split("\\.")[1].toLowerCase())) {
				temporal = true;
			}
		}
		return temporal;
	}

	public static boolean isEntityTemporal(String serviceName, String entitySetName) {
		List<String> temporalEntityList = getTemporalEntities(serviceName);
		Boolean temporal = false;
		for (int i = 0; i < temporalEntityList.size(); i++) {
			if (entitySetName.equals(temporalEntityList.get(i).split("\\.")[1])) {
				temporal = true;
				break;
			}
		}

		return temporal;
	}

	public static List<String> getRootEntitiesWithDeltaEnabled(String serviceName, String entityName){
		JsonNode csn = getCsn();
		if (csn == null)
			return null;

		Map<String, ArrayList<String>> entitiesAndCompositions = new HashMap<String, ArrayList<String>>();
		Map<String,ArrayList<String>> rootEntitiesWithDeltaenabled = new HashMap<String, ArrayList<String>>();
		
		ArrayList<String> listOfEntitiesToBeUpdated = new ArrayList<String>();
		ArrayList<String> listOfRootEntitiesToBeUpdated = new ArrayList<String>();

		if(serviceName == null || serviceName.isEmpty()) {
			return listOfRootEntitiesToBeUpdated;
		}

		rootEntitiesWithDeltaenabled = getRootEntitiesWithDeltaEnabled(csn, serviceName);
		entitiesAndCompositions = getEntitiesAndComposition(csn, serviceName);

		ArrayList<String> rootEntitesForService = rootEntitiesWithDeltaenabled.get(serviceName);
		updateListofEntitesToBeUpdated(listOfEntitiesToBeUpdated, serviceName+ "." + entityName, entitiesAndCompositions);

		for(int i=0;i<rootEntitesForService.size();i++) {
			for(int j = 0;j<listOfEntitiesToBeUpdated.size();j++) {
				if(rootEntitesForService.get(i).compareTo(listOfEntitiesToBeUpdated.get(j))==0) {
					listOfRootEntitiesToBeUpdated.add(listOfEntitiesToBeUpdated.get(j));
				}
			}
		}

		return listOfRootEntitiesToBeUpdated;

	}

	private static void updateListofEntitesToBeUpdated(ArrayList<String> listOfEntitiesToBeUpdated, String childEntity, Map<String, ArrayList<String>> entitiesAndCompositions) {
		for (Map.Entry<String, ArrayList<String>> entry : entitiesAndCompositions.entrySet()) {
			ArrayList<String> listOfChildren = entry.getValue();
			for(String child: listOfChildren) {
				if(childEntity.equalsIgnoreCase(child)) {
					listOfEntitiesToBeUpdated.add(entry.getKey());
					updateListofEntitesToBeUpdated(listOfEntitiesToBeUpdated, entry.getKey(), entitiesAndCompositions);
				}
			}
		}

		if(listOfEntitiesToBeUpdated.isEmpty() && entitiesAndCompositions.containsKey(childEntity)) {
			listOfEntitiesToBeUpdated.add(childEntity);
		}

	}

	private static Map<String,ArrayList<String>> getRootEntitiesWithDeltaEnabled(JsonNode csn, String serviceName) {

		Map<String,ArrayList<String>> rootEntitiesWithDeltaenabled = new HashMap<String, ArrayList<String>>();
		JsonNode definitions = csn.get(DEFINITIONS);
		ArrayList<String> rootEntities = new ArrayList<String>();
		rootEntitiesWithDeltaenabled.put(serviceName, new ArrayList<>());
		Iterator<String> fieldNames = definitions.fieldNames();
		while (fieldNames.hasNext()) {
			String fielName = fieldNames.next();
			if(fielName.startsWith(serviceName))
				rootEntities.add(fielName) ;
		}
		for(String rootString : rootEntities) {
			if(definitions.get(rootString).get(DELTA_ENABLED) != null && definitions.get(rootString).get(DELTA_ENABLED).asBoolean()==true) {
				rootEntitiesWithDeltaenabled.get(serviceName).add(rootString);
			}

		}
		
		return rootEntitiesWithDeltaenabled;
	}
	
	private static Map<String, ArrayList<String>> getEntitiesAndComposition(JsonNode csn, String serviceName) {

		Map<String, ArrayList<String>> entitiesAndCompositions = new HashMap<String, ArrayList<String>>();
		JsonNode definitions = csn.get(DEFINITIONS);
		ArrayList<String> rootEntities = new ArrayList<String>();
		Iterator<String> fieldNames = definitions.fieldNames();
		while (fieldNames.hasNext()) {
			String fielName = fieldNames.next();
			if(fielName.startsWith(serviceName))
				rootEntities.add(fielName) ;
		}
		
		for(String entity : rootEntities) {
			ArrayList<String> compositions = new ArrayList<String>();
			if(definitions.get(entity).get(ELEMENTS) != null) {
				Iterator<Entry<String, JsonNode>> iterator =  definitions.get(entity).get(ELEMENTS).fields();
				while(iterator.hasNext()) {
					Entry<String, JsonNode> entry = iterator.next();
					if(entry.getValue().get(TYPE)!=null && entry.getValue().get(TYPE).asText().equals(CDS_COMPOSITION)) {
						compositions.add(entry.getValue().get("target").asText());
					}

				}
				if(compositions.size() > 0)
					entitiesAndCompositions.put(entity, compositions);
			}

		}
		
		return entitiesAndCompositions;
	}

	public static String getcompositionPath(String serviceName, String root, String element) {

		JsonNode csn = getCsn();
		if (csn == null)
			return null;
		
		if(root.equals(element))
			return "";

		String path = getPathtoChild(serviceName + "." + root, serviceName + "." + element, serviceName, csn);
		if(path==null || path.length()==0)
			return null;
		return path;
	}

	private static String getAssociationPropertyNameForTarget(String rootEntity, String entityToBeFound) {
		JsonNode csn = getCsn();
		if (csn == null)
			return null;
		JsonNode elementNode = csn.get(DEFINITIONS).get(rootEntity);
		if (elementNode == null || elementNode.get(ELEMENTS)==null || elementNode.get(ELEMENTS).size()==0)
			return null;
		Iterator<Entry<String, JsonNode>> elements = elementNode.get(ELEMENTS).fields();
		while (elements.hasNext()) {
			Entry<String, JsonNode> property = elements.next();
			if(property.getValue().get("target")!=null && property.getValue().get("target").asText().compareTo(entityToBeFound)==0)
				return property.getKey();
		}
		return null;
	}

	public static List<String> getKeys(String serviceName, String element){
		List<String> keys = new ArrayList<String>();
		JsonNode csn = getCsn();
		if (csn == null)
			return null;
		JsonNode elementNode = csn.get(DEFINITIONS).get(serviceName + "." + element);
		if (elementNode == null || elementNode.get(ELEMENTS)==null || elementNode.get(ELEMENTS).size()==0)
			return null;
		Iterator<Entry<String, JsonNode>> elements = elementNode.get(ELEMENTS).fields();
		while (elements.hasNext()) {
			Entry<String, JsonNode> property = elements.next();
			if(property.getValue().get("key")!=null && property.getValue().get("key").asBoolean()==true)
				keys.add(property.getKey());
		}
		return keys;
	}

	/*
	 * Returns all orderby properties name based on serviceName and entityName from csn.json
	 */
	public static List<Map<String, String>> getOrderByProperties(String serviceName, String entityName) {
		List<Map<String,String>> orderByOptions = new ArrayList();
		JsonNode csn = getCsn();
		if(csn!=null&&csn.get(CSNConstants.DEFINITIONS)!=null&&csn.get(CSNConstants.DEFINITIONS).get(serviceName + "." + entityName)!=null) {
			JsonNode queryNode = csn.get(CSNConstants.DEFINITIONS).get(serviceName + "." + entityName).get(CSNConstants.QUERY);
			if (queryNode != null) {
				JsonNode selectNode = queryNode.get(CSNConstants.SELECT);
				if(selectNode!=null) {
					JsonNode propertyNode = selectNode.get(CSNConstants.ORDERBY);
					if(propertyNode!=null) {
						for(JsonNode property:propertyNode)  {
							JsonNode elementNameNode = property.get(CSNConstants.REF);
							HashMap<String,String> selectProp = new HashMap<>(1);
							String propertyName = elementNameNode.get(0).asText();
							String sort = property.get(CSNConstants.SORT)!=null?property.get(CSNConstants.SORT).asText():CSNConstants.ASC;
							selectProp.put(propertyName, sort);
							orderByOptions.add(selectProp);
						}
					}
				}
			}
		}
		return orderByOptions;
	}

	//Parse the @odata.delta.enabled  annotation for the given serviceName and entityName
	public static boolean isDeltaEnabled(String serviceName, String entityName) {
		JsonNode csn = getCsn();
		if (csn == null)
			return false;
		List<String> roots = getRootEntitiesWithDeltaEnabled(serviceName, entityName);
		if(roots!=null && roots.size()>0)
			return true;
		JsonNode deltaEnabled = csn.get(DEFINITIONS).get(serviceName + "." + entityName);
		if (deltaEnabled != null) {
			return deltaEnabled.get(DELTA_ENABLED) != null && deltaEnabled.get(DELTA_ENABLED).asText().equals("true");
		} else {
			return false;
		}

	}

	private static String getPathtoChild(String startEntity, String childTobeFound, String serviceName, JsonNode csn) {
		
		Map<String, ArrayList<String>> entitiesAndCompositions = new HashMap<String, ArrayList<String>>();
		entitiesAndCompositions = getEntitiesAndComposition(csn, serviceName);
		
		List<String> childEntities = entitiesAndCompositions.get(startEntity);
		if(childEntities==null)
			return null;
		for(int i=0;i<childEntities.size();i++) {
			String assocName = getAssociationPropertyNameForTarget(startEntity, childEntities.get(i));
			if(assocName==null)
				continue;
			if(childTobeFound.compareToIgnoreCase(childEntities.get(i))==0) {
				return assocName;
			}else {
				String path = getPathtoChild(childEntities.get(i), childTobeFound, serviceName, csn);
				if(path == null) {
					if(i==childEntities.size()-1)
						return null;
					else
						continue;
				} else {
					path = assocName + DOT + path;
					return path;
				}
			}
		}
		return null;
	}

	public static List<String> getAggregationColumns(String serviceName, String entityName) {
		JsonNode csn = getCsn();
		if (csn == null) {
			return null;
		}
		List<String> columnList = new ArrayList<String>();
		JsonNode entityNode = csn.get(DEFINITIONS).get(serviceName + "." + entityName);
		if (entityNode != null) {
			JsonNode propertyNode = entityNode.get(ELEMENTS);
			Iterator<String> it = propertyNode.fieldNames();
			while (it.hasNext()) {
				String colName = it.next();
				if(propertyNode.get(colName).get("@Aggregation.default") != null) {
					columnList.add(colName);
				}
			}
		}
		return columnList;
	}

	public static List<String> getAllColumns(String serviceName, String entityName) {
		JsonNode csn = getCsn();
		if (csn == null) {
			return new ArrayList<String>();
		}
		List<String> columnList = new ArrayList<String>();
		JsonNode entityNode = csn.get(DEFINITIONS).get(serviceName + "." + entityName);
		if (entityNode != null) {
			JsonNode propertyNode = entityNode.get(ELEMENTS);
			Iterator<Entry<String, JsonNode>> iterator =  propertyNode.fields();
			while(iterator.hasNext()) {
				Entry<String, JsonNode> entry = iterator.next();
				if(!CDS_COMPOSITION.equalsIgnoreCase(entry.getValue().get(TYPE).asText()) &&
						!CDS_ASSOCIATION.equalsIgnoreCase(entry.getValue().get(TYPE).asText())) {
					columnList.add(entry.getKey());
				}
			}
		}
		return columnList;
	}

	public static String getTemporalDataType(String serviceName) {
		JsonNode csn = getCsn();
		if (csn == null)
			return null;

		Iterator<Entry<String, JsonNode>> nodes = csn.get(DEFINITIONS).fields();

		Iterator<Entry<String, JsonNode>> nodes1 = csn.get(DEFINITIONS).fields();

		String temporalKey = null;
		boolean breakcondition = false;
		String temporaldatatype = null;

		while (nodes.hasNext()) {
			Map.Entry<String, JsonNode> entry = (Map.Entry<String, JsonNode>) nodes.next();
			Iterator<Entry<String, JsonNode>> FiledsinEachField = entry.getValue().fields();
			while (FiledsinEachField.hasNext()) {
				Map.Entry<String, JsonNode> fieldEntry = (Map.Entry<String, JsonNode>) FiledsinEachField.next();

				if (fieldEntry.getKey().startsWith(CDS_TEMPORAL)) {
					if (fieldEntry.getValue() != null) {
						if (fieldEntry.getValue().toString().contains("{")
								&& fieldEntry.getValue().toString().contains("}")
								&& fieldEntry.getValue().toString().contains(":")) {
							String parsedValue = fieldEntry.getValue().toString().replace("{", "").replaceAll("}", "")
									.split(":")[1];
							temporalKey = parsedValue.substring(1, parsedValue.length() - 1);
							break;
						}
					}
				}
			}
			if (temporalKey != null)
				break;
		}

		while (nodes1.hasNext()) {
			Map.Entry<String, JsonNode> entry = (Map.Entry<String, JsonNode>) nodes1.next();
			Iterator<Entry<String, JsonNode>> FiledsinEachField = entry.getValue().fields();
			while (FiledsinEachField.hasNext()) {
				Map.Entry<String, JsonNode> fieldEntry = (Map.Entry<String, JsonNode>) FiledsinEachField.next();
				Iterator<Entry<String, JsonNode>> elementsIterator = null;
				if (fieldEntry.getKey().startsWith(ELEMENTS)) {
					elementsIterator = fieldEntry.getValue().fields();
					if (temporalKey != null) {
						while (elementsIterator.hasNext()) {
							Entry<String, JsonNode> element = elementsIterator.next();
							if (entry.getKey().startsWith(serviceName)
									&& element.getKey().toString().equals(temporalKey)) {
								Iterator<Entry<String, JsonNode>> propertyIterator = element.getValue().fields();
								while (propertyIterator.hasNext()) {
									Entry<String, JsonNode> property = propertyIterator.next();
									if (property.getKey().toString().startsWith(TYPE)) {
										temporaldatatype = property.getValue().asText();
										breakcondition = true;
										break;
									}

								}
							}
							if (breakcondition)
								break;
						}
					} else {
						while (elementsIterator.hasNext()) {
							Entry<String, JsonNode> element = elementsIterator.next();
							if (entry.getKey().startsWith(serviceName)
									&& element.getValue().toString().contains(CDS_VALID_FROM)) {
								Iterator<Entry<String, JsonNode>> propertyIterator = element.getValue().fields();
								while (propertyIterator.hasNext()) {
									Entry<String, JsonNode> property = propertyIterator.next();
									if (property.getKey().toString().startsWith(TYPE)) {
										temporaldatatype = property.getValue().asText();
										breakcondition = true;
										break;
									}

								}
							}
							if (breakcondition)
								break;
						}
					}

				}
			}
		}
		return temporaldatatype;
	}

	public static boolean isAutoExposedAndAutoExpose(String entityName, String serviceName) {
		JsonNode csn = getCsn();
		if (csn!= null){
			JsonNode node = csn.get(DEFINITIONS).get(serviceName + "." + entityName);
			if (node != null)
			{
				if (node.get(CDS_AUTOEXPOSED)!= null && node.get(CDS_AUTOEXPOSE)!= null ){
					return true;
				}
				return false;
			}
		}
		return false;
	}

	public static boolean isAutoExposed(String entityName, String serviceName) {
		JsonNode csn = getCsn();
		if (csn!= null){
			JsonNode node = csn.get(DEFINITIONS).get(serviceName + "." + entityName);
			if (node != null){
				if (node.get(CDS_AUTOEXPOSED)!= null && node.get(CDS_AUTOEXPOSE)== null ){
					return true;
				}
				return false;
			}
		}
		return false;
	}

	public static boolean isAggregatedEntity(String serviceName, String entityName) {
		JsonNode csn = getCsn();
		if (csn != null) {
			JsonNode entityNode = csn.get(DEFINITIONS).get(serviceName + "." + entityName);
			if (entityNode != null 
					&& (entityNode.get(AGGREGATED_ENTITY) != null && entityNode.get(AGGREGATED_ENTITY).asBoolean())) {
				return true;
			}
		}
		return false;
	}
	public static String getEtagProperty(String fullyQualifiedEntityName) {
		String result = null;
		JsonNode csn = getCsn();
		if (csn == null)
			return null;
		JsonNode entityNode = csn.get(DEFINITIONS).get(fullyQualifiedEntityName);
		if(entityNode == null) {
			return null;
		}
		Iterator<Entry<String, JsonNode>> itr = entityNode.get("elements").fields();
		while(itr!=null && itr.hasNext()){
			Entry<String, JsonNode> entry = itr.next();
			JsonNode property = entry.getValue();
			if(property.get(ODATA_ETAG) != null && property.get(ODATA_ETAG).asBoolean() == true ||
					property.get(CDS_ETAG) != null && property.get(CDS_ETAG).asBoolean() == true){
				result = entry.getKey();
				break;
			}
		}

		return result;
	}

	public static Map<String, List<String>> getReferences(String serviceName, String entityName, boolean readCompositionsOnly) {
		Map<String, List<String>> references = new HashMap<>();
		Set<String> fromProperty = new HashSet<>();
		JsonNode csn = getCsn();
		if (csn == null) {
			return null;
		}
		JsonNode entityNode = csn.get(DEFINITIONS).get(serviceName + "." + entityName);
		if (entityNode != null) {
			JsonNode propertyNode = entityNode.get(ELEMENTS);
			Iterator<Entry<String, JsonNode>> iterator =  propertyNode.fields();
			String key = null;
			while(iterator.hasNext()) {
				Entry<String, JsonNode> entry = iterator.next();
				if (key == null && entry.getValue().get("key") != null && entry.getValue().get("key").asBoolean()) {
					key = entry.getKey();
				}
				if (CDS_COMPOSITION.equalsIgnoreCase(entry.getValue().get(TYPE).asText()) ||
						(CDS_ASSOCIATION.equalsIgnoreCase(entry.getValue().get(TYPE).asText()) && !readCompositionsOnly)) {
					List<String> fromToKeys = new ArrayList<>(2);
					String from = null, to = null;
					JsonNode ref = entry.getValue().get("on");
					if (ref != null) {
						if (ref.get(0).get("ref").size() == 2) {
							to = ref.get(0).get("ref").get(1).asText();
						} else {
							from = ref.get(0).get("ref").get(0).asText();
						}
						if (ref.get(2).get("ref").size() == 2) {
							to = ref.get(2).get("ref").get(1).asText();
						} else {
							from = ref.get(2).get("ref").get(0).asText();
						}
					} else if ((ref = entry.getValue().get("onCond")) != null) {
						if (ref.get("args").get(0).get("=").asText().contains(DOT)) {
							to = ref.get("args").get(0).get("=").asText();
							to = to.substring(to.lastIndexOf(DOT)+1);
						} else {
							from = ref.get("args").get(0).get("=").asText();
						}
						if (ref.get("args").get(1).get("=").asText().contains(DOT)) {
							to = ref.get("args").get(1).get("=").asText();
							to = to.substring(to.lastIndexOf(DOT)+1);
						} else {
							from = ref.get("args").get(1).get("=").asText();
						}
					} else if ((ref = entry.getValue().get("keys")) != null) {
						to = ref.get(0).get("ref").get(0).asText();
						from = ref.get(0).get("$generatedFieldName").asText();
					}
					if ("$self".equals(from)) { //multiple references
						from = key;
						to = key;
					}
					fromProperty.add(from);
					fromToKeys.add(from);
					fromToKeys.add(to);
					String navigationEntityName = entry.getValue().get("target").asText();
					navigationEntityName = navigationEntityName.substring(navigationEntityName.lastIndexOf(DOT)+1);
					references.put(navigationEntityName, fromToKeys);
				}
			}
		}
		if (!fromProperty.isEmpty()) {
			references.put(entityName, new ArrayList(fromProperty));
		}
		return references;
	}

	public static Map<String, List<String>> getReferences(String serviceName, String entityName) {
		return getReferences(serviceName, entityName, false);
	}

	public static Integer getSnapshotStorageDays() {
		JsonNode csn = getCsn();
		if (csn == null)
			return null;

		JsonNode entityNode = csn.get(DEFINITIONS).get(SNAPSHOT_STORAGE_DAYS);
		if (entityNode != null) {
			return entityNode.asInt();
		}
		return null;
	}

	public static boolean isConsistentPagingOrDeltaEnabled() {

		JsonNode csn = getCsn();
		if (csn == null)
			return false;

		Iterator<Entry<String, JsonNode>> nodes = csn.get(DEFINITIONS).fields();
		while (nodes.hasNext()) {
			Map.Entry<String, JsonNode> entry = (Map.Entry<String, JsonNode>) nodes.next();
			JsonNode serviceNode = entry.getValue();
			if(serviceNode != null && serviceNode.get(CDS_CONSISTENT_PAGING) != null
					&& serviceNode.get(CDS_CONSISTENT_PAGING).asBoolean() == true) {
				return true;
			}
			if(serviceNode != null && serviceNode.get(DELTA_ENABLED) != null
					&& serviceNode.get(DELTA_ENABLED).asBoolean() == true) {
				return true;
			}
		}
		return false;
	}

	public static String getNavName(String serviceName, String entityName ,String targetNav) {
		String navName = null;
		JsonNode csn = getCsn();
		if (csn == null) {
			return null;
		}
		JsonNode entityNode = csn.get(DEFINITIONS).get(serviceName + "." + entityName);
		if (entityNode != null) {
			JsonNode propertyNode = entityNode.get(ELEMENTS);
			Iterator<Entry<String, JsonNode>> iterator =  propertyNode.fields();
			while(iterator.hasNext()) {
				Entry<String, JsonNode> entry = iterator.next();
				JsonNode targetNode = entry.getValue().get(TARGET);
				if (targetNode != null){
					if (targetNode.asText().equals(serviceName+"."+targetNav)){
						return entry.getKey() ;
					}
				}
			}
		}
		return navName;
	}

	public static boolean isAuditLogEnabled(String serviceName, String entityName, String operation) {
		JsonNode csn = getCsn();
		if(csn == null){
			return false;
		}
		boolean chkEntityEligible = false, chkOperationEligible = false;
		JsonNode entityNode;
		JsonNode node = csn.get("definitions").get(serviceName + "." + entityName);
		if (node.get("kind").asText().equals("view")) {
			String entity = node.get("source").asText();
			entityNode = csn.get("definitions").get(entity);
		} else
			entityNode = node;
		if (entityNode.get(entitySemantics) != null) {
			chkEntityEligible = true;
			if (entityNode.get(auditLogOperation + "." + operation) != null
					&& entityNode.get(auditLogOperation + "." + operation).asBoolean()) {
				chkOperationEligible = true;
			}
		} else if (entityNode.get(Semantics) != null) {
			chkEntityEligible = true;
			if (entityNode.get(dataChangeLogOperation + "." + operation) != null
					&& entityNode.get(dataChangeLogOperation + "." + operation).asBoolean()) {
				chkOperationEligible = true;
			}
		}
		return chkEntityEligible && chkOperationEligible;
	}

	public static Map<String, AdminDataAnnotation> getDataMapForCoreComputed(String serviceName, String entityName) {
		JsonNode csn = getCsn();
		if (csn == null)
			return new HashMap<>();
		Map<String, AdminDataAnnotation> computeDataMap = new HashMap<>();
		JsonNode node = csn.get(DEFINITIONS).get(serviceName + "." + entityName).get(ELEMENTS);
		Iterator<String> it = node.fieldNames();
		while (it.hasNext()) {
			String colName = it.next();
			JsonNode computeNode = node.get(colName).get(ODATA_COMPUTED);
			JsonNode virtualNode = node.get(colName).get(VIRTUAL);
			if (computeNode != null && computeNode.get(ODATA_HASH) != null  && computeNode.get(ODATA_HASH).asText().equals("false")) {
				computeDataMap.put(colName,
						getAdminDataAnnotation(computeNode.get(ODATA_HASH).asText(), false));
			}
			else if ((computeNode != null &&  computeNode.asBoolean()) && !(virtualNode != null &&  virtualNode.asBoolean())){
				computeDataMap.put(colName,AdminDataAnnotation.COMPUTED_FIELD);
			}

		}
		return computeDataMap;
	}

	public static Map<String, AdminDataAnnotation> getDataMapForCoreImmutable(String serviceName, String entityName) {
		JsonNode csn = getCsn();
		if (csn == null)
			return new HashMap<>();
		Map<String, AdminDataAnnotation> computeDataMap = new HashMap<>();
		JsonNode node = csn.get(DEFINITIONS).get(serviceName + "." + entityName).get(ELEMENTS);
		Iterator<String> it = node.fieldNames();
		while (it.hasNext()) {
			String colName = it.next();
			JsonNode computeNode = node.get(colName).get(ODATA_IMMUTABLE);
			if (computeNode != null && computeNode.get(ODATA_HASH) != null  && computeNode.get(ODATA_HASH).asText().equals("false")) {
				computeDataMap.put(colName,
						getAdminDataAnnotation(computeNode.get(ODATA_HASH).asText(), false));
			}
			else if (computeNode != null &&  computeNode.asBoolean()){
				computeDataMap.put(colName,AdminDataAnnotation.IMMUTABLE_FIELD);
			}

		}
		return computeDataMap;
	}
	
	public static Set<String>  getVirtualElement(String serviceName, String entityName) {
		JsonNode csn = getCsn();
		if (csn == null)
			return new HashSet<>();
		Set<String> VirtualElements = new HashSet<>();
		if ((csn.get(DEFINITIONS).get(serviceName + "." + entityName)) != null){
			JsonNode node = csn.get(DEFINITIONS).get(serviceName + "." + entityName).get(ELEMENTS);
			Iterator<String> it = node.fieldNames();
			while (it.hasNext()) {
				String colName = it.next();
				JsonNode virtualNode = node.get(colName).get(VIRTUAL);
				if (virtualNode != null && virtualNode.asBoolean() == true) {
					VirtualElements.add(colName);
				}

			}
		}
		return VirtualElements;
	}
	
	public static boolean checkVirtualElement(String propName,Set<String> virtualElements) {
		
		if (virtualElements == null){
			return false;
		}
		if (virtualElements.isEmpty()){
			return false;
		}
		Iterator<String> value = virtualElements.iterator();
				
	    while(value.hasNext()){	
	    	String ve = value.next();
	    	if(propName.equals(ve)){
				  return true;
		    	}
	    	else if(propName.contains(ve.toUpperCase())){
			  return true;
	    	}
		  }
		  
		return false;
	}
	
	public static String getAggregationRefElement(String serviceName, String element, String entity) {

		JsonNode csn = getCsn();
		String refrenceElement = null;
		JsonNode aggregation = csn.get(DEFINITIONS).get(serviceName + "." + entity).get(ELEMENTS).get(element)
				.get("@Aggregation.referenceElement");
		if (aggregation != null) {
			StringBuilder refElement = new StringBuilder();
			aggregation.forEach((refEl)->{
				if(refElement.length() > 0)
					refElement.append(", "+refEl.asText());
				else
					refElement.append(refEl.asText());
			});
			refrenceElement = refElement.toString();
		}

		return refrenceElement;
	}

	public static String getCDSParamDataType(String entityName, String serviceName, String paramName) {
		JsonNode entityNode = getCsn().get(DEFINITIONS).get(serviceName + "." + entityName);
		if (entityNode != null) {
			return entityNode.get(PARAMS).get(paramName).get("type").asText();
		}
		return null;
	}
	
	public static Map<String, String> getReferentialConstarint(String serviceName, String entityName, String navigation) {
		Map<String, String> refConstarints = null;
		try {
			JsonNode csn = getCsn();
			String childProperty = null;
			String parentProperty = null;
			refConstarints = new HashMap<String, String>();
			if (csn == null)
				return null;
			JsonNode insertable = csn.get(DEFINITIONS).get(serviceName + "." + entityName).get(ELEMENTS).get(navigation);
			ArrayNode node = (ArrayNode) insertable.get(ON);
			for(int i =0;i<node.size(); i++) {
				ArrayNode refNode = (ArrayNode) node.get(i).get(REF);
				if(refNode != null && refNode.size() == 2 ) {
					childProperty = node.get(i).get(REF).get(1).asText();
				}
				else if(refNode != null && refNode.size() == 1 ) {
					parentProperty = node.get(i).get(REF).get(0).asText();
				}
				if(parentProperty != null && childProperty != null) {
					refConstarints.put(parentProperty, childProperty);
					parentProperty = null;
					childProperty= null;
				}
			}
		}
		catch(Exception e) {
			logger.error(NO_REF_CONSTRAINT);
			return null;
		}return refConstarints;

	}

}