/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.service.prov.api.internal;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import com.sap.cloud.sdk.service.prov.api.ElementType;
import com.sap.cloud.sdk.service.prov.api.EntityData;
import com.sap.cloud.sdk.service.prov.api.EntityDataBuilder;
import com.sap.cloud.sdk.service.prov.api.EntityMetadata;

public class DefaultEntityDataBuilder implements EntityDataBuilder {

	HashMap<String, Object> properties = new HashMap<>();
	List<String> elements = new ArrayList<>();
	List<String> associationElements = new ArrayList<>();
	List<String> keys = new ArrayList<>();
	HashMap<String, ElementType> elementTypes = new HashMap<>();
	String entityName;
	String namespace;
	Map<String, Object> associationProperties = new HashMap<>();
	
	public DefaultEntityDataBuilder() {
		elements = new ArrayList<>();
	}
	
	public DefaultEntityDataBuilder(EntityData ed) {
		EntityMetadata em = ((HasMetadata)ed).getEntityMetadata();
		elements.addAll(em.getElementNames());
		Map<String, Object> localAssociationProperties = ed.getAssociations();
		if(localAssociationProperties != null && !localAssociationProperties.isEmpty()){
			associationProperties.putAll(localAssociationProperties);
		}
	 
		for(String element : elements) {
			if(ed.contains(element)) {
				properties.put(element, ed.getElementValue(element));
			}
		}
		keys.addAll(em.getKeyNames());
		entityName = em.getName();
		namespace = em.getNamespace();
		elementTypes = copyElementTypes(em);
		//Expanded child entity not propagated @AfterExit for OData V2
		associationElements = em.getAssociationNames();
		if (associationElements != null) {
			for(String associationElement : associationElements) {
				if(ed.contains(associationElement)) {
					properties.put(associationElement, ed.getElementValue(associationElement));
				}
			}
		}
	}
	
	private HashMap<String, ElementType> copyElementTypes(EntityMetadata em) {
		if(em instanceof DefaultEntityData)
			return ((DefaultEntityMetadata)em).getElementTypes();
		
		HashMap<String, ElementType> elementTypes = new HashMap<>();
		for(String element : em.getElementNames()) {
			elementTypes.put(element, em.getElementType(element));
		}
		
		return elementTypes;
	}

	@SuppressWarnings("unchecked")
	@Override
	public DefaultEntityDataBuilder addElement(String name, Object value) {
		if(!properties.containsKey(name) && !elements.contains(name))
			elements.add(name);
		properties.put(name, value);
		
		if(value instanceof HashMap) {
			elementTypes.put(name, ElementType.STRUCTURED_TYPE);
		}
		else {
			elementTypes.put(name, ElementType.SIMPLE_TYPE);
		}	
		return this;
	}
	
	private List<String> getFlattenedElementsFromHashMap(String name, HashMap<String, Object> value) {
		
		List<String> flattened = new ArrayList<>();
		if(value == null || value.isEmpty()){
			return flattened;
		}
		
		for(Entry<String, Object> entry : value.entrySet()) {
			String path = name + "." + entry.getKey();
			if(entry.getValue() instanceof HashMap) {
				flattened.addAll(getFlattenedElementsFromHashMap(path, (HashMap<String, Object>) entry.getValue()));
			}
			else
				flattened.add(path);
		}
		
		return flattened;
	}

	@Override
	public DefaultEntityDataBuilder addKeyElement(String name, Object value) {
		if(!elements.contains(name))
			elements.add(name);
		properties.put(name, value);
		if(!keys.contains(name))
			keys.add(name);
		
		if(value instanceof HashMap) {
			elementTypes.put(name, ElementType.STRUCTURED_TYPE);
		}
		else {
			elementTypes.put(name, ElementType.SIMPLE_TYPE);
		}	
		return this;
	}
	
	@Override
	public DefaultEntityDataBuilder removeElement(String name) {
		properties.remove(name);
		elements.remove(name);
		keys.remove(name);
		return this;
	}
	
	@Override
	public EntityData buildEntityData(String name) {
		List<String> flattenedElements = new ArrayList<>();
		for(String element : elementTypes.keySet()) {
			ElementType type = elementTypes.get(element);
			if(type.equals(ElementType.SIMPLE_TYPE)) 
				flattenedElements.add(element);
			else {
				HashMap<String, Object> value = (HashMap<String, Object>) properties.get(element);
				flattenedElements.addAll(getFlattenedElementsFromHashMap(element, value));
			}

		}
		DefaultEntityMetadata em = new DefaultEntityMetadata()
				.setEntityName(name)
				.setElements(elements)
				.setKeys(keys)
				.setNamespace(namespace)
				.setElementTypes(elementTypes)
				.setFlattenedElements(flattenedElements);
		
		if(associationProperties.size() > 0){
			return new DefaultEntityData(properties, associationProperties, em);
		}else{
			return new DefaultEntityData(properties, em);
		}
	}

	@Override
	public EntityDataBuilder addAssociationElement(String name, Object value) {
		associationProperties.put(name, value);
		return this;
	}
	
}
