/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.service.prov.api.request;

import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cloud.sdk.service.prov.api.EntityMetadata;
import com.sap.cloud.sdk.service.prov.api.MessageContainer;
import com.sap.cloud.sdk.service.prov.api.locale.impl.LocaleTypeResolver;

/**
 * Provides the request information for an operation.
 */
public abstract class GenericRequest {	
	final static Logger log = LoggerFactory.getLogger(Request.class);
	protected MessageContainer messageContainer;
	protected EntityMetadata entityMetadata;
	protected String httpMethod;
	protected RequestContext requestContext;
	protected final Map<String, List<String>> headers = new HashMap<String, List<String>>();
	private Locale locale;
	protected String serviceName;
	private static final String ACCEPT_LANGUAGE = "Accept-Language";
	
	protected Map<String, List<String>> initialiseHeaders(Map<String,List<String>> headers) {
		for(Map.Entry<String, List<String>> entry: headers.entrySet()) {
			this.headers.put(entry.getKey(),entry.getValue());
		}
		String header = this.getHeader(ACCEPT_LANGUAGE);
		messageContainer = new MessageContainerImpl(LocaleTypeResolver.getLocale(header));
		header = LocaleTypeResolver.getLanguage(LocaleTypeResolver.getLocale(header).toString());
		this.locale = LocaleTypeResolver.getLocale(header);
		
		return this.headers;
	}
	
	/**
	 * Returns the language specified in the Accept-Language header of the request. 	 
	 * @return A {@link java.util.Locale Locale} object representing the preferred language
	 */
	public Locale getLocale(){
		return this.locale;
	}
	
	/**
	 * Returns a collection of all the header names that the request contains.
	 * @return An {@link java.util.Enumeration<java.lang.String> Enumeration<String>} object containing header names
	 */
	public Enumeration<String> getHeaderNames() {
		Set<String> headerNames = headers.keySet();
		return (headerNames.size() > 0 ? Collections.enumeration(headerNames) : null);
	}
	
	/**
	 * Returns all the values of the specified request header as a collection of String objects.
	 * Please note that header name is case-insensitive.
	 * @param name A <code>String</code> containing the header name
	 * @return An {@link java.util.Enumeration<java.lang.String> Enumeration<String>} object containing header values
	 */
	public Enumeration<String> getHeaders(String name) {
		String headerName = getHeaderName(name);
		if (log.isDebugEnabled()) {log.debug("Get header value for name "+name+". Returning value for name "+headerName+".");}
		List<String> valueList = headers.get(headerName);
		return (valueList == null ? null : Collections.enumeration(valueList));
	}
	
	/**
	 * Returns the first value of the specified request header as String object.
	 * Please note that header name is case-insensitive.
	 * @param name A <code>String</code> containing the header name
	 * @return A <code>String</code> containing header value 
	 */
	public String getHeader(String name) {
		String headerName = getHeaderName(name);
		if (log.isDebugEnabled()) {log.debug("Get first header value for name "+name+". Returning value for name "+headerName+".");}
		List<String> valueList = headers.get(headerName);
		return (valueList == null ? null : (valueList.size() > 0 ? valueList.get(0) : null));
	}
	
	/**
	 * Returns true if the header is present in the request, otherwise, it returns false.
	 * Please note that header name is case-insensitive.
	 * @return A <code>boolean</code> indicating the presence of the header 
	 */
	public boolean containsHeader(String name) {
		for (String headerName : headers.keySet()) {
			if (headerName.equalsIgnoreCase(name)) {
				if (log.isDebugEnabled()) {log.debug("Found header name "+name+" as name "+headerName+" in header names.");}
				return true;
		    }
		}
		return false;
	}
	
	/**
	 * Returns the name of the HTTP method with which this request was triggered.
	 * For example, GET, POST, or PUT.
	 * @return A <code>String</code> returning the name of the HTTP method 
	 */
	public String getHttpMethod() {
		return httpMethod;
	}
	
	protected String getHeaderName(String name) {
		//Header names are case insensitive
	    for (String headerName : headers.keySet()) {
	      if (headerName.equalsIgnoreCase(name)) {
	        return headerName;
	      }
	    }
	    return name;
	}
	
	/**
	 * Returns the additional request information in the context of the request protocol.
	 * For example, in the following code snippet this method returns an ODataRequestContext,
	 * using which an ODataRequest object is returned.
	 * <code>((ODataRequestContext)this.getRequestContext).getODataRequest()</code>
	 * @return A <code>RequestContext</code> object containing the additional request information
	 */
	public RequestContext getRequestContext() {
		return requestContext;
	}
	
 
	
	
	/**
	 * Returns the metadata of the requested entity.
	 * @return An <code>EntityMetadata</code> object containing metadata information
	 */
	public EntityMetadata getEntityMetadata() {
		return entityMetadata;
	}
	
	/**
	 * Gets the message container.
	 * Through the message container one can add info, warning, error messages, which can either come as the value of sap-message header,
	 * or can come as error-details in case of an error based on the calls the developer make.
	 * @return MessageContainer object for this request.
	 */
	public MessageContainer getMessageContainer() {
		if(messageContainer == null)
			messageContainer = new MessageContainerImpl(locale);
		return messageContainer;
	}
	
	/**
	 * Returns the  of the service 	 
	 * @return A <code>String</code> containing the name of the service
	 */
	public String getServiceName() {
		return serviceName;
	}

	public void setServiceName(String serviceName) {
		this.serviceName = serviceName;
	}
}
