/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.service.prov.api.request;

import java.util.List;
import java.util.Map;

import com.sap.cloud.sdk.service.prov.api.filter.Expression;
import com.sap.cloud.sdk.service.prov.api.filter.ExpressionBuilder;

/**
 * Provides the request information for querying an entity in a service.
 */
public abstract class QueryRequest extends Request {
	/**
	 * Gets the keys of the parent entity specified in a navigation. For example, if
	 * the URL is /EPMSampleService/SalesOrders(1)/ToSalesOrderLineItems, then this
	 * method returns (SalesOrderID,1).
	 * 
	 * @return A {@link java.util.Map Map} containing the keys of the parent entity
	 */
	public abstract Map<String, Object> getSourceKeys();

	/**
	 * Gets the value specified in the $top system query option. For example, if the
	 * URL is /EPMSampleService/Products?$top=1, then the value returned by this
	 * method is 1. if top option value is not specified in the URL then the value
	 * returned by this method is -1
	 * 
	 * @return An <code>int</code> representing the value in the $top system query
	 *         option
	 */
	public abstract int getTopOptionValue();

	/**
	 * Gets the custom query options specified in the URL. For example, if the URL
	 * is /EPMSampleService/Products?Search="abc", then the method returns a Map
	 * containing the value (Search,"abc"). If custom query option is not specified
	 * in the URL, then the value returned by this method is null.
	 * 
	 * @return A <code>Map<String, String></code> containing the key value pair of
	 *         the specified custom query options
	 */
	public abstract Map<String, String> getCustomQueryOptions();

	/**
	 * Gets the value specified in the $skip system query option. For example, if
	 * the URL is /EPMSampleService/Products?$skip=1, then the value returned by
	 * this method is 1. if skip option value is not specified in the URL then the
	 * value returned by this method is -1
	 * 
	 * @return An <code>int</code> representing the value in the $skip system query
	 *         option
	 */
	public abstract int getSkipOptionValue();

	/**
	 * Gets the list of properties specified in the $orderby system query option.
	 * For example, if the URL is /EPMSampleService/Products?$orderby=Category desc,
	 * then the property of the expression returned is Category and the sorting
	 * order is descending.
	 * 
	 * @return A <code>List</code> of <code>OrderByExpression</code> objects
	 *         containing the properties and their corresponding sorting order
	 */
	public abstract List<OrderByExpression> getOrderByProperties();

	/**
	 * Gets the list of properties specified in the $select system query option. For
	 * example, if the URL is /EPMSampleService/Products?$select=Category, then the
	 * property of the entity set returned by this method is Category. Please note
	 * that if the system query option used in the URL is $select=*, then this
	 * method returns an empty list.
	 * 
	 * @return A <code>List</code> of <code>String</code> objects containing the
	 *         properties specified in $select expression
	 */
	public abstract List<String> getSelectProperties();

	/**
	 * Returns true when it is a $count call(Only return the number of records.) For
	 * example, if the URL is /EPMSampleService/SalesOrders/$count then this method
	 * returns true
	 * 
	 * @return A <code>boolean</code> representing if it is a $count call
	 */
	public abstract boolean isCountCall();

	/**
	 * Indicates whether there is a $inlinecount=allpages or $count=true system
	 * query option in the request (depending on OData V2 or OData V4). For example,
	 * if the URL is /manageProducts/GuidTestHC?$inlinecount=allpages then this
	 * method returns true or if the URL is
	 * /manageProducts/GuidTestHC?$inlinecount=none then this method returns false.
	 * 
	 * @return A <code>boolean</code> indicating whether there is a
	 *         $inlinecount=allpages or $count=true system query option in the
	 *         request
	 */
	public abstract boolean isInlineCountCall();

	/**
	 * Gets the filter expression tree from the Query request.
	 * 
	 * @return An <code>Expression</code> object representing the filter expression
	 *         tree
	 */
	public abstract Expression getQueryExpression();

	/**
	 * Gets the <code>ExpressionBuilder</code> object with which you can create your
	 * own filter expressions.
	 * 
	 * @return An <code>ExpressionBuilder</code> object
	 */
	public abstract ExpressionBuilder getExpressionBuilder();

	/**
	 * Adds properties to the $select system query option.
	 * 
	 * @param selectProp A list of strings representing the properties to be added
	 */
	public abstract void addSelectProperties(List<String> selectProp);
	/**
	 * Gets the list of properties specified in the $expand system query option.
	 * For example, if the URL is /EPMSampleService/Product(1)?$expand=Category,
	 * then the property of the entity set returned by this method is Category.	 
	 * @return A <code>List</code> of <code>String</code> objects containing the properties
	 * specified in $expand expression
	 */
	public abstract List<String> getExpandProperties();

}
