/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.service.prov.api.request.impl;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.sap.cloud.sdk.service.prov.api.EntityData;
import com.sap.cloud.sdk.service.prov.api.EntityMetadata;
import com.sap.cloud.sdk.service.prov.api.exception.DataConversionException;
import com.sap.cloud.sdk.service.prov.api.request.RequestContext;
import com.sap.cloud.sdk.service.prov.api.request.UpdateRequest;
import com.sap.cloud.sdk.service.prov.api.util.DataConversionUtility;
import com.sap.cloud.sdk.service.prov.api.util.PojoUtil;

public class UpdateRequestImpl extends UpdateRequest {

	/*
	 * Just one object for storing data regardless of the format. Based on
	 * getter method invoked, this would be converted to appropriate format
	 * using the DataConversionUtility.
	 */
	private Object data;
	private Map<String, Object> keys;
	private Set<String> initialListOfProperties;

	public UpdateRequestImpl(RequestContext requestContext, String entityName, EntityMetadata entityMetadata,
			Map<String, List<String>> headers, Map<String, Object> keys, String httpMethod) {
		this.requestContext = requestContext;
		this.entityMetadata = entityMetadata;
		this.entityName = entityName;
		this.keys = keys;
		this.httpMethod = httpMethod;
		initialiseHeaders(headers);
	}

	@Override
	public Map<String, Object> getKeys() {
		return this.keys;
	}

	@Override
	public EntityData getData() {
		return DataConversionUtility.convertToEntityData(data, entityName, entityMetadata.getKeyNames());
	}

	public void setData(Object incomingData) {
		/*
		 * The below condition would happen when the UpdateRequestImpl is first
		 * created, likE when the first phased exit is being invoked in the
		 * autoexposure scenario. We initialize the initialListOfProperties
		 * based on the incoming data.
		 */
		if (initialListOfProperties == null) {
			if (incomingData instanceof Map)
				initialListOfProperties = ((Map) incomingData).keySet();
			else if (incomingData instanceof EntityData)
				initialListOfProperties = ((EntityData) incomingData).asMap().keySet();
		}

		/*
		 * The below condition is to check whether the data is of a format other
		 * than entityData, and if so convert it to entityData. We do so because
		 * we want an immutable object as the source object(unlike a pojo or a
		 * map). However we do not create EntityData directly, we first create a map,
		 * reasons are explained below.
		 */
		if (incomingData != null && !(incomingData instanceof EntityData)) {
			Map<String, Object> tempData = PojoUtil.getMapUsingBeanUtils(incomingData);

			// The below condition checks whether the data is a POJO.
			if (!(incomingData instanceof Map)) {
				/*
				 * At this point it is clear that incoming data is a POJO.
				 * Here we perform some 'removal logic'. Since POJO by
				 * default would have all the properties, we would like to
				 * remove the extra in the final entity data. This is done by
				 * checking how many properties are newly added, (by comparing
				 * with initialListOfProperties.)
				 */
				List<String> extraPropertiesInIncomingData = tempData.keySet().stream()
						.filter(prop -> !initialListOfProperties.contains(prop)).collect(Collectors.toList());
				/*
				 * Now for each of the newly added properties we check if the
				 * value is null. If this is the case, we assume that these
				 * properties are coming only because the developer has chosen
				 * POJO as his data format and not because he has actually added
				 * those properties.
				 */
				extraPropertiesInIncomingData.stream().filter(prop -> (tempData.get(prop) == null))
						.forEach(p -> tempData.remove(p));
			}

			data = DataConversionUtility.convertToEntityData(tempData, entityName, entityMetadata.getKeyNames());
		} else
			data = incomingData;
	}

	public Map<String, Object> getMapData() {
		return DataConversionUtility.convertToMap(data);
	}

	@Override
	public <T> T getDataAs(Class<T> clazz) throws DataConversionException {
		return DataConversionUtility.convertToClass(clazz, data);
	}
}
