/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.service.prov.api.response;

import com.sap.cloud.sdk.service.prov.api.Severity;

/**
 * Provides the methods for creating an error response and populating its status
 * code, cause, and message, in the case of a failed operation.
 */
public interface ErrorResponseBuilder {
	/**
	 * Sets the exception that caused the operation to fail in the
	 * ErrorResponseBuilder object.
	 * 
	 * @param throwable
	 *            A <code>Throwable</code> object that represents the exception
	 * @return An <code>ErrorResponseBuilder</code> object containing the
	 *         exception
	 */
	public ErrorResponseBuilder setCause(Throwable throwable);

	/**
	 * Sets the status code of the error in the ErrorResponseBuilder object.
	 * 
	 * @param statusCode
	 *            An <code>int</code> representing the status code
	 * @return An <code>ErrorResponseBuilder</code> object containing the status
	 *         code
	 */
	public ErrorResponseBuilder setStatusCode(int statusCode);

	/**
	 * Sets the error message considering localization.   
	 *  
	 * @param messageKey
	 *            A <code>String</code> key associated with the message in
	 *            properties file to retrieve the message text.
	 *            In case there is no value for such a key in the property
	 *            file, the message key itself will be taken as the message text.
	 *            This parameter is set as the error code in the error response
	 *            only if the error code is not explicitly set using setErrorCode API.
	 * 
	 * @param params
	 *            Variable length array of objects for this message. These
	 *            objects will be converted to strings and substituted in the messageText in place
	 *            of parameter placeholders. In case there are no parameters, do not
	 *            pass any objects for this argument.
	 * 
	 * @return An <code>ErrorResponseBuilder</code> object containing the
	 *         message.
	 */
	public ErrorResponseBuilder setMessage(String messageKey, Object... params);

	/**
	 * Returns an instance of ErrorResponse object.
	 * 
	 * @return An <code>ErrorResponse</code> object
	 */
	public ErrorResponse response();

	/**
	 * Adds the non-leading container messages with the given severities to the
	 * ErrorResponse. Note that if no severity is passed, all the non-leading
	 * container messages will be added to the error response.
	 * 
	 * @param severities
	 *            A variable length argument which denotes the severity of the
	 *            messages which have to be added from the MessageContainer to
	 *            the error response.
	 * @return An ErrorResponsBuilder which would build an ErrorResponse with
	 *         the container messages added as per severity.
	 */
	public ErrorResponseBuilder addContainerMessages(Severity... severities);

	/**
	 * Adds a new error detail. Severity is considered ERROR hence not taken as
	 * an argument. The message text of the error detail will be obtained from a
	 * properties file based on the messageKey parameter. The messageKey
	 * parameter also becomes the "code" of the error detail. Parameters is
	 * taken through a variable length array of objects. These objects will be
	 * stringified and substituted in the messageText in place of parameter
	 * place holders. In case there are no parameters, just don't pass any
	 * objects for this argument.
	 * 
	 * @param messageKey
	 *            associated with the message, used as the key to a properties
	 *            file to retrieve the message text.
	 * @param target
	 *            Target for which this error detail applies.
	 * @param params
	 *            Variable length array of parameters for this message.
	 * @return
	 */
	public ErrorResponseBuilder addErrorDetail(String messageKey, String target, Object... params);
	
	/**
	 * Sets the error code in the ErrorResponseBuilder object.
	 * 
	 * @param errorCode
	 *            A <code>String</code> representing the error code
	 * @return An <code>ErrorResponseBuilder</code> object containing the error
	 *         code
	 */
	public ErrorResponseBuilder setErrorCode(String errorCode);

}



