/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.service.prov.api.response;

import java.util.List;
import java.util.Map;

import com.sap.cloud.sdk.service.prov.api.EntityData;

/**
 * Provides the methods for creating a response, populating its header and content,
 * and enabling audit logging in the case of a successful query operation.
 */
public interface QueryResponseBuilder extends ResponseBuilder {
	/**
	 * Adds headers in the QueryResponseBuilder object. If headers are already present in the QueryResponseBuilder object,
	 * the specified headers are added to the existing ones. The header name is case-insensitive.
	 * Note: The SDK framework might overwrite the values you add to the header in the final
	 * response that gets created.
	 * @param headers A {@link java.util.Map<java.lang.String,java.util.List<java.lang.String>>} containing the headers to be added
	 * @return A <code>QueryResponseBuilder</code> object containing the added headers
	 */
	public QueryResponseBuilder addHeaders(Map<String, List<String>> headers);
	
	/**
	 * Sets a list of key-value pairs representing the properties of entities returned by the query operation
	 * in the QueryResponseBuilder object.
	 * @param data A <code>List</code> of {@link java.util.Map Map} objects that represents the queried entities
	 * @return A <code>QueryResponseBuilder</code> object containing the queried entities
	 */
	public QueryResponseBuilder setDataAsMap(List<Map<String,Object>> data);
	
	/**
	 * Sets a list of EntityData objects representing the entities returned by the query operation
	 * in the QueryResponseBuilder object.
	 * @param entityDataList A <code>List</code> of <code>EntityData</code> objects that represents the queried entities
	 * @return A <code>QueryResponseBuilder</code> object containing the queried entities
	 */
	public QueryResponseBuilder setEntityData(List<EntityData> entityDataList);
	
	/**
	 * Sets a list of POJO objects representing the entities returned by the query operation
	 * in the QueryResponseBuilder object.
	 * @param pojoList A <code>List</code> of POJO objects that represents the queried entities
	 * @return A <code>QueryResponseBuilder</code> object containing the queried entities
	 */
	public QueryResponseBuilder setData(List<?> pojoList);
	
	/**
	 * Sets a header with the specified name and value in the QueryResponseBuilder object.
	 * If the header is already present in the QueryResponseBuilder object,
	 * the new value overwrites the previous one. The header name is case-insensitive.
	 * Note: The SDK framework might overwrite the value you set in the header in the final
	 * response that gets created.
	 * @param name A <code>String</code> containing the name of the header
	 * @param value A <code>String</code> containing the header value
	 * @return A <code>QueryResponseBuilder</code> object containing the newly added or updated header
	 */
	public QueryResponseBuilder setHeader(String name, String value);
	
	/**
	 * Adds a header with the specified name and value in the QueryResponseBuilder object.
	 * If the header is already present in the QueryResponseBuilder object,
	 * the specified value is added to the existing values. The header name is case-insensitive.
	 * Note: The SDK framework might overwrite the value you add to the header in the final 
	 * response that gets created.
	 * @param name A <code>String</code> containing the name of the header
	 * @param value A <code>String</code> containing the additional header value
	 * @return A <code>QueryResponseBuilder</code> object containing the additional header
	 */
	public QueryResponseBuilder addHeader(String name, String value);
	
	/**
	 * Enables auditing for query operation.
	 * Please note that auditing is disabled by default.
	 * @param enable A <code>boolean</code> to enable or disable auditing
	 * @return A <code>QueryResponseBuilder</code> object containing the updated auditing attribute 
	 */
	public QueryResponseBuilder audit(boolean enable);
	
	/**
	 * Specifies the entity and its properties for which auditing must be done
	 * in the case of a query operation.
	 * Please note that auditing must be enabled before calling this method.
	 * @param entityData An <code>EntityData</code> object to be audited
	 * @param properties A <code>List</code> of <code>String</code> objects containing the properties to be audited 
	 * @return A <code>QueryResponseBuilder</code> object containing the entity and its properties to be audited 
	 */
	@Deprecated
	public QueryResponseBuilder audit(EntityData entityData,List<String> properties);
	
	/**
	 * Indicates that there is a custom implementation of the $top system query option. This overrides
	 * the default implementation of this option.
	 * @param isTopDone A <code>boolean</code> to indicate that there is a custom implementation of the $top system query option
	 * @return A <code>QueryResponseBuilder</code> object containing this flag
	 */
	public QueryResponseBuilder setTopDone(boolean isTopDone);
	
	/**
	 * Indicates that there is a custom implementation of the $skip system query option. This overrides
	 * the default implementation of this option.
	 * @param isSkipDone A <code>boolean</code> to indicate that there is a custom implementation of the $skip system query option
	 * @return A <code>QueryResponseBuilder</code> object containing this flag
	 */
	public QueryResponseBuilder setSkipDone(boolean isSkipDone);
	
	/**
	 * Sets a custom inline count value that overrides the default value returned in
	 * the response.
	 * 
	 * @param inlineCount An integer that represents the custom inline count value
	 * @return A <code>QueryResponseBuilder</code> object containing this value
	 */
	public QueryResponseBuilder setInlineCount(int inlineCount);
	
	/**
	 * Returns an instance of QueryResponse object.
	 * @return A <code>QueryResponse</code> object
	 */
	
	public QueryResponse response();
	
	/*
	 * Sets a custom count value that overrides the default count value returned in the response.
	 * 
	 * @param count An integer that represents the custom count value
	 * @return A <code>QueryResponseBuilder</code> object containing this value
	 */
			
	public QueryResponseBuilder setCount(int count);
}
