/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.service.prov.api.security;

/**
 * Base class for all Authorization related APIs.
 * 
 */
public abstract class Authorization {
	
	private String whereCondition;

	/**
	 * Checks if current user is authenticated and given OData service requires
	 * authenticated user to access it. If there is no restriction defined for the
	 * service, It returns true.
	 * 
	 * @param serviceName
	 *            the OData service name.
	 * @return true if no rule found or satisfies rule, else false.
	 * 
	 */
	public abstract boolean isAuthenticatedUser(String serviceName);
	
	/**
	 * Checks if current user is registered user and given OData service requires
	 * registered user to access it. If there is no restriction defined for the
	 * service, It returns true.
	 * 
	 * @param serviceName
	 *            the OData service name.
	 * @return true if no rule found or satisfies rule, else false.
	 * 
	 */
	public abstract boolean isRegisteredUser(String serviceName);
	
	/**
	 *  Checks if current user is granted access to access the given Odata entity to perform 
	 *  the given operation(READ/WRITE). If no rule found, it returns false. (True only when an operation granted explicitly.)
	 * 
	 * @param entityName the OData entity name.
	 * @param operation READ or WRITE.
	 * @return true if no rule found or satisfies rule, else false.
	 */
	public abstract boolean hasEntityAccess(String entityName, String operation);
	
	/**
	 * Returns the name of this current user.
	 *  
	 * @return <code>String</code> containing the current principal's user name. 
	 */
	public abstract String getUserName();
	
	/**
	 * Returns the ID of this current user.
	 *  
	 * @return <code>String</code> containing the current principal's user id. 
	 */
	public abstract String getUserId();
	
	/**
	 * Checks the current user has the given role or not.
	 * 
	 * @param roleName the role name.
	 * @return <code>true</code> if user has the given role otherwise <code>false</code>
	 */
	public abstract boolean hasUserRole(String roleName) ;
	
	/**
	 * Returns the value for the specified attribute which is part of JWT.
	 * It supports fetching attribute values from first level JWT attributes, user attributes and system attributes.
	 *
	 * 
	 * @param attributeName the attribute name same as in JWT token.
	 * @return <code>String</code> if user has the given value for given attribute otherwise <code>null</code>
	 * 
	 */
	public abstract String getUserAttribute(String attributeName);

	/**
	 * Gets the string that has to be updated in the SQL Query as a where condition.
	 * 
	 * @return where condition as a String.
	 */
	public String getWhereCondition() {
		return whereCondition;
	}

	/**
	 * Sets the string that has to be updated in the SQL Query as a where condition.
	 * This is maintained request context level.
	 * 
	 * @param whereCondition as String
	 */
	public void setWhereCondition(String whereCondition) {
		this.whereCondition = whereCondition;
	}
	
	/**
	 * It rejects the request and throws an Unauthorized Exception.
	 * @throws UnauthorizedException
	 */
	public void rejectRequest() throws UnauthorizedException{
		throw new UnauthorizedException("You are not authorized.");
	}
	
}
