/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.service.prov.api.security;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cloud.sdk.service.prov.api.internal.CSNUtil;

/**
 * Contains authorization related information that are defined in service.cds file.
 *  
 *
 */
public final class AuthorizationRulesContainer {

	private static final Logger log = LoggerFactory.getLogger(AuthorizationRulesContainer.class);
	private static Map<String, AuthorizationDetails> authorizationDefinition;
	private static List<String> servicesAndEntitiesWithoutAuthRule;

	private AuthorizationRulesContainer() {
	}

	private static void init() {
		if (authorizationDefinition == null) {
			log.debug("Intializing local storage for authorizationDefinition");
			authorizationDefinition = new HashMap<>();
			servicesAndEntitiesWithoutAuthRule = new ArrayList<>();
		}
	}

	/**
	 * Reads the rule on-demand from CSN.json and builds it in local data store.
	 * Based on availability of the rule returns value.
	 *  
	 * @param serviceOrEntityName the OData service name / entity name. 
	 * @return true if rule found for given service/entity name.
	 */
	public static boolean isRuleExist(String serviceOrEntityName) {
		init();
		if(servicesAndEntitiesWithoutAuthRule.contains(serviceOrEntityName)) {
			return false;
		}else if (authorizationDefinition.containsKey(serviceOrEntityName)) {
			return true;
		} else {
			AuthorizationDetails authDetails = CSNUtil.getAuthorizationDetails(serviceOrEntityName);
			if (authDetails != null && (null != authDetails.getRequires() || null != authDetails.getRestrict())) {
				authorizationDefinition.put(serviceOrEntityName, authDetails);
				return true;
			}
			servicesAndEntitiesWithoutAuthRule.add(serviceOrEntityName);
		}
		return false;
	}
	
	/**
	 * 
	 * Reads the rule on-demand from CSN.json and builds it in local data store.
	 * Returns the Rule for the given service/entity.
	 * 
	 * @param serviceOrEntityName the OData service name / entity name. 
	 * @return the value of AuthorizationDetails, or null if there is no authorization rule found.
	 * 
	 */
	public static AuthorizationDetails getRule(String serviceOrEntityName) {
		if (isRuleExist(serviceOrEntityName)) {
			return authorizationDefinition.get(serviceOrEntityName);
		}
		return null;
	}

	public static Map<String, AuthorizationDetails> getAuthorizationDefinition() {
		return authorizationDefinition;
	}

	public static void setAuthorizationDefinition(Map<String, AuthorizationDetails> authorizationDefinition) {
		AuthorizationRulesContainer.authorizationDefinition = authorizationDefinition;
	}

	public static List<String> getServicesAndEntitiesWithoutAuthRule() {
		return servicesAndEntitiesWithoutAuthRule;
	}

	public static void setServicesAndEntitiesWithoutAuthRule(List<String> servicesAndEntitiesWithoutAuthRule) {
		AuthorizationRulesContainer.servicesAndEntitiesWithoutAuthRule = servicesAndEntitiesWithoutAuthRule;
	}
	
}
