/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.service.prov.api.security;

import com.google.gson.JsonObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * Provides methods for securing the service and entities exposed as part of the service. Specifically, these methods check user-level access to
 * the service and its entities.
 */
public final class AuthorizationService {

    final static Logger logger = LoggerFactory.getLogger(AuthorizationService.class);
    private static final Logger log = LoggerFactory.getLogger(AuthorizationService.class);
    private static final String BASE_AUTH_CLASS = "com.sap.cloud.sdk.service.prov.rt.security.SpringSecurityImpl";
    private static final InheritableThreadLocal<AuthJWTToken> jwtTokenStore = new InheritableThreadLocal<>();
    private static final InheritableThreadLocal<Authorization> authorizationFactory = new InheritableThreadLocal<>();
    // used to store import com.sap.xs2.security.container.UserInfo object instances only
    private static final InheritableThreadLocal<Object> userInfoStore = new InheritableThreadLocal<>();
    //private static final String USER_INFO_CLASS = "com.sap.xs2.security.container.UserInfo";

    private AuthorizationService() {
    }

    /**
     * Creates an {@link com.sap.cloud.sdk.service.prov.api.security.Authorization Authorization} object whenever an authorization check is required. Once created, this object is
     * available till the request is processed.
     *
     * @return An {@link com.sap.cloud.sdk.service.prov.api.security.Authorization Authorization} object
     */
    public static Authorization getAuthorization() {
        Authorization auth = authorizationFactory.get();
        if (null == auth) {
            log.debug("Authorization object is not available. Creating a new one.");
            log.info("Enabling container security.");
            auth = getContainerAuthObject();
            if (null != auth) {
                authorizationFactory.set(auth);
            } else {
                log.error("Unable to instantiate Authorization object.");
            }
        }
        return auth;
    }

    private static Authorization getSpringAuthObject(String baseClass) {
        Authorization auth = null;
        try {
            Authorization bc = (Authorization) Class.forName(baseClass).newInstance();
            auth = new BackwardCompatibility(bc);
        } catch (Exception e) {
            log.error(String.format("Error while creating an Authorization object: %s", e.getMessage()), e);
            auth = null;
        }
        return auth;
    }

    private static Authorization getContainerAuthObject() {
        Authorization auth = null;
        try {
            Authorization bc = new ContainerSecurityImpl();
            auth = new BackwardCompatibility(bc);
        } catch (Exception e) {
            log.error(String.format("Error while creating an Authorization object: %s", e.getMessage()), e);
            auth = null;
        }
        return auth;
    }

    /**
     * Checks if the current user is authenticated to access the service. If there is no restriction defined for the
     * service in the model, it always returns true.
     *
     * @param serviceName Name of the service for which the authentication check is performed
     * @return <code>true</code> if there is no restriction on the service or if the user is authenticated to access the service
     * , <code>false</code> otherwise
     */
    public static boolean isAuthenticatedUser(String serviceName) {
        log.debug("Authorization check on service {}.", serviceName);
        Authorization auth = getAuthorization();
        if (null != auth) {
            return auth.isAuthenticatedUser(serviceName);
        }
        return false;
    }

    /**
     * Checks if the current user is registered to access the service. If there is no restriction defined for the
     * service in the model, it always returns true.
     *
     * @param serviceName Name of the service for which the authentication check is performed
     * @return <code>true</code> if there is no restriction on the service or if the user is authenticated to access the service
     * , <code>false</code> otherwise
     */
    public static boolean isRegisteredUser(String serviceName) {
        log.debug("Registerd user check on service {}.", serviceName);
        Authorization auth = getAuthorization();
        if (null != auth) {
            return auth.isRegisteredUser(serviceName);
        }
        return false;
    }

    /**
     * Checks if the current user has rights to perform
     * the specified operation on an entity. If there is no restriction
     * defined on the entity in the service model, it always returns true.
     *
     * @param entityName Name of the entity being accessed
     * @param operation  String representing the operation. Allowed values are READ or WRITE.
     * @return <code>true</code> if the user is granted rights to perform the specified operation on the entity
     * , <code>false</code> otherwise
     */
    public static boolean hasEntityAccess(String entityName, String operation) {
        log.debug("Access control check on entity {} for the operation {}.", entityName, operation);
        // Remove any existing where clause from previous entity access checks like $expand and navigation or batch calls
        AuthorizationService.setWhereCondition(null);
        Authorization auth = getAuthorization();
        if (null != auth) {
            return auth.hasEntityAccess(entityName, operation);
        }
        return false;
    }
    
    /**
     * Checks if the current user has rights to perform
     * the specified operation on an expanded entity. If there is no restriction
     * defined on the entity in the service model, it always returns true.
     *
     * @param entityName Name of the entity being accessed
     * @param operation  String representing the operation. Allowed values are READ or WRITE.
     * @return <code>true</code> if the user is granted rights to perform the specified operation on the entity
     * , <code>false</code> otherwise
     */
    public static boolean hasExpandedEntityAccess(String entityName, String operation) {
        log.debug("Access control check on entity {} for the operation {}.", entityName, operation);
        Authorization auth = getAuthorization();
        if (null != auth) {
            return auth.hasEntityAccess(entityName, operation);
        }
        return false;
    }

    /**
     * Gets the condition specified in the service model for granting entity-level access.
     *
     * @return <code>String</code> containing the resolved <code>where</code> condition
     */
    public static String getWhereCondition() {
        log.debug("Getting where condition to restrict query.");
        Authorization auth = getAuthorization();
        if (null != auth) {
            return auth.getWhereCondition();
        }
        return "";
    }

    public static void setWhereCondition(String whereCondition) {
        log.debug("Setting where condition to restrict query {}", whereCondition);
        Authorization auth = getAuthorization();
        if (null != auth) {
            auth.setWhereCondition(whereCondition);
        }
    }

    /**
     * Returns the name of this current user.
     *
     * @return <code>String</code> containing the current principal's user name.
     */
    public static String getUserName() {
        log.debug("Getting where condition to restrict query.");
        Authorization auth = getAuthorization();
        if (null != auth) {
            return auth.getUserName();
        }
        return "";
    }

    /**
     * Returns the ID of this current user.
     *
     * @return <code>String</code> containing the current principal's user id.
     */
    public static String getUserId() {
        log.debug("Getting current user's id.");
        Authorization auth = getAuthorization();
        if (null != auth) {
            return auth.getUserId();
        }
        return null;
    }

    /**
     * Checks if the current user has the given role associated.
     *
     * @param roleName the role name
     * @return <code>true</code> if the user has the specified role, <code>false</code> otherwise
     */
    public static boolean hasUserRole(String roleName) {
        log.debug("Checking current user's role has {}", roleName);
        Authorization auth = getAuthorization();
        if (null != auth) {
            return auth.hasUserRole(roleName);
        }
        return false;
    }

    /**
     * Returns the value for the specified attribute which is part of JWT.
     * It supports fetching attribute values from first level JWT attributes, user attributes and system attributes.
     *
     *
     * @param attributeName the attribute name same as in JWT token.
     * @return <code>String</code> if user has the given value for given attribute otherwise <code>null</code>
     *
     */
    public static String getUserAttribute(String attributeName) {
        log.debug("Getting current user's attribute {}", attributeName);
        Authorization auth = getAuthorization();
        if (null != auth) {
            return auth.getUserAttribute(attributeName);
        }
        return null;
    }


    /**
     * Returns the JSON Element representation for attribute value.
     * The parameter is similar to JSON object access notation each object access is delimited by dot(.) notation
     * e.g ext_attr.customer.name.
     *  It doesn't support cases where attribute name/key itself has dot(.) notation in it.
     *  For e.g "company.customer" is a key in JWT token then we cannot access it via this method.
     *  As this delimited object access will look like ext_attr.company.customer.name , here each
     *  delimited string is treated as a key name. So, 'company' and 'customer' will be treated as different
     *  keys and 'company' will become parent of 'customer' which is wrong as it should be treated as a single key.
     *  This will be improved further to handle the above scenario.
     *  @param dotDelimitedJSONObjectAccessKey dot delimited json object access key
     * @return <code>JsonElement</code> containing the current information for specified field.
     */
    /*public static JsonElement getJSONValueForAttribute(String dotDelimitedJSONObjectAccessKey) {
        log.debug("Getting current user's JWT JSON attribute {}", dotDelimitedJSONObjectAccessKey);
        Authorization auth = getAuthorization();
        if (null != auth) {
            return auth.getJSONValueForAttribute(dotDelimitedJSONObjectAccessKey);
        }
        return null;
    }*/

    
    /**
     * This method gets JWTAuthToken
     *
     * @return
     */

    public static AuthJWTToken getJWTToken() {
        /*
         Note : All sidecar operations depend upon getJWTToken() method.
        if jwtToken is null then spring security may have been enabled and AuthorizationService.getAuthorization()
        hasn't been called yet like in initdb call, therefore JWT token isn't stored in jwtTokenStore yet
        So lets initialise getAuthorization() so all threadlocals are initialised.
        */
        if(jwtTokenStore.get() == null){
            getAuthorization();
        }
        return jwtTokenStore.get();
    }

    /**
     * Sets the JWT Token using Base64 Encoded string for current request.
     *
     * @param base64EncodedJWTStr
     */
    public static void setJWTToken(String base64EncodedJWTStr) {
        jwtTokenStore.set(new AuthJWTToken(base64EncodedJWTStr));
    }

    /**
     * Sets the JWT Token using Base64 Encoded string and its converted JsonNode for current request.
     *
     * @param base64EncodedJWTStr
     */
    public static void setJWTToken(String base64EncodedJWTStr, JsonObject jwtJson) {
        jwtTokenStore.set(new AuthJWTToken(base64EncodedJWTStr, jwtJson));
    }

    /**
     * Gets the USERInfo object for the current request
     *
     * @return
     */
    public static Object getUserInfo() {
         /*
        if userInfo is null then spring security may have been enabled and AuthorizationService.getAuthorization()
        hasn't been called
        So lets initialise getAuthorization() so all threadlocals are initialised.
        */
        if(userInfoStore.get() == null){
            getAuthorization();
        }
        return userInfoStore.get();
    }


    /**
     * Removes the Authorization data for current request.
     */

    public static void purgeCurrentAuthorization() {

        if (null != authorizationFactory.get()) {
            log.debug("Destroying Authorization Object.");
            authorizationFactory.remove();
        }
        if (null != jwtTokenStore.get()) {
            log.debug("Destroying Authorization JWT Token.");
            jwtTokenStore.remove();
        }

        if (null != userInfoStore.get()) {
            log.debug("Destroying Authorization UserInfo Object.");
            userInfoStore.remove();
        }
    }
}
