package com.sap.cloud.sdk.service.prov.api.security;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

public class JWTUtil {
	private static final Logger log = LoggerFactory.getLogger(JWTUtil.class);

	private static final String SYSTEM_ATTRIBUTES = "xs.system.attributes";
	private static final String USER_ATTRIBUTES = "xs.user.attributes";
	private static final String GRANT_TYPE = "grant_type";

	private JWTUtil(){}
	/*
		Fetches the first level attribute,SYSTEM_ATTRIBUTES and USER_ATTRIBUTES
		from JWT Token.
	 */

	public static JsonElement getValueFromJwt(String attributeName) {
		log.debug("Reading value of the attibute '{}' from JWT.", attributeName);
		JsonElement attributeValueElement = null;
		if(AuthorizationService.getJWTToken() == null){
			return null;
		}
		JsonObject attributes = AuthorizationService.getJWTToken().getJwtAsJsonNode();
		if (null != attributes) {
			attributeValueElement = attributes.get(attributeName);
		}
		// to read system attributes specific
		if (attributeValueElement == null && attributes != null) {
			JsonObject temp = attributes.getAsJsonObject(SYSTEM_ATTRIBUTES);
			if (null != temp) {
				attributeValueElement = temp.get(attributeName);
			}
		}
		// to read user attribute specific
		if (attributeValueElement == null && attributes != null) {
			JsonObject temp = attributes.getAsJsonObject(USER_ATTRIBUTES);
			if (null != temp) {
				attributeValueElement = temp.get(attributeName);
			}
		}
		return attributeValueElement;
	}

	public static List<String> getScopes() {
		log.debug("Reading all scopes of the current user from JWT");
		List<String> lstScopes = new ArrayList<>();
		JsonElement scopes = JWTUtil.getValueFromJwt("scope");
		if (null != scopes && scopes.isJsonArray()) {
			Iterator<JsonElement> scopeIterator = scopes.getAsJsonArray().iterator();
			while (scopeIterator.hasNext()) {
				String curScope = getScopeName(scopeIterator.next().getAsString());
				lstScopes.add(curScope);
			}
		}
		return lstScopes;
	}

	/*
	 * scope can be of type "CXSODataV2InstanceAuth!t3580.Customer.Config.Show" here
	 * actual scope is "Customer.Config.Show"
	 */
	private static String getScopeName(String fullScopeName) {
		String[] actualScopeDetails = fullScopeName.split("\\.", 2);
		int size = actualScopeDetails.length - 1;
		return actualScopeDetails[size].trim();
	}

	public static String getGrantType() {
		log.debug("Reading agrant type from JWT");

		String grantType = null;
		JsonElement grantElement = JWTUtil.getValueFromJwt(GRANT_TYPE);
		if (grantElement != null) {
			grantType = grantElement.getAsString();
		}
		return grantType;
	}
	/*
		Fetches the first level attribute,SYSTEM_ATTRIBUTES and USER_ATTRIBUTES
		from JWT Token as String
		Needs to improved later to handle all the complex level depth access using getValueForAnyAttributeFromJWT
	 */

	public static String getJWTAttribute(String attributeName) {
		log.debug("Reading all value of {} from JWT", attributeName);
		String attributeValue = "";
		JsonElement attributeValueElement = JWTUtil.getValueFromJwt(attributeName);
		if (attributeValueElement == null){
			attributeValueElement = JWTUtil.getValueForAnyAttributeFromJWT(attributeName);
		}
		if (null != attributeValueElement && attributeValueElement.isJsonPrimitive()) {
			attributeValue = attributeValueElement.getAsString();
		} else if (null != attributeValueElement && attributeValueElement.isJsonArray()) {
			// array of attribute handling. It returns comma separated values if multiple
			// values found
			JsonArray attributes = attributeValueElement.getAsJsonArray();
			if (attributes.size() == 1) {
				attributeValue = attributeValueElement.getAsJsonArray().get(0).getAsString();
			} else {
				for (int i = 0; i < attributes.size(); i++) {
					attributeValue += attributeValueElement.getAsJsonArray().get(i).getAsString();
					if (i != attributes.size() - 1) {
						attributeValue += ",";
					}
				}
			}
		}		
		return attributeValue;
	}


	/**
	 * Return the JSON Element representation for attribute value
	 * Similar to JSON object access notation each object access is delimited by dot(.).
	 * It won't work if attribute name/key itself has dot like "xs.user.attributes".
	 *  This will be improved further to handle the above scenario.
	 *  @param dotDelimitedJSONObjectAccessKey dot delimited json object access key
	 * @return <code>JsonElement</code> containing the current information for specified field.
	 */

	public static JsonElement getValueForAnyAttributeFromJWT(String dotDelimitedJSONObjectAccessKey) {
		JsonElement attributeValueElement = null;
		String [] hierarchicalKeys = dotDelimitedJSONObjectAccessKey.split("\\.");
		if (AuthorizationService.getJWTToken()!= null && AuthorizationService.getJWTToken().getJwtAsJsonNode() != null){
			JsonObject jwtNode = AuthorizationService.getJWTToken().getJwtAsJsonNode();
			if(hierarchicalKeys.length == 1 ){
					return jwtNode.get(hierarchicalKeys[0]);
				}
				JsonObject parentNode = jwtNode.getAsJsonObject(hierarchicalKeys[0]);
				int sz = hierarchicalKeys.length;
				for(int i = 1 ; i < sz - 1 ; i++ ){
					if(parentNode == null) {
						return null;
					}
					parentNode = parentNode.getAsJsonObject(hierarchicalKeys[i]);
				}
				if(parentNode != null ){
					attributeValueElement = parentNode.get(hierarchicalKeys[sz-1]);
				}
		}
		return attributeValueElement;
	}

}
