/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.service.prov.api.security;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.script.ScriptException;
import java.util.*;

import static com.sap.cloud.sdk.service.prov.api.security.ExpressionExecutorUtil.EMPTY;
import static com.sap.cloud.sdk.service.prov.api.security.ExpressionExecutorUtil.SPECIAL;

public final class SecurityUtil {
    public static final List<String> SYSTEM_USER_GRANT_TYPES = Collections.unmodifiableList(Arrays.asList("client_credentials", "client_x509"));
    private static final String USER_ATTRIBUTES = "xs.user.attributes";
    private static final String SYSTEM_USER_SCOPE = "system-user";
    private static final String AUTHENTICATED_USER = "authenticated-user";
    private static final String SYSTEM_USER = "system-user";
    private static final String ANY = "any";
    private static final String USER_NAME = "user_name";
    private static final String ZID = "zid";
    private static final String USER = "user";
    private static final String USER_ID = "user_id";
    private static final String TENANT = "tenant";
    private static final String OPEN_ID = "openid";
    private static final String ONLY_WHERE_GRANTS = "OnlyWhereGrants";
    private static final String BOTH_TO_AND_WHERE_GRANTS = "BothToAndWhereGrants";
    private static final String XSUAA = "xsuaa";
    private static final String CLIENT_ID= "client_id";
    private static final String XSUAA_CLIENTID = "clientid";
    private static final String VCAP_SERVICES = "VCAP_SERVICES";
    private static final String XSUAA_CREDENTIALS = "credentials";


    private static final Logger log = LoggerFactory.getLogger(SecurityUtil.class);

    private SecurityUtil() {
    }

    public static boolean checkUserAccess(AuthorizationDetails authDetails, List<String> scopes,
                                          JsonObject userAttributes, String operation) {
        boolean isAllowed = false;
        List<AccessGrantDetails> grantDetails = authDetails.getRestrict();
        AuthorizationService.setWhereCondition(null);

        Map<String, List<AccessGrantDetails>> grantTypes = new HashMap<>();

        grantTypes.put(ONLY_WHERE_GRANTS, new ArrayList<>());
        grantTypes.put(BOTH_TO_AND_WHERE_GRANTS, new ArrayList<>());


        for (AccessGrantDetails grantDetail : grantDetails) {
            if (checkGrant(grantDetail, operation)) {
                /*
                 If user has scope and GRANT type is pure TO without any WHERE,
                  no need to do any further check on where condition directly allow access
                  */
                if (grantDetail.getTo() != null
                        && grantDetail.getWhere() == null

                        && scopes.contains(grantDetail.getTo())) {
                    // no need to do any further check return access allowed
                    isAllowed = true;
                    break;

                } else if (grantDetail.getTo() != null
                        && scopes.contains(grantDetail.getTo())
                        && grantDetail.getWhere() != null) {

                    grantTypes.get(BOTH_TO_AND_WHERE_GRANTS).add(grantDetail);

                } else if (grantDetail.getTo() == null
                        && grantDetail.getWhere() != null) {

                    grantTypes.get(ONLY_WHERE_GRANTS).add(grantDetail);

                }
            }
        }


        if (!isAllowed && userAttributes != null) {
            List<AccessGrantDetails> grantDetailsWithWhere = null;
            /*
                If user has matching scopes and there are matching
                GRANT statements having both to and where then use these where conditions
             */
            if (grantTypes.get(BOTH_TO_AND_WHERE_GRANTS).size() > 0) {
                grantDetailsWithWhere = grantTypes.get(BOTH_TO_AND_WHERE_GRANTS);
            } else {
                /*
                    If user doesn't have any matching scopes and there are individual where
                    conditions then use these where conditions.
                */
                grantDetailsWithWhere = grantTypes.get(ONLY_WHERE_GRANTS);

            }
            List<String> whereList = new ArrayList<>();

            for (AccessGrantDetails grantDetail : grantDetailsWithWhere) {

                // process all where clauses and them to list
                try {

                    String updatedExpression = ExpressionExecutorUtil.buildExpression(grantDetail.getWhere(), userAttributes);
                    /*
                        TODO: Handling for null values needs to be performed. Here null value is represented as #SKIPVAL# ,
                         if an expression as the skip value then ignore that expression or don't evaluate it.
                         Currently since there is no logic for handling above scenario, so throw unauthorised access when value in jwt attribute is null.
                         Once logic is implemented allow user access with the where clause having only conditions with valid values.
                     */
                  /*  if(updatedExpression.contains(ExpressionExecutorUtil.SKIP_VALUE)){
                        log.error("Where condition has null value throw unauthorised error");
                        return true;
                    }*/
                    if (updatedExpression.startsWith(SPECIAL)) {
                        String whereCondition = updatedExpression.replace(SPECIAL, EMPTY);
                        whereList.add(whereCondition);
                    } else {
                        // If anytime static check is true give access and don't add any where condition
                        isAllowed |= ExpressionExecutorUtil.executeExpression(updatedExpression);

                    }

                } catch (ScriptException e) {
                    log.error("Error while evaluating expression ", e);
                }
                if (isAllowed) {
                    break;
                }


            }
            if (!isAllowed && whereList.size() > 0) {
                StringBuilder sb = new StringBuilder();

                boolean firstFlag = true;
                for (String whereCond : whereList) {
                    if (firstFlag) {
                        firstFlag = false;
                        sb.append(whereCond);
                        continue;
                    }
                  /*  switch (OR) {
                        case OR:
                            sb.append(" OR ");
                            sb.append(whereCond);
                            break;
                        case AND:
                            // not supported yet
                            break;
                    }*/
                    sb.append(" " + Operator.OR.getValue() + " ");
                    sb.append(whereCond);
                }
                AuthorizationService.setWhereCondition(sb.toString());
                isAllowed = true;
            }
        }
        return isAllowed;
    }

    private static boolean checkGrant(AccessGrantDetails grantDetail, String operation) {
		    	if (operation.equals("*")){
		    		return true;
		    	}
		    	List<String> grants = grantDetail.getGrant();
		    	if (grants != null)
		    		return grants.contains(operation);
		    	else 
		    		return false;   
      
    }

    public static boolean isAuthenticatedUser(AuthorizationDetails authDetails, String serviceName) {
        boolean isAuthenticatedUser = false;
        List<String> requires = authDetails.getRequires();

        //first check for custom roles in scopes of jwt then check for pseudo roles
        // don't check in the role collection in jwt token as role collections can change any time only check for scopes
        List<String> assignedScopes = JWTUtil.getScopes();
        for (String require : requires) {
            if (assignedScopes.contains(require)) {
                isAuthenticatedUser = true;
            } else if (AUTHENTICATED_USER.equalsIgnoreCase(require)) {
                isAuthenticatedUser = assignedScopes.contains(OPEN_ID);
            } else if (SYSTEM_USER.equalsIgnoreCase(require)) {
                isAuthenticatedUser = SYSTEM_USER_GRANT_TYPES.contains(JWTUtil.getGrantType());
            } else if (ANY.equalsIgnoreCase(require)) {
                isAuthenticatedUser = true;
            }
            if (isAuthenticatedUser) {
                break;
            }
        }
        return isAuthenticatedUser;
    }

    public static boolean hasEntityAccess(String entityName, String operation) {
        log.debug("Authorization Check for Service: {} and Operation: {}", entityName, operation);
        JsonObject userAttributes = null;
        AuthorizationDetails authDetails = AuthorizationRulesContainer.getRule(entityName);
        JsonElement userAttributesElement = JWTUtil.getValueFromJwt(USER_ATTRIBUTES);
        if (null != userAttributesElement) {
            userAttributes = userAttributesElement.getAsJsonObject();
            userAttributes.add(USER, JWTUtil.getValueFromJwt(USER_NAME));
        }
        JsonElement zid = JWTUtil.getValueFromJwt(ZID);
        // If no USER_ATTRIBUTES present in jwt token then create a new empty userAttributes json object
        if (userAttributes == null && zid != null) {
            userAttributes = new JsonObject();
        }
        if (zid != null) {
            userAttributes.add(TENANT, zid);
        }


        List<String> userScopes = JWTUtil.getScopes();
        //first figure  out user type and add system-user scope if current user is system-user
        if (SecurityUtil.SYSTEM_USER_GRANT_TYPES.contains(JWTUtil.getGrantType())) {
            userScopes.add(SYSTEM_USER_SCOPE);
        }
        // userAttributes can be null for a system-user
        if (authDetails != null) {
        	Boolean isAllowed = SecurityUtil.checkUserAccess(authDetails, userScopes, userAttributes, operation);
        	if(isAllowed == false)
        		return isAllowed;
        	String envVarAsString = System.getenv(VCAP_SERVICES);
        	if(envVarAsString != null) {
        		JsonArray envVar = new JsonParser().parse(envVarAsString).getAsJsonObject().get(XSUAA).getAsJsonArray();
        		ArrayList<String> clientIds = new ArrayList<String>();
        		for(JsonElement jsonElement : envVar) {
        			if(jsonElement.getAsJsonObject().keySet().contains(XSUAA_CREDENTIALS)) {
        				clientIds.add(((JsonObject) jsonElement.getAsJsonObject().get(XSUAA_CREDENTIALS)).get(XSUAA_CLIENTID).getAsString());
        			}
        		}
        		for(String clientId : clientIds) {
        			if(clientId.equals(JWTUtil.getValueFromJwt(CLIENT_ID).getAsString()))
        				return true;
        		}
        		return false;
        	}
        	return isAllowed;
        }
        return false;
    }

    public static boolean hasUserRole(String roleName) {
        List<String> userScopes = JWTUtil.getScopes();
        return userScopes.contains(roleName);
    }

    public static String getUserId() {
        return JWTUtil.getJWTAttribute(USER_ID);
    }

    public static String getUserName() {
        return JWTUtil.getJWTAttribute(USER_NAME);
    }
}
