/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.service.prov.api.util;

import java.io.File;
import java.io.FileFilter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 */
public class ClassHelper {

	private static final String JAR_FILE_ENDING = "jar";
	private static final char RESOURCE_SEPARATOR = '/';
	private static final char PACKAGE_SEPARATOR = '.';
	private static final File[] EMPTY_FILE_ARRAY = new File[0];
	private static final String CLASSFILE_ENDING = ".class";

	private static final FilenameFilter CLASSFILE_FILTER = (dir, name) -> name.endsWith(CLASSFILE_ENDING);

	private static final FileFilter FOLDER_FILTER = File::isDirectory;

	final static Logger logger = LoggerFactory.getLogger(ClassHelper.class);

	public static List<Class<?>> loadClassesFromClasspath(final ClassValidator cv) {
		try {
			return loadClasses("", cv);
		} catch (Exception e) {
			// TODO: mibo_160323: change exception behavior
			logger.error("loading classes from classpath error", e);
			return Collections.emptyList();
		}
	}

	/**
	 * Returns the classes in the classpath that satisfies the passed
	 * classvalidator.
	 * 
	 * @param packageToScan
	 * @param cv
	 * @return
	 */
	public static List<Class<?>> loadClasses(final String packageToScan, ClassValidator cv) {
		ClassLoader classloader = Thread.currentThread().getContextClassLoader();
		List<URL> urls = new ArrayList<>();
		/**
		 * If package is not given(null), then we use the getUrls() method to get all
		 * the urls in the classpath, On the other hand if the package is given, we use
		 * the classloader.getResources method to get urls for resources under that
		 * package.
		 * 
		 * Note that this code is also invoked currently for Spring boot apps, even
		 * though classloading is separately in such apps(SpringExtensionConfig class).
		 * With spring boot the getURLs method returns empty list and hence this method
		 * returns quickly without any wastage of performance.
		 */
		if (packageToScan == null) {
			if (classloader instanceof URLClassLoader)
				urls = Arrays.asList(((URLClassLoader) classloader).getURLs());
		} else {
			Enumeration<URL> urlEnumeration;
			try {
				urlEnumeration = classloader.getResources(packageToScan.replace('.', '/'));
			} catch (IOException e) {
				logger.error("Error while scanning the classpath", e);
				throw new RuntimeException("Error while scanning the classpath", e);
			}

			while (urlEnumeration.hasMoreElements()) {
				urls.add(urlEnumeration.nextElement());
			}
		}
		List<String> fqnForClasses = new ArrayList<String>();
		urls.parallelStream().forEach(url -> {
			String protocol = url.getProtocol();
			/*
			 * We check for 2 values of protocol, jar and file. jar is when the Url points
			 * to a resource within a jar file. This will happen if the package that is
			 * specified is part of a jar. If the package were not specified, we would still
			 * get urls to jar files which are there in the class path. But this is to the
			 * jar file itself and not to some file WITHIN THE jar. Hence in this case the
			 * protocol would be file.
			 */
			if (protocol.equals("jar")) {
				URI uri = null;
				try {
					uri = url.toURI();
				} catch (URISyntaxException e) {
					logger.error("Error while reading jar.", e);
					return; //This returns from the lambda expression not the containing method.
				}
				String filepath = uri.getSchemeSpecificPart().substring(5);
				String[] split = filepath.split("!");
				String jarFilePath = split[0];
				fqnForClasses.addAll(getClassFqnsFromJar(new File(jarFilePath), packageToScan));
			} else if (protocol.equals("file")) {
				File resource = new File(url.getPath());
				if (resource.isDirectory()) {
					fqnForClasses.addAll(getClassFqnFromDir(CLASSFILE_FILTER, resource, packageToScan));
				} else if (resource.getName().endsWith(CLASSFILE_ENDING)) {
					fqnForClasses.add(
							resource.getName().substring(0, resource.getName().length() - CLASSFILE_ENDING.length()));
				} else if (resource.getName().endsWith(".jar")) {
					fqnForClasses.addAll(getClassFqnsFromJar(resource, packageToScan));
				}
			}
		});

		List<Class<?>> validClasses = fqnForClasses.parallelStream().map(c -> {
			try {
				return classloader.loadClass(c);
				/*
				 * We catch Throwable which is not recommended in general usage,
				 * We do so because, occasionally some unchecked exceptions come when classes from entire
				 * classpath are being loaded. In our scenario we just need to discard such classes,
				 * Hence we catch throwable. Apart from the fatal error VirtualMachineError, we 
				 * ignore all other throwables(by returning null). For VirtualMachineErrors(subclasses of which
				 * are OutOfMemoryError, StackOverflowError, InternalError and UnknownError) we directly throw
				 * it because this is too serious to ignore.
				 */
			}catch (Throwable t) {
				if(t instanceof VirtualMachineError) {
					throw (VirtualMachineError)t;
				}
				logger.debug("Error while loading class " + c, t);
				return null;
			}
			
		}).filter(c -> {
			if (c == null)
				return false;
			return cv.isClassValid(c);

		}).collect(Collectors.toList());

		return validClasses;
	}

	private static List<String> getClassFqnsFromJar(File resource, String packageToScan) {
		List<String> classes = new ArrayList<>();
		try (JarFile jar = new JarFile(resource);) {
			if (packageToScan != null && !packageToScan.equals("")) {
				packageToScan = packageToScan.replace('.', '/');
			} else
				packageToScan = null;
			if (jar != null) {
				Enumeration<JarEntry> jarEntries = jar.entries();
				while (jarEntries.hasMoreElements()) {
					JarEntry entry = jarEntries.nextElement();
					String name = entry.getName();
					if (!entry.isDirectory()
							&& (packageToScan == null ? !isPackageExcluded(name) : name.startsWith(packageToScan))
							&& name.endsWith(CLASSFILE_ENDING)) {
						classes.add(name.substring(0, name.length() - CLASSFILE_ENDING.length()).replaceAll("/", "."));
					}
				}
			}
		} catch (IOException e) {
			logger.error("Error while reading a jar file", e);
		}

		return classes;
	}

	private static boolean isPackageExcluded(String name) {
		List<String> excludedPackages = Arrays.asList("org", "com/sap/cloud/sdk", "com/sap/cloud/servicesdk", "javax",
				"com/sap/it/commons", "com/sap/gateway/core", "javassist", "com/netflix", "rx", "com/auth0",
				"com/ctc/wstx", "com/google", "com/ibm", "com/microsoft");
		for (String pkg : excludedPackages) {
			if (name.startsWith(pkg + "/"))
				return true;
		}
		return false;
	}

	private static Collection<String> getClassFqnFromDir(final FilenameFilter ff, final File folder,
			final String packageToScan) {
		List<String> classFiles = new ArrayList<>();
		String packagePrefix = "";
		if (packageToScan != null && !packageToScan.isEmpty())
			packagePrefix = packageToScan + ".";

		String[] classFilesForFolder = folder.list(ff);
		for (String name : classFilesForFolder) {
			String fqn = packagePrefix + name.substring(0, name.length() - CLASSFILE_ENDING.length());
			classFiles.add(fqn);
		}
		// recursive search
		File[] subfolders = listSubFolder(folder);
		for (File file : subfolders) {
			classFiles.addAll(getClassFqnFromDir(ff, file, packagePrefix + file.getName()));
		}
		//
		return classFiles;
	}

	private static File[] listSubFolder(final File folder) {
		File[] subfolders = folder.listFiles(FOLDER_FILTER);
		if (subfolders == null) {
			return EMPTY_FILE_ARRAY; // NOSONAR
		}
		return subfolders;
	}

	public interface ClassValidator {
		boolean isClassValid(Class<?> c);
	}
}
