package com.sap.cloud.sdk.service.prov.api.util;

import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.beanutils.BeanMap;

import com.fasterxml.jackson.annotation.JsonAutoDetect.Visibility;
import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.sap.cloud.sdk.service.prov.api.annotations.Key;

public class PojoUtil {

	private static ObjectMapper getMapper( ){
		ObjectMapper mapper = new ObjectMapper();
		/*-- Ignore Null Values --*/
		mapper.configure(SerializationFeature.WRITE_NULL_MAP_VALUES, false);
		/*-- do not fail on empty beans  --*/
		mapper.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false);
		/*-- First: hide all elements in the class ; make property,getter,setter hidden  --*/
		mapper.setVisibility(PropertyAccessor.ALL, Visibility.NONE);
		/*-- Second: now only make the properties discoverable   --*/
		mapper.setVisibility(PropertyAccessor.FIELD, Visibility.ANY);
		/*-- To ensure that during POJO to Map conversion, the null valued properties are not ignored. --*/
		mapper.setSerializationInclusion(Include.ALWAYS);
		mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
		
		return mapper ;
	}
	
	public static Map<String, Object> getMapFromPojo(Object pojo) {
		return (Map<String, Object>) (getMapper().convertValue(pojo, HashMap.class));
	}

	public static <T> T getPojoFromMap(Class<T> clazz, Map<String, Object> data) {
		ObjectMapper mapper = new ObjectMapper();
		return mapper.convertValue(data, clazz);
	}
	
	public static <T> List<String> getKeyProperties(T p) {
		Field[] ss = p.getClass().getDeclaredFields();
		return Arrays
				.asList(ss)
				.stream()
				.filter(propertyElement -> propertyElement
						.getDeclaredAnnotation(Key.class) != null)
				.map(propertyElement -> propertyElement.getName())
				.collect(Collectors.toList());

	}
	
	public static Map<String, Object> getMapUsingBeanUtils(Object pojo) {
		if(pojo instanceof Map)
			return (Map<String, Object>) pojo;
		List<String> pojoFields = Arrays.asList(pojo.getClass().getDeclaredFields()).stream().map(f->f.getName()).collect(Collectors.toList());
		Map<String, Object> pojoAsMap = new HashMap<String, Object>();
		Map<Object, Object> beanMap = new BeanMap(pojo);
		Map<String, Object> finalMap = new HashMap<String, Object>();
		// We need to convert Map<Object, Object> to Map<String, Object> in order to convert it to EntityData.
		beanMap.entrySet().stream().forEach(e->pojoAsMap.put(e.getKey().toString().toLowerCase(), e.getValue()));
		
		/* 
		 * The beanUtils api will determine field names based on their setters and getters, 
		 * and thus may not have the exact case as the original field name. For this purpose we refer to
		 * earlier obtained pojoFields map which has the exact names, and replace the map with the correct names
		 * if necessary.
		 */
		for(String fieldName : pojoFields) {
			if(pojoAsMap.containsKey(fieldName.toLowerCase())) {
				finalMap.put(fieldName, pojoAsMap.get(fieldName.toLowerCase()));
			}
		}
		
		return finalMap;
		
	}
	
	
	
}
