/*
 * Decompiled with CFR 0.152.
 */
package com.thinkaurelius.titan.diskstorage.locking.consistentkey;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.thinkaurelius.titan.core.TitanConfigurationException;
import com.thinkaurelius.titan.diskstorage.PermanentStorageException;
import com.thinkaurelius.titan.diskstorage.StaticBuffer;
import com.thinkaurelius.titan.diskstorage.StorageException;
import com.thinkaurelius.titan.diskstorage.TemporaryStorageException;
import com.thinkaurelius.titan.diskstorage.common.DistributedStoreManager;
import com.thinkaurelius.titan.diskstorage.keycolumnvalue.Entry;
import com.thinkaurelius.titan.diskstorage.keycolumnvalue.KeyColumnValueStore;
import com.thinkaurelius.titan.diskstorage.keycolumnvalue.KeySliceQuery;
import com.thinkaurelius.titan.diskstorage.keycolumnvalue.StaticBufferEntry;
import com.thinkaurelius.titan.diskstorage.keycolumnvalue.StoreTransaction;
import com.thinkaurelius.titan.diskstorage.locking.AbstractLocker;
import com.thinkaurelius.titan.diskstorage.locking.LocalLockMediator;
import com.thinkaurelius.titan.diskstorage.locking.LocalLockMediators;
import com.thinkaurelius.titan.diskstorage.locking.Locker;
import com.thinkaurelius.titan.diskstorage.locking.LockerState;
import com.thinkaurelius.titan.diskstorage.locking.PermanentLockingException;
import com.thinkaurelius.titan.diskstorage.locking.TemporaryLockingException;
import com.thinkaurelius.titan.diskstorage.locking.consistentkey.ConsistentKeyLockStatus;
import com.thinkaurelius.titan.diskstorage.locking.consistentkey.ConsistentKeyLockerSerializer;
import com.thinkaurelius.titan.diskstorage.locking.consistentkey.TimestampRid;
import com.thinkaurelius.titan.diskstorage.util.ByteBufferUtil;
import com.thinkaurelius.titan.diskstorage.util.KeyColumn;
import com.thinkaurelius.titan.diskstorage.util.StaticArrayBuffer;
import com.thinkaurelius.titan.diskstorage.util.TimestampProvider;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.TimeUnit;
import org.apache.commons.configuration.Configuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConsistentKeyLocker
extends AbstractLocker<ConsistentKeyLockStatus>
implements Locker {
    private final KeyColumnValueStore store;
    private final long lockWaitNS;
    private final int lockRetryCount;
    private static final StaticBuffer zeroBuf = ByteBufferUtil.getIntBuffer(0);
    private static final Logger log = LoggerFactory.getLogger(ConsistentKeyLocker.class);

    private ConsistentKeyLocker(KeyColumnValueStore store, StaticBuffer rid, TimestampProvider times, ConsistentKeyLockerSerializer serializer, LocalLockMediator<StoreTransaction> llm, long lockWaitNS, int lockRetryCount, long lockExpireNS, LockerState<ConsistentKeyLockStatus> lockState) {
        super(rid, times, serializer, llm, lockState, lockExpireNS, log);
        this.store = store;
        this.lockWaitNS = lockWaitNS;
        this.lockRetryCount = lockRetryCount;
    }

    private long getLockWait(TimeUnit tu) {
        return tu.convert(this.lockWaitNS, TimeUnit.NANOSECONDS);
    }

    @Override
    protected ConsistentKeyLockStatus writeSingleLock(KeyColumn lockID, StoreTransaction txh) throws Throwable {
        StaticBuffer lockKey = this.serializer.toLockKey(lockID.getKey(), lockID.getColumn());
        StaticBuffer oldLockCol = null;
        for (int i = 0; i < this.lockRetryCount; ++i) {
            WriteResult wr = this.tryWriteLockOnce(lockKey, oldLockCol, txh);
            if (wr.isSuccessful() && wr.getDurationNS() <= this.getLockWait(TimeUnit.NANOSECONDS)) {
                return new ConsistentKeyLockStatus(wr.getBeforeNS(), TimeUnit.NANOSECONDS, wr.getBeforeNS() + this.lockExpireNS, TimeUnit.NANOSECONDS);
            }
            oldLockCol = wr.getLockCol();
            this.handleMutationFailure(lockID, lockKey, wr, txh);
        }
        this.tryDeleteLockOnce(lockKey, oldLockCol, txh);
        throw new TemporaryStorageException("Lock write retry count exceeded");
    }

    /*
     * Enabled aggressive block sorting
     */
    private void handleMutationFailure(KeyColumn lockID, StaticBuffer lockKey, WriteResult wr, StoreTransaction txh) throws Throwable {
        Throwable error = wr.getThrowable();
        if (null == error) {
            log.warn("Lock write succeeded but took too long: duration {} ms exceeded limit {} ms", (Object)wr.getDuration(TimeUnit.MILLISECONDS), (Object)this.getLockWait(TimeUnit.MILLISECONDS));
            return;
        }
        if (error instanceof TemporaryStorageException) {
            log.warn("Temporary exception during lock write", error);
            return;
        }
        log.error("Fatal exception encountered during attempted lock write", error);
        WriteResult dwr = this.tryDeleteLockOnce(lockKey, wr.getLockCol(), txh);
        if (dwr.isSuccessful()) throw error;
        log.warn("Failed to delete lock write: abandoning potentially-unreleased lock on " + lockID, dwr.getThrowable());
        throw error;
    }

    private WriteResult tryWriteLockOnce(StaticBuffer key, StaticBuffer del, StoreTransaction txh) {
        StorageException t = null;
        long before = this.times.getApproxNSSinceEpoch();
        StaticBuffer newLockCol = this.serializer.toLockCol(before, this.rid);
        StaticBufferEntry newLockEntry = new StaticBufferEntry(newLockCol, zeroBuf);
        try {
            this.store.mutate(key, Arrays.asList(newLockEntry), (List<StaticBuffer>)(null == del ? ImmutableList.of() : Arrays.asList(del)), ConsistentKeyLocker.overrideTimestamp(txh, before));
        }
        catch (StorageException e) {
            t = e;
        }
        long after = this.times.getApproxNSSinceEpoch();
        return new WriteResult(before, after, newLockCol, t);
    }

    private WriteResult tryDeleteLockOnce(StaticBuffer key, StaticBuffer col, StoreTransaction txh) {
        StorageException t = null;
        long before = this.times.getApproxNSSinceEpoch();
        try {
            this.store.mutate(key, (List<Entry>)ImmutableList.of(), Arrays.asList(col), ConsistentKeyLocker.overrideTimestamp(txh, before));
        }
        catch (StorageException e) {
            t = e;
        }
        long after = this.times.getApproxNSSinceEpoch();
        return new WriteResult(before, after, null, t);
    }

    @Override
    protected void checkSingleLock(final KeyColumn kc, ConsistentKeyLockStatus ls, StoreTransaction tx) throws StorageException, InterruptedException {
        if (ls.isChecked()) {
            return;
        }
        final long nowNS = this.times.sleepUntil(ls.getWriteTimestamp(TimeUnit.NANOSECONDS) + this.getLockWait(TimeUnit.NANOSECONDS));
        KeySliceQuery ksq = new KeySliceQuery(this.serializer.toLockKey(kc.getKey(), kc.getColumn()), ByteBufferUtil.zeroBuffer(9), ByteBufferUtil.oneBuffer(9));
        List<Entry> claimEntries = this.getSliceWithRetries(ksq, tx);
        Iterable iter = Iterables.transform(claimEntries, (Function)new Function<Entry, TimestampRid>(){

            public TimestampRid apply(Entry e) {
                return ConsistentKeyLocker.this.serializer.fromLockColumn(e.getColumn());
            }
        });
        iter = Iterables.filter((Iterable)iter, (Predicate)new Predicate<TimestampRid>(){

            public boolean apply(TimestampRid tr) {
                if (tr.getTimestamp() < nowNS - ConsistentKeyLocker.this.lockExpireNS) {
                    log.warn("Discarded expired claim on {} with timestamp {}", (Object)kc, (Object)tr.getTimestamp());
                    return false;
                }
                return true;
            }
        });
        this.checkSeniority(kc, ls, iter);
        ls.setChecked();
    }

    private List<Entry> getSliceWithRetries(KeySliceQuery ksq, StoreTransaction tx) throws StorageException {
        for (int i = 0; i < this.lockRetryCount; ++i) {
            try {
                return this.store.getSlice(ksq, tx);
            }
            catch (PermanentStorageException e) {
                log.error("Failed to check locks", (Throwable)e);
                throw new PermanentLockingException(e);
            }
            catch (TemporaryStorageException e) {
                log.warn("Temporary storage failure while checking locks", (Throwable)e);
                continue;
            }
        }
        throw new TemporaryStorageException("Maximum retries (" + this.lockRetryCount + ") exceeded while checking locks");
    }

    private void checkSeniority(KeyColumn target, ConsistentKeyLockStatus ls, Iterable<TimestampRid> claimTRs) throws StorageException {
        int trCount = 0;
        for (TimestampRid tr : claimTRs) {
            ++trCount;
            if (!this.rid.equals(tr.getRid())) {
                String msg = "Lock on " + target + " already held by " + tr.getRid() + " (we are " + this.rid + ")";
                log.debug(msg);
                throw new TemporaryLockingException(msg);
            }
            if (tr.getTimestamp() == ls.getWriteTimestamp(TimeUnit.NANOSECONDS)) {
                log.debug("Checked lock {}", (Object)target);
                return;
            }
            log.warn("Skipping outdated lock on {} with our rid ({}) but mismatched timestamp (actual ts {}, expected ts {})", new Object[]{target, tr.getRid(), tr.getTimestamp(), ls.getWriteTimestamp(TimeUnit.NANOSECONDS)});
        }
        if (0 == trCount) {
            throw new TemporaryLockingException("No lock columns found for " + target);
        }
        String msg = "Read " + trCount + " locks with our rid " + this.rid + " but mismatched timestamps; no lock column contained our timestamp (" + ls.getWriteTimestamp(TimeUnit.NANOSECONDS) + ")";
        throw new PermanentStorageException(msg);
    }

    @Override
    protected void deleteSingleLock(KeyColumn kc, ConsistentKeyLockStatus ls, StoreTransaction tx) {
        ImmutableList dels = ImmutableList.of((Object)this.serializer.toLockCol(ls.getWriteTimestamp(TimeUnit.NANOSECONDS), this.rid));
        for (int i = 0; i < this.lockRetryCount; ++i) {
            try {
                long before = this.times.getApproxNSSinceEpoch();
                this.store.mutate(this.serializer.toLockKey(kc.getKey(), kc.getColumn()), (List<Entry>)ImmutableList.of(), (List<StaticBuffer>)dels, ConsistentKeyLocker.overrideTimestamp(tx, before));
                return;
            }
            catch (TemporaryStorageException e) {
                log.warn("Temporary storage exception while deleting lock", (Throwable)e);
                continue;
            }
            catch (StorageException e) {
                log.error("Storage exception while deleting lock", (Throwable)e);
                return;
            }
        }
    }

    private static StoreTransaction overrideTimestamp(StoreTransaction tx, long nanoTimestamp) {
        tx.getConfiguration().setTimestamp(nanoTimestamp);
        return tx;
    }

    private static class WriteResult {
        private final long beforeNS;
        private final long afterNS;
        private final StaticBuffer lockCol;
        private final Throwable throwable;

        public WriteResult(long beforeNS, long afterNS, StaticBuffer lockCol, Throwable throwable) {
            this.beforeNS = beforeNS;
            this.afterNS = afterNS;
            this.lockCol = lockCol;
            this.throwable = throwable;
        }

        public long getBeforeNS() {
            return this.beforeNS;
        }

        public long getDurationNS() {
            return this.afterNS - this.beforeNS;
        }

        public long getDuration(TimeUnit tu) {
            return tu.convert(this.afterNS - this.beforeNS, TimeUnit.NANOSECONDS);
        }

        public boolean isSuccessful() {
            return null == this.throwable;
        }

        public StaticBuffer getLockCol() {
            return this.lockCol;
        }

        public Throwable getThrowable() {
            return this.throwable;
        }
    }

    public static class Builder
    extends AbstractLocker.Builder<ConsistentKeyLockStatus, Builder> {
        private final KeyColumnValueStore store;
        private long lockWaitNS;
        private int lockRetryCount;

        public Builder(KeyColumnValueStore store) {
            this.store = store;
            this.lockWaitNS = TimeUnit.NANOSECONDS.convert(100L, TimeUnit.MILLISECONDS);
            this.lockRetryCount = 3;
        }

        public Builder lockWaitNS(long wait, TimeUnit unit) {
            this.lockWaitNS = TimeUnit.NANOSECONDS.convert(wait, unit);
            return this.self();
        }

        public Builder lockRetryCount(int count) {
            this.lockRetryCount = count;
            return this.self();
        }

        public Builder fromCommonsConfig(Configuration config) {
            this.rid(new StaticArrayBuffer(DistributedStoreManager.getRid(config)));
            String llmPrefix = config.getString("local-lock-mediator-prefix");
            if (null != llmPrefix) {
                this.mediator(LocalLockMediators.INSTANCE.get(llmPrefix));
            }
            this.lockRetryCount(config.getInt("lock-retries", 3));
            this.lockWaitNS(config.getLong("lock-wait-time", 100L), TimeUnit.MILLISECONDS);
            this.lockExpireNS(config.getLong("lock-expiry-time", 300000L), TimeUnit.MILLISECONDS);
            return this;
        }

        public ConsistentKeyLocker build() {
            this.preBuild();
            return new ConsistentKeyLocker(this.store, this.rid, this.times, this.serializer, this.llm, this.lockWaitNS, this.lockRetryCount, this.lockExpireNS, this.lockState);
        }

        @Override
        protected Builder self() {
            return this;
        }

        @Override
        protected LocalLockMediator<StoreTransaction> getDefaultMediator() {
            throw new TitanConfigurationException("Local lock mediator prefix must not be empty or null");
        }
    }
}

